/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


/*
 * simple flat db for the vnodes
 */

#include "voldb_locl.h"

RCSID("$Id: voldb.c,v 1.5 1999/07/04 11:23:02 lha Exp $");

/*
 * Various internal data definitions
 */

#define VOLDB_HEADER_SIZE (10*4)
#define VOLDB_FILE_SIZE (20*4)
#define VOLDB_DIR_SIZE (20*4+(FS_MAX_ACL*2*4))

/*
 * Various internal data structures
 */

struct voldb_header {
    u_int32_t magic;	/* network order */
#define VOLDB_MAGIC_HEADER 0x47111147
    u_int32_t num;	/* Number of entries in the db */
    u_int32_t freeptr;	/* ptr to next free entry */
    u_int32_t flags;	/* flags */
    u_int32_t spare[6]; /* spare ? */
};

struct voldb {
    struct voldb_header hdr;
    void *ptr;
    size_t size;		/* Size of file */
    int fd;
    union {
	struct voldb_file_entry *file;
	struct voldb_dir_entry *dir;
    } u;
};

static void
set_volhdr_union_ptr (struct voldb *db)
{
    if (db->hdr.flags & VOLDB_DIR)
	db->u.dir = (struct voldb_dir_entry *)
	    ((unsigned char *) db->ptr + VOLDB_HEADER_SIZE);
    else if (db->hdr.flags & VOLDB_FILE)
	db->u.file = (struct voldb_file_entry *)
	    ((unsigned char *) db->ptr + VOLDB_HEADER_SIZE);
    else
	abort();
}

/*
 * boot up the db
 * `fd' is a file descriptor to a db that is returned in `db'.
 * `fd' is saved if everythings works out ok, otherwise its
 * up to the caller to close it.
 */

int
voldb_init (int fd, struct voldb **db)
{
    struct stat sb;
    unsigned char *ptr;
    int ret;

    assert (db);
    *db = NULL;

    ret = fstat (fd, &sb);
    if (ret)
	return errno; /* XXX */
    
    if (!S_ISREG(sb.st_mode))
	return ENOENT; /* XXX */
    
    if (sb.st_size < VOLDB_HEADER_SIZE)
	return EIO; /* XXX */

    *db = calloc (1, sizeof (**db));
    if (*db == NULL)
	return ENOMEM; /* XXX */

    (*db)->size = sb.st_size;
    (*db)->fd = -1;

    {
	u_int32_t tmp;

	(*db)->ptr = mmap (NULL, sb.st_size, PROT_READ|PROT_WRITE,
			   MAP_SHARED, fd, 0);
	if ((*db)->ptr == MAP_FAILED) {
	    free (*db);
	    return errno; /* XXX */
	}

	ptr = (*db)->ptr;

	memcpy (&tmp, ptr, sizeof (tmp));
	(*db)->hdr.magic = ntohl(tmp); ptr += sizeof (tmp);

	if ((*db)->hdr.magic != VOLDB_MAGIC_HEADER) {
	    free (*db);
	    return EINVAL; /* XXX */
	}

	memcpy (&tmp, ptr, sizeof (tmp));
	(*db)->hdr.num = ntohl(tmp); ptr += sizeof (tmp);

	memcpy (&tmp, ptr, sizeof (tmp));
	(*db)->hdr.freeptr = ntohl(tmp); ptr += sizeof (tmp);

	memcpy (&tmp, ptr, sizeof (tmp));
	(*db)->hdr.flags = ntohl(tmp); ptr += sizeof (tmp);
    }

    {
	int size;
	
	if (((*db)->hdr.flags & (VOLDB_FILE|VOLDB_DIR)) == 0) {
	    free (*db);
	    return EIO;
	} else if ((*db)->hdr.flags & VOLDB_FILE)
	    size = VOLDB_FILE_SIZE;
	else if ((*db)->hdr.flags & VOLDB_DIR)
	    size = VOLDB_DIR_SIZE;
	else
	    abort();
	
	if ((sb.st_size - VOLDB_HEADER_SIZE) % size != 0) {
	    free (*db);
	    return EIO; /* XXX */
	}
	if ((*db)->hdr.num != (sb.st_size - VOLDB_HEADER_SIZE) / size)
	    return EINVAL; /* XXX */
    }
    
    set_volhdr_union_ptr (*db);
    
    (*db)->fd = fd;

    return 0;
}    

/*
 * closes the db
 * The saved `fd' is also closed.
 */

int
voldb_close (struct voldb *db)
{
    assert (db);

    assert (db->hdr.magic = VOLDB_MAGIC_HEADER);

    munmap (db->ptr, db->size);
    close (db->fd);
    free (db);
    return 0;
}


/*
 * fetch num'th dir entry (n is 0 based), convert it
 * to hostorder and store it in `e'.
 */

int
voldb_get_dir (struct voldb *db, u_int32_t num, struct voldb_dir_entry *e)
{
    assert (db && e);

    if ((db->hdr.flags & VOLDB_DIR) == 0)
	return EINVAL; /* XXX */

    if (db->hdr.num < num)
	return EINVAL; /* XXX */

#if defined(BYTE_ORDER) &&  BYTE_ORDER	== BIG_ENDIAN && 0
    memcpy (e, &db->u.dir[num], sizeof (*e));
#else
    {
	u_int32_t tmp;
	int i;
	unsigned char *ptr = (unsigned char *)&db->u.dir[num];

	memcpy (&tmp, ptr, sizeof(tmp));
	e->nextptr = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->unique = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ino = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare2 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->InterfaceVersion = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->FileType = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->LinkCount = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Length = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare3 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->DataVersion = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Author = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Owner = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->UnixModeBits = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ParentVnode = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ParentUnique = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->SegSize = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ServerModTime = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare4 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Group = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare5 = ntohl(tmp); ptr += sizeof(tmp);

	for (i = 0 ; i < FS_MAX_ACL; i++) {
	    memcpy (&tmp, ptr, sizeof(tmp));
	    e->acl[i].owner = ntohl(tmp); ptr += sizeof(tmp);
	    memcpy (&tmp, ptr, sizeof(tmp));
	    e->acl[i].flags = ntohl(tmp); ptr += sizeof(tmp);
	}
    }
#endif
    return 0;
}

/*
 * store e as num'th dir entry (n is 0 based), convert it to
 * network order. DO NOT STORE the ACL.
 */

int
voldb_put_dir (struct voldb *db, u_int32_t num, struct voldb_dir_entry *e)
{
    assert (db && e);

    if ((db->hdr.flags & VOLDB_DIR) == 0)
	return EINVAL; /* XXX */

    if (db->hdr.num < num)
	return EINVAL; /* XXX */

#if defined(BYTE_ORDER) &&  BYTE_ORDER	== BIG_ENDIAN && 0
    memcpy (&db->u.dir[num], e, sizeof (*e));
#else
    {
	u_int32_t tmp;
	unsigned char *ptr = (unsigned char *)&db->u.dir[num];

	tmp = htonl(e->nextptr);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->unique);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ino);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare2);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->InterfaceVersion);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->FileType);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->LinkCount);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Length);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare3);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->DataVersion);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Author);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Owner);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->UnixModeBits);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ParentVnode);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ParentUnique);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->SegSize);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ServerModTime);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare4);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Group);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare5);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);
    }
#endif
    return 0;
}

/*
 * store e as num'th acl entry (n is 0 based), convert it to
 * network order.
 */

int
voldb_put_acl (struct voldb *db, u_int32_t num, struct voldb_dir_entry *e)
{
    assert (db && e);

    if ((db->hdr.flags & VOLDB_DIR) == 0)
	return EINVAL; /* XXX */

    if (db->hdr.num < num)
	return EINVAL; /* XXX */

#if defined(BYTE_ORDER) &&  BYTE_ORDER	== BIG_ENDIAN && 0
    memcpy ((char *)&db->u.dir[num] + VOLDB_FILE_SIZE, &e->acl, 
	    sizeof (*e->acl));
#else
    {
	u_int32_t tmp;
	unsigned char *ptr = (unsigned char *)&db->u.dir[num];
	int i;

	ptr += VOLDB_FILE_SIZE * sizeof (tmp);

	for (i = 0 ; i < FS_MAX_ACL; i++) {
	    tmp = htonl(e->acl[i].owner);
	    memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);
	    tmp = htonl(e->acl[i].flags);
	    memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);
	}
    }
#endif
    return 0;
}


/*
 * fetch num'th file entry (n is 0 based), convert it
 * to hostorder and store it in `e'.
 */

int
voldb_get_file (struct voldb *db, u_int32_t num, struct voldb_file_entry *e)
{
    assert (db && e);

    if ((db->hdr.flags & VOLDB_FILE) == 0)
	return EINVAL; /* XXX */

    if (db->hdr.num < num)
	return EINVAL; /* XXX */

#if defined(BYTE_ORDER) &&  BYTE_ORDER	== BIG_ENDIAN && 0
    memcpy (e, &db->u.file[num], sizeof (*e));
#else
    {
	u_int32_t tmp;
	unsigned char *ptr = (unsigned char *)&db->u.file[num];

	memcpy (&tmp, ptr, sizeof(tmp));
	e->nextptr = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->unique = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ino = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare2 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->InterfaceVersion = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->FileType = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->LinkCount = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Length = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare3 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->DataVersion = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Author = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Owner = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->UnixModeBits = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ParentVnode = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ParentUnique = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->SegSize = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->ServerModTime = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare4 = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->Group = ntohl(tmp); ptr += sizeof(tmp);

	memcpy (&tmp, ptr, sizeof(tmp));
	e->spare5 = ntohl(tmp); ptr += sizeof(tmp);
    }
#endif
    return 0;
}

/*
 * store e as num'th file entry (n is 0 based), convert it to
 * network order.
 */

int
voldb_put_file (struct voldb *db, u_int32_t num, struct voldb_file_entry *e)
{
    assert (db && e);

    if ((db->hdr.flags & VOLDB_FILE) == 0)
	return EINVAL; /* XXX */

    if (db->hdr.num < num)
	return EINVAL; /* XXX */

#if defined(BYTE_ORDER) &&  BYTE_ORDER	== BIG_ENDIAN && 0
    memcpy (&db->u.file[num], e, sizeof (*e));
#else
    {
	u_int32_t tmp;
	unsigned char *ptr = (unsigned char *)&db->u.file[num];

	tmp = htonl(e->nextptr);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->unique);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ino);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare2);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->InterfaceVersion);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->FileType);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->LinkCount);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Length);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare3);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->DataVersion);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Author);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Owner);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->UnixModeBits);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ParentVnode);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ParentUnique);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->SegSize);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->ServerModTime);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare4);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->Group);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);

	tmp = htonl(e->spare5);
	memcpy (ptr, &tmp, sizeof(tmp)); ptr += sizeof(tmp);
    }
#endif
    return 0;
}


/*
 * write `hdr' to `fd' using write.
 */

static int
write_hdr (int fd, struct voldb_header *hdr)
{
    int ret;
    u_int32_t tmp[4];
    struct iovec iov[4];

    ret = lseek (fd, 0, SEEK_SET);
    if (ret)
	return ret;
    
    tmp[0] = htonl(hdr->magic);
    iov[0].iov_base = (void *)&tmp[0] ; iov[0].iov_len = sizeof(tmp[0]);
    tmp[1] = htonl(hdr->num);
    iov[1].iov_base =  (void *)&tmp[1] ; iov[1].iov_len = sizeof(tmp[1]);
    tmp[2] = htonl(hdr->freeptr);
    iov[2].iov_base =  (void *)&tmp[2] ; iov[2].iov_len = sizeof(tmp[2]);
    tmp[3] = htonl(hdr->flags);
    iov[3].iov_base =  (void *)&tmp[3] ; iov[3].iov_len = sizeof(tmp[3]);
    
    ret = writev (fd, iov, 4);
    if (ret != 4*4)
	return errno; /* XXX */
    return 0;
}


/*
 * create a db header in a empty existing file pointed by `fd'
 * with flags set to `flags'.
 */

int
voldb_create_header (int fd, int flags)
{
    struct stat sb;
    struct voldb_header hdr;
    int ret;

    ret = fstat (fd, &sb);
    if (ret)
	return errno; /* XXX */

    if (sb.st_size != 0)
	return EINVAL; /* XXX */

    ret = ftruncate (fd, sizeof (hdr));
    if (ret)
	return errno; /* XXX */

    hdr.magic = VOLDB_MAGIC_HEADER;
    hdr.num = 0;
    hdr.freeptr = 0;
    hdr.flags = flags;

    return write_hdr (fd, &hdr);
}

/*
 * Assure `db' is stored to disk.
 */

int
voldb_flush (struct voldb *db)
{
    int ret;
    assert (db);
    ret = msync (db->ptr, db->size, 0);
    if (ret)
	return errno;
    return 0;
}

/*
 * Extend the db and add more entries.
 */

int
voldb_extend_db (struct voldb *db, unsigned int num)
{
    int dirp = 0, ret;
    size_t newsize;
    u_int32_t oldnum = db->hdr.num;

    if (db->hdr.magic != VOLDB_MAGIC_HEADER)
	return EINVAL; /* XXX */

    if ((db->hdr.flags & (VOLDB_DIR|VOLDB_FILE)) == 0)
	return EINVAL; /* XXX */
    else if (db->hdr.flags & VOLDB_DIR) {
	dirp = 1;
	newsize = db->size + num * VOLDB_DIR_SIZE;
    } else if (db->hdr.flags & VOLDB_FILE)
	newsize = db->size + num * VOLDB_FILE_SIZE;

    ret = ftruncate (db->fd, newsize);
    if (ret)
	return errno;
    
    ret = munmap (db->ptr, db->size);
    if (ret)
	return errno;

    db->size = newsize;
    db->hdr.num += num;

    db->ptr = mmap (NULL, db->size, PROT_READ|PROT_WRITE,
		       MAP_SHARED, db->fd, 0);
    if (db->ptr == MAP_FAILED) 
	return errno; /* XXX */

    set_volhdr_union_ptr (db);

    if (dirp) {
	u_int32_t i = oldnum;
	struct voldb_dir_entry e;

	memset (&e, 0, sizeof (e));

	while (i < db->hdr.num) {
	    if (i < db->hdr.num - 1)
		e.nextptr = i + 1;
	    else
		e.nextptr = db->hdr.freeptr;
	    ret = voldb_put_dir (db, i, &e);
	    i++;
	    if (ret) {
		abort();  /* XXX need ftruncate and remap */
		return ret;
	    }
	}
    } else {
	u_int32_t i = oldnum;
	struct voldb_file_entry e;

	memset (&e, 0, sizeof (e));

	while (i < db->hdr.num) {
	    if (i < db->hdr.num - 1)
		e.nextptr = i + 1;
	    else
		e.nextptr = db->hdr.freeptr;
	    ret = voldb_put_file (db, i, &e);
	    i++;
	    if (ret) {
		abort();  /* XXX need ftruncate and remap */
		return ret;
	    }
	}
    }

    db->hdr.freeptr = oldnum;
    
    ret = write_hdr (db->fd, &db->hdr);
    if (ret)
	return errno;

    return 0;
}

int
voldb_new_entry (struct voldb *db, u_int32_t *num, u_int32_t *unique)
{
    u_int32_t oldfreeptr, newfreeptr, newunique;
    int ret;

    if (db->hdr.freeptr == 0) {
	ret = voldb_extend_db (db, VOLDB_ENTEND_NUM);
	if (ret)
	    return ret;
    }

    oldfreeptr = db->hdr.freeptr;

    if ((db->hdr.flags & VOLDB_DIR) == VOLDB_DIR) {
	struct voldb_dir_entry e;

	ret = voldb_get_dir (db, oldfreeptr, &e);
	if (ret)
	    return ret;

	newfreeptr = e.nextptr;
	newunique = ++e.unique;
	e.nextptr = 0;

	ret = voldb_put_dir (db, oldfreeptr, &e);
	if (ret)
	    return ret;

    } else if ((db->hdr.flags & VOLDB_FILE) == VOLDB_FILE) {
	struct voldb_file_entry e;

	ret = voldb_get_file (db, oldfreeptr, &e);
	if (ret)
	    return ret;

	newfreeptr = e.nextptr;
	newunique = ++e.unique;
	e.nextptr = 0;

	ret = voldb_put_file (db, oldfreeptr, &e);
	if (ret)
	    return ret;
    } else
	abort();
    
    db->hdr.freeptr = newfreeptr;
    ret = write_hdr (db->fd, &db->hdr);
    if (ret) {
	abort();
	return ret;
    }

    *num = oldfreeptr;
    *unique = newunique;
   
    return 0;
}

/*
 *
 */

int
voldb_del_entry (struct voldb *db, u_int32_t num, int32_t *ino)
{
    int ret;

    assert (ino);

    if ((db->hdr.flags & VOLDB_DIR) == VOLDB_DIR) {
	struct voldb_dir_entry e;

	ret = voldb_get_dir (db, num, &e);
	if (ret)
	    return ret;

	assert (e.nextptr == 0);
	e.nextptr = db->hdr.freeptr;
	*ino = e.ino;
	e.ino = 0;

	ret = voldb_put_dir (db, num, &e);
	if (ret)
	    return ret;

    } else if ((db->hdr.flags & VOLDB_FILE) == VOLDB_FILE) {
	struct voldb_file_entry e;

	ret = voldb_get_file (db, num, &e);
	if (ret)
	    return ret;

	assert (e.nextptr == 0);
	e.nextptr = db->hdr.freeptr;
	*ino = e.ino;
	e.ino = 0;

	ret = voldb_put_file (db, num, &e);
	if (ret)
	    return ret;
    } else
	abort();

    db->hdr.freeptr = num;

    ret = write_hdr (db->fd, &db->hdr);
    if (ret) {
	abort();
	return ret;
    }
    return 0;
}


/*
 *
 */

int
voldb_header_info (struct voldb *db, 
		   u_int32_t *num,
		   u_int32_t *freeptr,
		   u_int32_t *flags)
{
    if (num)
	*num = db->hdr.num;
    if (freeptr)
	*freeptr = db->hdr.freeptr;
    if (flags)
	*flags = db->hdr.flags;

    return 0;
}

/*
 *
 */

int
voldb_pretty_print_file (struct voldb_file_entry *e)
{
    printf ("  nextptr		= %d\n", e->nextptr);
    printf ("  unique		= %d\n", e->unique);
    printf ("  ino		= 0x%x\n", e->ino);
    printf ("  InterfaceVersion = %d\n", e->InterfaceVersion);
    printf ("  FileType		= %d\n", e->FileType);
    printf ("  LinkCount	= %d\n", e->LinkCount);
    printf ("  Length		= %d\n", e->Length);
    printf ("  DataVersion	= %d\n", e->DataVersion);
    printf ("  Author		= %d\n", e->Author);
    printf ("  Owner		= %d\n", e->Owner);
    printf ("  UnixModeBits	= 0%o\n", e->UnixModeBits);
    printf ("  ParentVnode	= %d\n", e->ParentVnode);
    printf ("  ParentUnique	= %d\n", e->ParentUnique);
    printf ("  SegSize		= %d\n", e->SegSize);
    printf ("  ServerModTime	= %d\n", e->ServerModTime);
    printf ("  Group		= %d\n", e->Group);

    return 0;
}

int
voldb_pretty_print_dir (struct voldb_dir_entry *e)
{
    int i;

    printf ("  nextptr		= %d\n", e->nextptr);
    printf ("  unique		= %d\n", e->unique);
    printf ("  ino		= 0x%x\n", e->ino);
    printf ("  InterfaceVersion = %d\n", e->InterfaceVersion);
    printf ("  FileType		= %d\n", e->FileType);
    printf ("  LinkCount	= %d\n", e->LinkCount);
    printf ("  Length		= %d\n", e->Length);
    printf ("  DataVersion	= %d\n", e->DataVersion);
    printf ("  Author		= %d\n", e->Author);
    printf ("  Owner		= %d\n", e->Owner);
    printf ("  UnixModeBits	= 0%o\n", e->UnixModeBits);
    printf ("  ParentVnode	= %d\n", e->ParentVnode);
    printf ("  ParentUnique	= %d\n", e->ParentUnique);
    printf ("  SegSize		= %d\n", e->SegSize);
    printf ("  ServerModTime	= %d\n", e->ServerModTime);
    printf ("  Group		= %d\n", e->Group);

    printf ("  ACL:");
    for (i = 0; i < FS_MAX_ACL; i++)
	printf ("    owner %d flags 0x%x\n", 
		e->acl[i].owner, e->acl[i].flags);

    return 0;
}
