/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: vld.c,v 1.10 1999/07/18 19:51:13 map Exp $ */

#include <sys/types.h>
#include <stdio.h>

#include <assert.h>

#include <rx/rx.h>

#include <vldb.h>
#include <pts.h>

#include <voldb.h>
#include <vld.h>

#include <svol.h>
#include <vstatus.h>
#include <voldb.h>

#include <mdir.h>

#include <err.h>

#include <hash.h>
#include <list.h>

#include <errno.h>

/*
 * Local define and limitations
 */

int vld_storestatus_to_dent (struct voldb_dir_entry *e, 
			     const AFSStoreStatus *ss, prlist *cps);
int vld_storestatus_to_fent (struct voldb_file_entry *e, 
			     const AFSStoreStatus *ss, prlist *cps);

#define VLD_MAX_BACKSTORE_TYPES 10

/*
 * Local variables
 */

/* volume types */
static vol_op *backstoretypes[VLD_MAX_BACKSTORE_TYPES];

/* hashtable of volumes */
static Hashtab *volume_htab = NULL;
static int volume_htab_sz = 1024;

/* lru for db in use */
static List *db_lru = NULL;

/*
 * Useful macro's and short functions
 */

#define VLD_VALID_BACKSTORETYPE(backstoretype) (!(backstoretype >= 0 && backstoretype < VLD_MAX_BACKSTORE_TYPES && backstoretypes[backstoretype] != NULL))

static int 
VLD_VALID_VOLNAME(const char *volname) 
{
    while (volname && *volname) {
	if (!((*volname >= 'a' && *volname <= 'z') ||
	      (*volname >= 'A' && *volname <= 'Z') ||
	      (*volname >= '0' && *volname <= '9') ||
	      *volname == '.'))
	    return *volname;
	volname++;
    }
    return 0;
}

#define VOLOP_OPEN(type,part,volid,flags,vol) ((backstoretypes[(type)]->open)((part),(volid),(flags),(vol)))
#define VOLOP_FREE(vol) ((backstoretypes[(vol)->type]->free)(vol))
#define VOLOP_ICREATE(vol,ino) ((backstoretypes[(vol)->type]->icreate)(vol,ino))
#define VOLOP_IOPEN(vol,ino,flags,fd) ((backstoretypes[(vol)->type]->iopen)(vol,ino,flags,fd))
#define VOLOP_IUNLINK(vol,ino) ((backstoretypes[(vol)->type]->iunlink)(vol,ino))
#define VOLOP_REMOVE(vol) ((backstoretypes[(vol)->type]->remove)(vol))


/*
 * Translate to real-name
 */

const char *
vld_backstoretype_name (int32_t backstoretype)
{
    if (VLD_VALID_BACKSTORETYPE(backstoretype))
	return "UKWN";

    return backstoretypes[backstoretype]->name;
}

/*
 * Bootstrap
 */

int
vld_boot (void)
{
    memset(backstoretypes, 0, sizeof (backstoretypes));
    backstoretypes[VLD_SVOL] = &svol_volume_ops;
    return 0;
}

/*
 * Free volume `vol'
 */

void
vld_free (volume_handle *vol)
{
    
    if (--vol->ref == 0) {
	VOLOP_FREE(vol);
	if (vol->flags.voldbp) {
	    voldb_close (vol->db[0]);
	    voldb_close (vol->db[1]);
	    vol->flags.voldbp = FALSE;
	} else {
	    assert (vol->db[0] == NULL);
	    assert (vol->db[1] == NULL);
	}
	dp_free (vol->dp);
	free (vol);
    }    
}

/*
 * Ref `vol'
 */

void
vld_ref (volume_handle *vol)
{
    assert (vol->ref);
    vol->ref++;
}

/*
 * create a `vol' from the bootstrap infomation in `vs'.
 */

static int
vstatus2volume_handle (vstatus *vs, struct dp_part *dp, volume_handle **vol)
{
    volume_handle *v;

    assert (vs && dp && vol);

    v = malloc (sizeof(*v));
    if (v == NULL)
	return ENOMEM;

    memset (v, 0, sizeof(*v));
    v->ref = 1;
    v->vol  = vs->volid;
    dp_ref (dp);
    v->dp   = dp;
    v->type = vs->type;
    v->sino = vs->volinfoinode;
    v->dino = vs->dirinode;
    v->fino = vs->fileinode;
    v->flags.infop = FALSE;
    v->flags.voldbp = FALSE;
    v->flags.offlinep = FALSE;

    *vol = v;

    return 0;
}

/*
 * Read in all partitions and register all volumes.
 */

static void
register_vols_cb (void *data, int fd)
{
    struct dp_part *dp = data;
    vstatus vs;
    int ret;
    volume_handle *vol;

    ret = vstatus_read (fd, &vs);
    if (ret)
	return;

    ret = vstatus2volume_handle (&vs, dp, &vol);
    if (ret)	
	return;

    ret = VOLOP_OPEN(vol->type, dp, vol->vol, 
		     VOLOP_NOFLAGS, &vol->data);
    if (ret)
	vol->flags.offlinep = TRUE;

    hashtabadd (volume_htab, vol);
    
    return;
}

/*
 *
 */

static int
volume_cmp (void *ptr1, void *ptr2)
{
    volume_handle *v1 = (volume_handle *) ptr1;
    volume_handle *v2 = (volume_handle *) ptr2;

    return v1->vol - v2->vol;
}

/*
 *
 */

static unsigned
volume_hash (void *ptr)
{
    volume_handle *v = (volume_handle *) ptr;

    return v->vol;
}

/*
 *
 */

int
vld_init (void)
{
    struct dp_part *dp;
    int ret, partnum, i;
    

    db_lru = listnew();
    if (db_lru == NULL)
	errx (1, "vld_init: db_lru == NULL");

    for (i = 0; i < 100 /* XXX */ ; i++)
	listaddhead (db_lru, NULL);

    volume_htab = hashtabnew(volume_htab_sz, volume_cmp, volume_hash);
    if (volume_htab == NULL)
	errx (1, "vld_init: volume_htab == NULL");

    for (partnum = 0; partnum < 'z'-'a'; partnum++) {

	ret = dp_create (partnum , &dp);
	if (ret) {
	    warnx ("vld_init: dp_create(%d) returned %d", partnum, ret);
	    continue;
	}
	
	ret = dp_findvol (dp, register_vols_cb, dp);
	if (ret)
	    warnx ("vld_init: dp_findvol returned: %d", ret);

	dp_free (dp);
    }	
    return 0;
}

/*
 *
 */

static int32_t
dir_afs2local (int32_t vno)
{
    if (vno > 1)
	vno -= 1;
    return vno >> 1;
}
    
static int32_t
dir_local2afs (int32_t vno)
{
    vno = vno << 1;
    if (vno == 0)
	return 1;
    return vno + 1;
}

static int32_t
file_afs2local (int32_t vno)
{
    return (vno >> 1) - 1;
}
    
static int32_t
file_local2afs (int32_t vno)
{
    return (vno + 1) << 1 ;
}

/*
 *
 */

int
vld_create_file (volume_handle *vol, const AFSFid *parent, AFSFid *child,
		 int32_t type, const AFSStoreStatus *ss, prlist *cps)
{
    struct voldb_file_entry e;
    ino_t rootdir;
    int ret;
    u_int32_t real_vnode, unique;

    assert (type == TYPE_FILE || type == TYPE_LINK);

    ret = voldb_new_entry (vol->db[1], &real_vnode, &unique);
    if (ret)
	return ret;

    ret = voldb_get_file (vol->db[1], real_vnode, &e);
    if (ret)
	return ret;
    
    child->Vnode = file_local2afs(real_vnode);
    child->Unique = unique;

    ret = VOLOP_ICREATE(vol, &rootdir);
    if (ret)
	abort();
    
    e.ino = rootdir;
    e.FileType = type;
    e.LinkCount = 1;
    e.ParentVnode = parent->Vnode;
    e.ParentUnique = parent->Unique;

    ret = vld_storestatus_to_fent (&e, ss, cps);
    if (ret)
	abort();
    
    ret = voldb_put_file (vol->db[1], real_vnode, &e);
    if (ret)
	abort();
    
    return 0;
}

/*
 *
 */

int
vld_create_directory (volume_handle *vol, const AFSFid *parent, AFSFid *child,
		      const AFSStoreStatus *ss, prlist *cps)
{
    struct voldb_dir_entry e;
    ino_t rootdir;
    int ret, fd;
    u_int32_t real_vnode, unique;

    ret = voldb_new_entry (vol->db[0], &real_vnode, &unique);
    if (ret)
	return ret;

    ret = voldb_get_dir (vol->db[0], real_vnode, &e);
    if (ret)
	return ret;
    
    child->Vnode = dir_local2afs(real_vnode);
    child->Unique = unique;

    ret = VOLOP_ICREATE(vol, &rootdir);
    if (ret)
	abort();
    
    ret = VOLOP_IOPEN(vol,rootdir,O_RDWR,&fd);
    if (ret) {
	VOLOP_IUNLINK(vol,rootdir);
	/* delete entry */
	abort();
	return ret;
    }
	
    
    {
	AFSFid dot, dot_dot;

	dot_dot.Volume = dot.Volume = vol->vol;
	dot_dot.Vnode = parent->Vnode;
	dot_dot.Unique = parent->Unique;
	dot.Vnode = child->Vnode;
	dot.Unique = child->Unique;
	
	ret = mdir_mkdir (fd, dot, dot_dot);
	if (ret)
	    abort(); /* XXX */
    }

    e.ino = rootdir;
    e.FileType = TYPE_DIR;
    e.LinkCount = 2;
    e.ParentVnode = parent->Vnode;
    e.ParentUnique = parent->Unique;
    
    ret = vld_storestatus_to_dent (&e, ss, cps);
    if (ret)
	abort();

    ret = voldb_put_dir (vol->db[0], real_vnode, &e);
    if (ret)
	abort();
    
    return 0;
}

int
vld_adjust_linkcount (volume_handle *vol, int32_t vno, int adjust)
{
    int32_t real_vnode;
    int ret;

    assert (vol);
    
    if (vno & 1) {
	struct voldb_dir_entry e;

	real_vnode = dir_afs2local(vno);
	
	ret = voldb_get_dir (vol->db[0], real_vnode, &e);
	if (ret)
	    return ret;
	
	e.LinkCount+=adjust;
	
	ret = voldb_put_dir (vol->db[0], real_vnode, &e);
	if (ret)
	    return ret;
	
    } else {
	struct voldb_file_entry e;

	real_vnode = dir_afs2local(vno);
	
	ret = voldb_get_file (vol->db[1], real_vnode, &e);
	if (ret)
	    return ret;
	
	e.LinkCount+=adjust;
	
	ret = voldb_put_file (vol->db[1], real_vnode, &e);
	if (ret)
	    return ret;
    }
    return 0;
}

/*
 * create a new volume on partition `part' with volume id `volid' and name
 * `name' of type `backstoretype'
 */

int
vld_create_volume (struct dp_part *dp, int32_t volid, 
		   char *name, int32_t backstoretype)
{
    volume_handle *vol;
    int ret;

    if (VLD_VALID_BACKSTORETYPE(backstoretype) ||
	VLD_VALID_VOLNAME(name))
	return EINVAL;

    if (volume_htab) {
	ret = vld_find_vol (volid, &vol);
	if (ret == 0) {
	    vld_free (vol);
	    return EEXIST;
	} else if (ret != ENOENT)
	    return ret;
    } else {
	ret = vld_open_volume_by_num (dp, volid, &vol);
	if (ret == 0) {
	    vld_free (vol);
	    return EEXIST;
	} else if (ret != ENOENT)
	    return ret;
    }

    vol = malloc (sizeof(*vol));
    if (vol == NULL)
	return ENOMEM;

    memset (vol, 0, sizeof (*vol));
    vol->vol = volid;
    dp_ref (dp);
    vol->dp = dp;

    ret = VOLOP_OPEN(backstoretype, dp, volid, VOLOP_CREATE, &vol->data);
    if (ret) {
	dp_free (dp);
	return ret;
    }

    ret = VOLOP_ICREATE(vol, &vol->fino);
    if (ret) {
	VOLOP_REMOVE (vol);
	dp_free (dp);
	return ret;
    }

    ret = VOLOP_ICREATE(vol, &vol->dino);
    if (ret) {
	VOLOP_IUNLINK(vol, vol->fino);
	VOLOP_REMOVE(vol);
	dp_free (dp);
	return ret;
    }

    ret = VOLOP_ICREATE(vol, &vol->sino);
    if (ret) {
	VOLOP_IUNLINK(vol, vol->fino);
	VOLOP_IUNLINK(vol, vol->dino);
	VOLOP_REMOVE(vol);
	dp_free (dp);
	return ret;
    }

    {
	int fd;

	ret = VOLOP_IOPEN(vol,vol->sino,O_RDWR,&fd);
	if (ret)
	    abort();

	ret = vol_create (fd, volid, name, RWVOL, 0);
	if (ret) {
	    VOLOP_IUNLINK(vol, vol->fino);
	    VOLOP_IUNLINK(vol, vol->dino);
	    VOLOP_IUNLINK(vol, vol->sino);
	    VOLOP_REMOVE (vol);
	    close (fd);
	    dp_free (dp);
	    return ret;
	}
	close (fd);
    }

    {
	char path[MAXPATHLEN];
	int fd = -1;
	vstatus vs;

#define ERR_OUT_VSTATUS(ret) \
	do { \
	VOLOP_IUNLINK(vol, vol->fino); \
	VOLOP_IUNLINK(vol, vol->dino); \
	VOLOP_IUNLINK(vol, vol->sino); \
	VOLOP_REMOVE (vol); \
	dp_free (dp); \
	if (fd != -1) \
	    close (fd); \
	return ret; \
	} while (0)
	
	ret = vol_getfullname (DP_NUMBER(dp), volid, path, sizeof(path));
	if (ret)
	    ERR_OUT_VSTATUS(ret);

	fd = open (path, O_RDWR|O_CREAT, 0600);
	if (fd < 0)
	    ERR_OUT_VSTATUS(errno);

	if (sizeof (vs.volinfoinode) != sizeof(vol->sino))
	    abort();

	memset (&vs, 0, sizeof(vs));
	vs.volid = volid;
	vs.type = vol->type;
	vs.volinfoinode = vol->sino;
	vs.dirinode = vol->dino;
	vs.fileinode = vol->fino;

	ret = vstatus_write (fd, &vs);
	if (ret)
	    ERR_OUT_VSTATUS(ret);

	close (fd);
#undef ERR_OUT_VSTATUS
    }

    {
	int fd;
	ret = VOLOP_IOPEN(vol,vol->fino,O_RDWR,&fd);
	if (ret)
	    abort();
	    
	ret = voldb_create_header (fd, VOLDB_FILE);
	if (ret)
	    abort();
	close (fd);

	ret = VOLOP_IOPEN(vol,vol->dino,O_RDWR,&fd);
	if (ret)
	    abort();
	    
	ret = voldb_create_header (fd, VOLDB_DIR);
	if (ret)
	    abort();

	close (fd);
    }
    {
	AFSFid fid, child;
	AFSStoreStatus ss;

	memset (&ss, 0, sizeof (ss));

	ret = vld_db_uptodate (vol);
	if (ret)
	    abort();

	fid.Volume = vol->vol;
	fid.Vnode = 1;
	fid.Unique = 1;
	ret = vld_create_directory (vol, &fid, &child, &ss, NULL);
	if (ret)
	    abort();
	
	assert (child.Vnode == 1 && child.Unique == 1);

	ret = vld_db_flush (vol);
	if (ret)
	    abort();
    }
	    
    vld_free (vol);

    return 0;
}

/*
 * Register a new volume type
 */

int
vld_register_vol_type (const int32_t backstoretype, vol_op *ops)
{
    if (backstoretype < 0 || backstoretype > VLD_MAX_BACKSTORE_TYPES)
	return EINVAL;

    assert (ops);

    backstoretypes[backstoretype] = ops;

    return 0;
}

/*
 * Find volume in incore db
 */

int
vld_find_vol (const int32_t volid, struct volume_handle **vol)
{
    volume_handle key, *r;

    key.vol = volid;
    
    r = hashtabsearch (volume_htab, &key);
    if (r == NULL) {
	*vol = NULL;
	return ENOENT;
    } else {
	vld_ref (r);
	*vol = r;
    }

    return 0;
}

static int
verify_parent_privs (int32_t vnode, int32_t unique, int operation, prlist *cps)
{
    /* XXX */
    return 0;
}

/*
 * return a non zero value if the user is the superuser.
 */

#define SUPER_USER 0

int
super_user (prlist *cps)
{
    /* XXX */
    return 1;
}

/*
 *
 */

static int
vld_dent_to_fetchstatus (struct voldb_dir_entry *e, AFSFetchStatus *fs, 
			 int fd, prlist *cps)
{
    int ret;

    ret = verify_parent_privs (e->ParentVnode, e->ParentUnique, 0/*opr*/, cps);
    if (ret)
	return ret;
    
    ret = vld_update_size_fd (fs, fd);
    if (ret)
	return ret;

    fs->InterfaceVersion = 1;
    fs->FileType 	= e->FileType;
    fs->LinkCount 	= e->LinkCount;
    fs->DataVersion 	= e->DataVersion;
    
    fs->ParentVnode	= e->ParentVnode;
    fs->ParentUnique	= e->ParentUnique;
    
    fs->SegSize 	= e->SegSize;
    fs->ClientModTime 	= e->ServerModTime;
    fs->ServerModTime	= e->ServerModTime;
    fs->SyncCount 	= 4711;
    fs->spare1		= 0;
    fs->spare2		= 0;
    fs->spare3		= 0;
    fs->spare4		= 0;
    fs->Author		= e->Author;
    fs->Owner		= e->Owner;
    fs->Group		= e->Group;
    fs->UnixModeBits	= e->UnixModeBits;
    
    /*
     * Dummy
     */
    
    fs->CallerAccess = fs->AnonymousAccess =
	S_IRUSR | S_IWUSR | S_IXUSR | 
	S_IRGRP | S_IWGRP | S_IXGRP | 
	S_IROTH | S_IWOTH | S_IXOTH ;

    return 0;
}

/*
 *
 */

int
vld_storestatus_to_dent (struct voldb_dir_entry *e, 
			 const AFSStoreStatus *ss, prlist *cps)
{
    int ret;

    ret = verify_parent_privs (e->ParentVnode, e->ParentUnique, 0/*opr*/, cps);
    if (ret)
	return ret;
    
    if (ss->Mask & SS_MODTIME) {
	e->ServerModTime = ss->ClientModTime;
    } else {
	struct timeval tv;
	gettimeofday (&tv, NULL);
	e->ServerModTime = tv.tv_sec;
    }
    if (ss->Mask & SS_OWNER) {
	if (ss->Owner == SUPER_USER && !super_user (cps))
	    return EPERM;
	e->Owner = ss->Owner;
    }
    if (ss->Mask & SS_GROUP)
	e->Group = ss->Group;
    if (ss->Mask & SS_MODEBITS)
	e->UnixModeBits = 0777 & ss->UnixModeBits;
    if (ss->Mask & SS_SEGSIZE)
	e->SegSize = ss->SegSize;
    
    return 0;
}

/*
 *
 */

int
vld_storestatus_to_fent (struct voldb_file_entry *e, 
			 const AFSStoreStatus *ss, prlist *cps)
{
    int ret;

    ret = verify_parent_privs (e->ParentVnode, e->ParentUnique, 0/*opr*/, cps);
    if (ret)
	return ret;
    
    if (ss->Mask & SS_MODTIME) {
	e->ServerModTime = ss->ClientModTime;
    } else {
	struct timeval tv;
	gettimeofday (&tv, NULL);
	e->ServerModTime = tv.tv_sec;
    }
    if (ss->Mask & SS_OWNER) {
	if (ss->Owner == SUPER_USER && !super_user (cps))
	    return EPERM;
	e->Owner = ss->Owner;
    }
    if (ss->Mask & SS_GROUP)
	e->Group = ss->Group;
    if (ss->Mask & SS_MODEBITS)
	e->UnixModeBits = 0777 & ss->UnixModeBits;
    if (ss->Mask & SS_SEGSIZE)
	e->SegSize = ss->SegSize;
    
    return 0;
}

/*
 *
 */

static int
vld_fent_to_fetchstatus (struct voldb_file_entry *e, AFSFetchStatus *fs, 
			 int fd, prlist *cps)
{
    int ret;

    ret = verify_parent_privs (e->ParentVnode, e->ParentUnique, 0/*opr*/, cps);
    if (ret)
	return ret;
    
    ret = vld_update_size_fd (fs, fd);
    if (ret)
	return ret;

    fs->InterfaceVersion = 1;
    fs->FileType 	= e->FileType;
    fs->LinkCount 	= e->LinkCount;
    fs->DataVersion 	= e->DataVersion;
    
    fs->ParentVnode	= e->ParentVnode;
    fs->ParentUnique	= e->ParentUnique;
    
    fs->SegSize 	= e->SegSize;
    fs->ClientModTime 	= e->ServerModTime;
    fs->ServerModTime	= e->ServerModTime;
    fs->SyncCount 	= 4711;
    fs->spare1		= 0;
    fs->spare2		= 0;
    fs->spare3		= 0;
    fs->spare4		= 0;
    fs->Author		= e->Author;
    fs->Owner		= e->Owner;
    fs->Group		= e->Group;
    fs->UnixModeBits	= e->UnixModeBits;
    
    /*
     * Dummy
     */
    
    fs->CallerAccess = fs->AnonymousAccess =
	S_IRUSR | S_IWUSR | S_IXUSR | 
	S_IRGRP | S_IWGRP | S_IXGRP | 
	S_IROTH | S_IWOTH | S_IXOTH ;

    return 0;
}

/*
 *
 */

/*
 * open an vnode `vno' in volume `vol' with `flags' as specifed above.
 * if `fs' != NULL fetch status for `vno' .
 * if `fd' != NULL return a filedescriptor in `fd' to `vno'.
 * `flags' are ignored if `fd' == NULL.
 */

int
vld_open_vnode (volume_handle *vol, int32_t vno, AFSFetchStatus *fs, 
		int flags, int *rfd, prlist *cps)
{
    int ret, fd;

    assert (vol && fs);

    if (vno & 1) {
	struct voldb_dir_entry e;

	ret = voldb_get_dir (vol->db[0], dir_afs2local(vno), &e);
	if (ret)
	    return ret;
	
	ret = VOLOP_IOPEN(vol, e.ino, flags, &fd);
	if (ret)
	    return ret;

	ret = vld_dent_to_fetchstatus (&e, fs, fd, cps);
	if (ret) {
	    close(fd);
	    return ret;
	}

    } else {
	struct voldb_file_entry e;

	ret = voldb_get_file (vol->db[1], file_afs2local(vno), &e);
	if (ret)
	    return ret;
	
	ret = VOLOP_IOPEN(vol, e.ino, flags, &fd);
	if (ret)
	    return ret;

	ret = vld_fent_to_fetchstatus (&e, fs, fd, cps);
	if (ret) {
	    close (fd);
	    return ret;
	}

    }
    if (rfd)
	*rfd = fd;
    else
	close (fd);
    return 0;
}

/*
 *
 */

int
vld_update_size_fd (AFSFetchStatus *fs, int fd)
{
    struct stat sb;
    int ret;

    ret = fstat (fd, &sb);
    if (ret)
	return errno;

    fs->Length = sb.st_size;
    return 0;
}

/*
 *
 */

int
vld_update_size_fbuf (AFSFetchStatus *fs, fbuf *the_fbuf)
{
    fs->Length = the_fbuf->len;
    return 0;
}

/*
 *
 */

int
vld_modify_vnode (volume_handle *vol, int32_t vno, const AFSStoreStatus *ss,
		  AFSFetchStatus *fs, int flags, int *rfd, prlist *cps)
{
    int ret, fd;
    int32_t real_vnode;

    assert (vol && fs);

    if (vno & 1) {
	struct voldb_dir_entry e;

	real_vnode = dir_afs2local(vno);

	ret = voldb_get_dir (vol->db[0], real_vnode, &e);
	if (ret)
	    return ret;
	
	ret = vld_storestatus_to_dent (&e, ss, cps);
	if (ret)
	    return ret;

	ret = voldb_put_dir (vol->db[0], real_vnode, &e);
	if (ret) {
	    close(fd);
	    return ret;
	}

	ret = VOLOP_IOPEN(vol, e.ino, flags, &fd);
	if (ret)
	    return ret;

	ret = vld_dent_to_fetchstatus (&e, fs, fd, cps);
	if (ret) {
	    close(fd);
	    return ret;
	}

    } else {
	struct voldb_file_entry e;

	real_vnode = dir_afs2local(vno);

	ret = voldb_get_file (vol->db[1], real_vnode, &e);
	if (ret)
	    return ret;
	
	ret = vld_storestatus_to_fent (&e, ss, cps);
	if (ret)
	    return ret;

	ret = voldb_put_file (vol->db[1], real_vnode, &e);
	if (ret) {
	    close(fd);
	    return ret;
	}

	ret = VOLOP_IOPEN(vol, e.ino, flags, &fd);
	if (ret)
	    return ret;

	ret = vld_fent_to_fetchstatus (&e, fs, fd, cps);
	if (ret) {
	    close(fd);
	    return ret;
	}
    }
    if (rfd)
	*rfd = fd;
    else
	close (fd);
    return 0;
}


/*
 * Convert a volume handle to AFS volsync
 */

int
vld_vld2volsync (const struct volume_handle *vld, AFSVolSync *volsync)
{
    if (volsync)
	memset (volsync, 0, sizeof (*volsync));
    return 0;
}

/*
 * Make sure the db is stored to disk
 */

int
vld_db_flush (volume_handle *vol)
{
    if (vol->flags.voldbp) {
	assert (vol->db[0] != NULL && vol->db[1] != NULL);

	voldb_flush (vol->db[0]);
	voldb_flush (vol->db[1]);
    } else {
	assert (vol->db[0] == NULL && vol->db[1] == NULL);
    }
    return 0;
}

/*
 * Bring db in volume uptodate
 */

int
vld_db_uptodate (volume_handle *vol)
{
    int ret, fd;

    if (vol->flags.voldbp) {
	if (db_lru) {
	    listdel (db_lru, vol->db_li);
	    vol->db_li = listaddhead (db_lru, vol);
	    if (vol->db_li == NULL)
		abort (); /* XXX */
	} else {
	    assert (vol->db_li == NULL);
	}
	return 0;
    }
    assert (vol->db[0] == NULL && vol->db[1] == NULL);

    if (db_lru && !listemptyp(db_lru)) {
	volume_handle *old_vol;

	old_vol = listdeltail (db_lru);
	if (old_vol) {
	    assert (old_vol->db[0] && old_vol->db[1]);
	    
	    voldb_close (old_vol->db[0]);
	    voldb_close (old_vol->db[1]);
	    old_vol->db[0] = old_vol->db[0] = NULL;
	    old_vol->db_li = NULL;
	}
    }
	    
    ret = VOLOP_IOPEN(vol, vol->dino, O_RDWR, &fd);
    if (ret)
	return ret;

    ret = voldb_init (fd, &vol->db[0]);
    if (ret) {
	close (fd);
	return ret;
    }

    ret = VOLOP_IOPEN(vol, vol->fino, O_RDWR, &fd);
    if (ret) {
	voldb_close (vol->db[0]);
	vol->db[0] = NULL;
	return ret;
    }

    ret = voldb_init (fd, &vol->db[1]);
    if (ret) {
	close (fd);
	voldb_close (vol->db[0]);
	vol->db[0] = NULL;
	vol->db[1] = NULL;
	return ret;
    }

    vol->flags.voldbp = TRUE;

    if (db_lru)
	vol->db_li = listaddhead (db_lru, vol);

    return 0;
}

/*
 * Open volume on partition `dp' with volume id `volid'
 * and return it ref:ed in `vol'.
 */

int
vld_open_volume_by_num (struct dp_part *dp, int32_t volid,
			volume_handle **vol)
{
    int ret, fd;
    char path[MAXPATHLEN];
    vstatus vs;

    if (volume_htab) {
	volume_handle *r, key;
	key.vol = volid;

	r = hashtabsearch (volume_htab, &key);
	if (r == NULL)
	    return ENOENT;
	else
	    vld_ref (r);
	*vol = r;
	return 0;
    }

    ret = vol_getfullname (DP_NUMBER(dp), volid, path, sizeof (path));
    if (ret)
	return ret;

    fd = open (path, O_RDONLY, 0600);
    if (fd < 0)
	return errno;

    ret = vstatus_read (fd, &vs);
    if (ret) {
	close (fd);
	return ret;
    }
    close (fd);

    ret = vstatus2volume_handle (&vs, dp, vol);
    if (ret)
	return ret;

    ret = VOLOP_OPEN((*vol)->type, (*vol)->dp, (*vol)->vol, 
		     VOLOP_NOFLAGS, &(*vol)->data);
    if (ret)
	vld_free (*vol);

    return ret;
}

int
vld_remove_node (volume_handle *vol, int32_t node)
{
    int ret;
    int32_t ino; /* XXX */

    if (node & 1) {
	ret = voldb_del_entry (vol->db[0], dir_afs2local(node), &ino);
	if (ret)
	    return ret;

    } else {
	ret = voldb_del_entry (vol->db[1], file_afs2local(node), &ino);
	if (ret)
	    return ret;

    }

    return VOLOP_IUNLINK (vol, ino);
}


/*
 * Open volume that `fd' points to, if volume_hash is loaded try to
 * find the volume in the hash.
 */

int
vld_open_volume_by_fd (struct dp_part *dp, int fd,
		       volume_handle **vol)
{
    int ret;
    vstatus vs;

    ret = vstatus_read (fd, &vs);
    if (ret)
	return ret;

    if (volume_htab) {
	volume_handle *r, key;
	key.vol = vs.volid;

	r = hashtabsearch (volume_htab, &key);
	if (r == NULL)
	    return ENOENT;
	else
	    vld_ref (r);
	*vol = r;
	return 0;
    }

    ret = vstatus2volume_handle (&vs, dp, vol);
    if (ret)
	return ret;

    ret = VOLOP_OPEN((*vol)->type, (*vol)->dp, (*vol)->vol, 
		     VOLOP_NOFLAGS, &(*vol)->data);
    if (ret)
	vld_free (*vol);

    return ret;
}

/*
 * Make sure info is uptodate
 */

int
vld_info_uptodatep (volume_handle *vol)
{
    int ret, fd;

    assert (vol);
    
    if (vol->flags.infop)
	return 0;

    ret = VOLOP_IOPEN(vol, vol->sino, O_RDONLY, &fd);
    if (ret)
	return ret;

    ret = vol_read_header (fd, &vol->info);

    close (fd);
    if (ret == 0)
	vol->flags.infop = TRUE;
    return ret;
}


/*
 * Shutdown time
 */

void
vld_end (void)
{
    /* XXX flush volume_htab to disk */

    return;
}
