/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>

RCSID("$Id: svol.c,v 1.2 1999/06/23 09:22:16 lha Exp $");

/*
 * Description:
 *  svol is a simple (and dumb (and slow)) implemetation of volume-
 *  operations. It store the files in a tree-structure where
 *  inode-number is used as ``key'' to where the node is stored in 
 *  the tree.
 */

#include <sys/types.h>
#include <atypes.h>

#include <rx/rx.h>

#include <pts.h>
#include <dpart.h>
#include <vld.h>

#include <assert.h>

/*
 * Create a svol and return it in `vol' that is generic volume.
 * ignore the flags.
 */

static int
svol_open (struct dp_part *part, int32_t volid, int flags,
	   void **data)
{
    *data = NULL;

    return 0;
}

/*
 * free all svol related to the volume `vol'.
 */

static void
svol_free (volume_handle *vol)
{
    assert (vol->data == NULL);
}

/*
 * create the name of `vol' and `ino' and save it in `name'
 * the `sz' is the size of name and MUST be large enough to store name.
 */

static int
create_name (volume_handle *vol, ino_t ino, char *name, size_t sz)
{
    size_t i;
    
    assert (name);

    if (sizeof(ino_t) != 4)
	abort();

    i = snprintf (name, sz, "%s/%02x/%02x/%02x/%02x",
		  vol->dp->part,
		  (ino >> 24) & 0xff,
		  (ino >> 16) & 0xff,
		  (ino >> 8) & 0xff,
		  ino & 0xff);
    
    if (i >= sz)
	abort();
    return 0;
}

/*
 * create a inode on `vol', return inode number in `ino'.
 */

static int
svol_icreate (volume_handle *vol, ino_t *ino)
{
    char name[MAXPATHLEN];
    char nodename[MAXPATHLEN];
    int fd, error;
    struct stat sb;

    if (sizeof(ino_t) != 4)
	abort();

    snprintf(nodename, sizeof(nodename), "%s/inodeXXXXXX", vol->dp->part);
    fd = mkstemp(nodename);
    if (fd == -1)
	return errno;
    error = fstat(fd, &sb);
    if (error) {
	close(fd);
	unlink(nodename);
	return errno;
    }
    close(fd);

    snprintf(name, sizeof(name), "%s/%02x", vol->dp->part, 
	     (sb.st_ino >> 24) & 0xff);
    mkdir(name, 0700);
    snprintf(name, sizeof(name), "%s/%02x/%02x", vol->dp->part, 
	     (sb.st_ino >> 24) & 0xff, 
	     (sb.st_ino >> 16) & 0xff);
    mkdir(name, 0700);
    snprintf(name, sizeof(name), "%s/%02x/%02x/%02x", vol->dp->part, 
	     (sb.st_ino >> 24) & 0xff, 
	     (sb.st_ino >> 16) & 0xff,
	     (sb.st_ino >> 8) & 0xff);
    mkdir(name, 0700);
    create_name (vol, sb.st_ino, name, sizeof(name));
    snprintf(name, sizeof(name), "%s/%02x/%02x/%02x/%02x", vol->dp->part, 
	     (sb.st_ino >> 24) & 0xff, 
	     (sb.st_ino >> 16) & 0xff,
	     (sb.st_ino >> 8) & 0xff,
	     sb.st_ino & 0xff );

    error = rename(nodename, name);
    if (error) {
	unlink (nodename);
	return errno; /* XXX errno / save errno */
    }

    *ino = sb.st_ino;

    return 0;
}

/*
 * open `ino' in `vol' with open(2) `flags', return filedescriptor in `fd'
 */

static int
svol_iopen (volume_handle *vol, ino_t ino, int flags, int *fd)
{
    char name[MAXPATHLEN];
    int ret;

    ret = create_name (vol, ino, name, sizeof (name));
    if (ret)
	return ret;

    *fd = open (name, flags, 0666);
    if (*fd < 0)
	return errno;

    return 0;
}

static int
svol_unlink (volume_handle *vol, ino_t ino)
{
    char name[MAXPATHLEN];
    int ret;

    ret = create_name (vol, ino, name, sizeof (name));
    if (ret)
	return ret;

    ret = unlink (name);
    if (ret)
	return errno;

    return 0;
}

/*
 *
 */

static int
svol_remove (volume_handle *vol)
{
    svol_free (vol);
    return 0;
}

/*
 *
 */

vol_op svol_volume_ops = {
    "svol",
    svol_open,
    svol_free,
    svol_icreate,
    svol_iopen,
    svol_unlink,
    svol_remove
};    
