/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <dirent.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>

#include <unistd.h>

#include <atypes.h>

#include <err.h>

#include <dpart.h>

RCSID("$Id: dpart.c,v 1.2 1999/07/12 08:32:11 map Exp $");

/*
 * Allocate a dp_part structure for partition `num' and
 * return it in `dp'.
 */

int
dp_create (u_int32_t num, struct dp_part **dp)
{
    struct dp_part *d;
    int ret;

    assert (dp);

    *dp = NULL;

    if (num > 'z' - 'a')
	return EINVAL; /* XXX */

    d = calloc (sizeof (**dp), 1);
    if (d == NULL)
	return ENOMEM; /* XXX */

    ret = asprintf (&d->part, "/vicep%c", num + 'a');
    if (ret != 7) {
	free (d);
	return EINVAL;
    }
    
    d->num = num;
    d->ref = 1;

    *dp = d;
    return 0;
}

void
dp_ref (struct dp_part *dp)
{
    if (dp)
	++dp->ref;
}

/*
 * free the dpart structure in `dp'.
 */

void
dp_free (struct dp_part *dp)
{
    if (dp) {
	if (--dp->ref)
	    return;
	if (dp->part)
	    free (dp->part);
	free (dp);
    }
}

/*
 * Iterate over all local partition and returns when one found.
 * `*dp' should one the first call be set to NULL. For each
 * call next dpart is returned. When last entry is found `*dp' is
 * set to NULL. On error != 0 is returned.
 */

int
dp_find (struct dp_part **dp)
{
    u_int32_t num;
    int found = 0, ret;
    struct stat sb;
    
    assert (dp);

    if (*dp == NULL) {
	num = -1;
    } else {	
	num = (*dp)->num;
	dp_free (*dp);
    }
    do { 
	++num;
	
	if (num > 'z' - 'a') {
	    *dp = NULL;
	    break;
	}
	    
	ret = dp_create (num, dp);
	if (ret)
	    return ret;
	
	if (stat ((*dp)->part, &sb) == 0 &&
	    S_ISDIR(sb.st_mode))
	    found = 1;
	
    } while (!found);

    return 0;
}


/*
 *
 */

int
dp_findvol (struct dp_part *d, void (*cb)(void *, int), void *data)
{
    DIR *dir;
    struct dirent *e;
    int fd, ret;

    assert (d);

    ret = chdir (d->part);
    if (ret)
	return 0;

    dir = opendir (d->part);
    if (dir == NULL)
	return 0;

    while ((e = readdir (dir)) != NULL) {
	if (strncmp (e->d_name, "vol", 3))
	    continue;

	fd = open (e->d_name, O_RDWR, 0600);
	if (fd < 0)
	    errx (1, "can't open %s/%s", d->part, e->d_name);

	(cb) (data, fd);
	close (fd);
    }
    closedir (dir);
    return 0;
}
