/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "fsrv_locl.h"

RCSID("$Id: fsprocs.c,v 1.18 1999/07/04 10:27:31 map Exp $");

static int
ropa_getcallback (void *client, const struct AFSFid *fid,
		  struct AFSCallBack *cb)
{
    struct timeval tv;
    assert (client == NULL && fid && cb);

    gettimeofday (&tv, NULL);

    cb->CallBackVersion = 1;
    cb->ExpirationTime  = tv.tv_sec + 3600; /* Grant for one hour */
    cb->CallBackType    = CBSHARED;

    return 0;
}


/*
 *
 */

int
RXAFS_FetchData(struct rx_call *call,
		const struct AFSFid *a_fidToFetchP,
		const int32_t a_offset,
		const int32_t a_lenInBytes,
		struct AFSFetchStatus *a_fidStatP,
		struct AFSCallBack *a_callBackP,
		struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, fd, close_ret;
    int32_t len, net_len;
    
    ret = vld_find_vol (a_fidToFetchP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_fidToFetchP->Vnode, a_fidStatP,
			  O_RDONLY, &fd, NULL);
    if (ret)
	goto bad;
    
    len = min(a_lenInBytes, a_fidStatP->Length - a_offset);
    net_len = ntohl(len);

    if (rx_Write (call, &net_len, sizeof(net_len)) != sizeof(net_len)) {
	ret = errno;
	close_ret = close (fd);
	assert (close_ret == 0);
	goto bad;
    }
    
    ret = copyfd2rx (fd, call, a_offset, len);
    close_ret = close (fd);
    assert (close_ret == 0);

    if (ret == 0)
	ropa_getcallback (NULL, a_fidToFetchP, a_callBackP);

 bad:
    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_FetchACL(struct rx_call *call,
	       const struct AFSFid *a_dirFidP,
	       AFSOpaque *a_ACLP,
	       struct AFSFetchStatus *a_dirNewStatP,
	       struct AFSVolSync *a_volSyncP)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_FetchStatus(struct rx_call *call,
		  const struct AFSFid *a_fidToStatP,
		  struct AFSFetchStatus *a_currStatP,
		  struct AFSCallBack *a_callBackP,
		  struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret;
    
    ret = vld_find_vol (a_fidToStatP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_fidToStatP->Vnode, a_currStatP,
			  O_RDONLY, NULL, NULL);
    
    if (ret == 0)
	ropa_getcallback (NULL, a_fidToStatP, a_callBackP);

    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);
    return ret;
}

/*
 *
 */

int
RXAFS_StoreData(struct rx_call *call,
		const struct AFSFid *a_fidToStoreP,
		const struct AFSStoreStatus *a_fidStatusP,
		const int32_t a_offset,
		const int32_t a_lenInBytes,
		const int32_t a_fileLenInBytes,
		struct AFSFetchStatus *a_fidStatP,
		struct AFSCallBack *a_callBackP,
		struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, fd, close_ret;
    
    if(a_offset + a_lenInBytes > a_fileLenInBytes)
	return EINVAL;

    ret = vld_find_vol (a_fidToStoreP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }

    ret = vld_modify_vnode (volh, a_fidToStoreP->Vnode, a_fidStatusP, 
			    a_fidStatP, O_RDWR, &fd, NULL);

    if (ret == 0) {
	if (a_fileLenInBytes != a_fidStatP->Length) {
	    ftruncate (fd, a_fileLenInBytes);
	    a_fidStatP->Length = a_fileLenInBytes;
	}
	if (a_fileLenInBytes != 0) {
	    ret = copyrx2fd (call, fd, a_offset, a_lenInBytes);
	    vld_update_size_fd (a_fidStatP, fd);
	    vld_vld2volsync (volh, a_volSyncP);
	}
	if (ret == 0)
	    ropa_getcallback (NULL, a_fidToStoreP, a_callBackP); /* XXX */
    
    }
    close_ret = close (fd);
    assert (close_ret == 0);

    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_StoreACL(struct rx_call *call,
	       const struct AFSFid *a_dirFidP,
	       const AFSOpaque *a_ACLToStoreP,
	       struct AFSFetchStatus *a_dirNewStatP,
	       struct AFSVolSync *a_volSyncP)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_StoreStatus(struct rx_call *call,
		  const struct AFSFid *a_fidP,
		  const struct AFSStoreStatus *a_currStatusP,
		  struct AFSFetchStatus *a_srStatusP,
		  struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret;
    
    ret = vld_find_vol (a_fidP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_modify_vnode (volh, a_fidP->Vnode, a_currStatusP, a_srStatusP,
			    O_RDONLY, NULL, NULL);
    
    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);
    return ret;
}

/*
 *
 */

static int
removenode (struct rx_call *call,
	    const struct AFSFid *a_dirFidP,
	    const char *a_name,
	    struct AFSFetchStatus *a_srvStatusP,
	    struct AFSVolSync *a_volSyncP,
	    int dirp)
{    
    struct volume_handle *volh;
    int ret, fd;
    fbuf the_fbuf;
    VenusFid fid;
    
    ret = vld_find_vol (a_dirFidP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_dirFidP->Vnode, a_srvStatusP,
			  O_RDWR, &fd, NULL);
    
    if (ret) {
	vld_free (volh);
	return ret;
    }

    ret = fbuf_create (&the_fbuf, fd, a_srvStatusP->Length, FBUF_WRITE);
    if (ret) {
	close (fd);
	vld_free (volh);
	return ret;
    }

    if (dirp) {
	fbuf dirfbuf;
	int close_ret; 
	int dirfd;
	AFSFetchStatus fs;
	VenusFid parentFid;

	parentFid.Cell = 0;
	parentFid.fid = *a_dirFidP;

	ret = fdir_lookup (&the_fbuf, &parentFid, a_name, &fid);
	if (ret) {
	    fbuf_end (&the_fbuf);
	    vld_free (volh);
	    return 0;
	}
	
	ret = vld_open_vnode (volh, fid.fid.Vnode, &fs, 
			      O_RDONLY, &dirfd, NULL);
	if (ret) {
	    fbuf_end (&the_fbuf);
	    vld_free (volh);
	    return ret;
	}

	ret = fbuf_create (&dirfbuf, dirfd, fs.Length, FBUF_READ);
	if (ret) {
	    close_ret = close (dirfd);
	    assert (close_ret == 0);
	    fbuf_end (&the_fbuf);
	    vld_free (volh);
	    return ret;
	}

	ret = fdir_emptyp (&dirfbuf);
	fbuf_end (&dirfbuf);
	if (!ret) {
	    fbuf_end (&the_fbuf);
	    vld_free (volh);
	    return ENOTEMPTY;
	}
    }

    ret = fdir_remove (&the_fbuf, a_name, &fid.fid);

    vld_update_size_fbuf (a_srvStatusP, &the_fbuf);

    fbuf_end (&the_fbuf);

    if (ret) {
	vld_free (volh);
	return ret;
    }

    ret = vld_remove_node (volh, fid.fid.Vnode);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    if (dirp) {
	ret = vld_adjust_linkcount (volh, a_dirFidP->Vnode, -1);
	if (ret) {
	    vld_free (volh);
	    return ret;
	}
	a_srvStatusP->LinkCount--;
    }
    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_RemoveFile(struct rx_call *call,
		 const struct AFSFid *a_dirFidP,
		 const char *a_name,
		 struct AFSFetchStatus *a_srvStatusP,
		 struct AFSVolSync *a_volSyncP)
{
    return removenode (call, a_dirFidP, a_name, a_srvStatusP, 
		       a_volSyncP, FALSE);
}

/*
 *
 */

int
RXAFS_CreateFile(struct rx_call *call,
		 const struct AFSFid *DirFid,
		 const char *Name,
		 const struct AFSStoreStatus *InStatus,
		 struct AFSFid *OutFid,
		 struct AFSFetchStatus *OutFidStatus,
		 struct AFSFetchStatus *OutDirStatus,
		 struct AFSCallBack *CallBack,
		 struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, fd, close_ret;
    fbuf the_fbuf;
    AFSFid child;
    
    ret = vld_find_vol (DirFid->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, DirFid->Vnode, OutDirStatus,
			  O_RDWR, &fd, NULL);
    
    if (ret) {
	vld_free (volh);
	return ret;
    }

    child.Volume = volh->vol;
    ret = vld_create_file (volh, DirFid, &child, 
			   TYPE_FILE, InStatus, NULL /*XXX*/);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    ret = fbuf_create (&the_fbuf, fd, OutDirStatus->Length, FBUF_WRITE);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    /* XXX check name ! */
    ret = fdir_creat (&the_fbuf, Name, child);

    if (ret == 0)
	ret = vld_update_size_fbuf (OutDirStatus, &the_fbuf);

    fbuf_end (&the_fbuf); /* XXX error ? */

    if (ret) {
	vld_free (volh);
	vld_remove_node (volh, child.Vnode);
	return ret;
    }

    ret = vld_open_vnode (volh, child.Vnode, OutFidStatus,
			  O_RDONLY, NULL, NULL);

    if (ret == 0)
	*OutFid = child;

    if (ret == 0)
	ropa_getcallback (NULL, OutFid, CallBack);

    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_Rename(struct rx_call *call,
	     const struct AFSFid *a_origDirFidP,
	     const char *a_origNameP,
	     const struct AFSFid *a_newDirFidP,
	     const char *a_newNameP,
	     struct AFSFetchStatus *a_origDirStatusP,
	     struct AFSFetchStatus *a_newDirStatusP,
	     struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, origfd, newfd, close_ret;
    fbuf origfbuf;
    fbuf newfbuf;
    fbuf *newfbufP = &newfbuf;
    VenusFid child, origVFid;
    
    origVFid.Cell = 0;
    origVFid.fid = *a_origDirFidP;

    if (a_origDirFidP->Volume != a_newDirFidP->Volume)
	return EXDEV;

    ret = vld_find_vol (a_origDirFidP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_origDirFidP->Vnode, a_origDirStatusP,
			  O_RDWR, &origfd, NULL);
    
    if (ret) {
	vld_free (volh);
	return ret;
    }

    ret = fbuf_create (&origfbuf, origfd, 
		       a_origDirStatusP->Length, FBUF_WRITE);
    if (ret) {
	close_ret = close (origfd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    ret = fdir_lookup(&origfbuf, &origVFid, a_origNameP, &child);
    if (ret) {
	fbuf_end (&origfbuf);
	vld_free (volh);
	return ret;
    }

    if (a_origDirFidP->Vnode == a_newDirFidP->Vnode &&
	a_origDirFidP->Unique == a_newDirFidP->Unique) {

	newfbufP = &origfbuf;
	*a_newDirStatusP =  *a_origDirStatusP;
    } else {

	ret = vld_open_vnode (volh, a_newDirFidP->Vnode, a_newDirStatusP,
			  O_RDWR, &newfd, NULL);
	
	if (ret) {
	    vld_free (volh);
	    fbuf_end (&origfbuf);
	    return ret;
	}

	ret = fbuf_create (&newfbuf, newfd, 
			   a_newDirStatusP->Length, FBUF_WRITE);
	if (ret) {
	    fbuf_end (&origfbuf);
	    close_ret = close (newfd);
	    assert (close_ret == 0);
	    vld_free (volh);
	    return ret;
	}
    }

    ret = fdir_creat (newfbufP, a_newNameP, child.fid);
    if (ret) 
	goto out1;

    ret = fdir_remove (&origfbuf, a_origNameP, NULL);
    if (ret)
	fdir_remove (newfbufP, a_newNameP, NULL);

    vld_update_size_fbuf (a_origDirStatusP, &origfbuf);
    vld_update_size_fbuf (a_newDirStatusP, newfbufP);

 out1:
    fbuf_end (&origfbuf);
    if (newfbufP == &newfbuf)
	fbuf_end (&newfbuf);

    if (ret) {
	vld_free (volh);
	return ret;
    }

    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_Symlink(struct rx_call *call,
	      const struct AFSFid *a_dirFidP,
	      const char *a_nameP,
	      const char *a_linkContentsP,
	      const struct AFSStoreStatus *a_origDirStatP,
	      struct AFSFid *a_newFidP,
	      struct AFSFetchStatus *a_newFidStatP,
	      struct AFSFetchStatus *a_newDirStatP,
	      struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, fd, close_ret;
    fbuf the_fbuf;
    AFSFid child;
    VenusFid a_vdirFid, a_vfid;
    AFSStoreStatus ss;

    memset (&ss, 0, sizeof(ss));

    a_vdirFid.Cell = 0;
    a_vdirFid.fid = *a_dirFidP;
    
    ret = vld_find_vol (a_dirFidP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_dirFidP->Vnode, a_newDirStatP,
			  O_RDWR, &fd, NULL);
    
    if (ret) {
	vld_free (volh);
	return ret;
    }

    ret = fbuf_create (&the_fbuf, fd, a_newDirStatP->Length, FBUF_WRITE);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    ret = fdir_lookup (&the_fbuf, &a_vdirFid, a_nameP, &a_vfid);
    if (ret != ENOENT) {
	fbuf_end (&the_fbuf); /* XXX error ? */
	vld_free (volh);
	if (ret == 0)
	    return EEXIST;
	else
	    return ret;
    }

    child.Volume = volh->vol;
    ret = vld_create_file (volh, a_dirFidP, &child, 
			   TYPE_LINK, &ss, NULL /*XXX*/);
    if (ret) {
	fbuf_end (&the_fbuf); /* XXX error ? */
	vld_free (volh);
	return ret;
    }

    /* XXX check name ! */
    ret = fdir_creat (&the_fbuf, a_nameP, child);

    if (ret == 0)
	vld_update_size_fbuf (a_newDirStatP, &the_fbuf);

    fbuf_end (&the_fbuf); /* XXX error ? */

    if (ret) {
	vld_free (volh);
	vld_remove_node (volh, child.Vnode);
	return ret;
    }

    ret = vld_open_vnode (volh, child.Vnode, a_newFidStatP,
			  O_RDWR, &fd, NULL);

    if (ret == 0) {
	int len = strlen (a_linkContentsP);
	ret = write (fd, a_linkContentsP, len);
	if (ret != len)
	    ret = errno;
	else
	    ret = 0;

	close_ret = close (fd);
	assert (close_ret == 0);

	if (ret == 0) {
	    *a_newFidP = child;
	    vld_update_size_fd(a_newFidStatP, fd);
	}
    }

    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    return ret;
}

/*
 *
 */

int
RXAFS_Link(struct rx_call *call,
	   const struct AFSFid *a_dirFidP,
	   const char *a_nameP,
	   const struct AFSFid *a_existingFidP,
	   struct AFSFetchStatus *a_newFidStatP,
	   struct AFSFetchStatus *a_newDirStatP,
	   struct AFSVolSync *a_volSyncP)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_MakeDir(struct rx_call *call,
	      const struct AFSFid *a_parentDirFidP,
	      const char *a_newDirNameP,
	      const struct AFSStoreStatus *a_currStatP,
	      struct AFSFid *a_newDirFidP,
	      struct AFSFetchStatus *a_dirFidStatP,
	      struct AFSFetchStatus *a_parentDirStatP,
	      struct AFSCallBack *a_newDirCallBackP,
	      struct AFSVolSync *a_volSyncP)
{
    struct volume_handle *volh;
    int ret, fd, close_ret;
    fbuf the_fbuf;
    AFSFid child;
    
    ret = vld_find_vol (a_parentDirFidP->Volume, &volh);
    if (ret)
	return ret;

    ret = vld_db_uptodate (volh);
    if (ret) {
	vld_free (volh);
	return ret;
    }
    
    ret = vld_open_vnode (volh, a_parentDirFidP->Vnode, a_parentDirStatP,
			  O_RDWR, &fd, NULL);
    
    if (ret) {
	vld_free (volh);
	return ret;
    }

    child.Volume = volh->vol;
    ret = vld_create_directory (volh, a_parentDirFidP, &child,
				a_currStatP, NULL);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    ret = vld_adjust_linkcount (volh, a_parentDirFidP->Vnode, 1);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }
    a_parentDirStatP->LinkCount++;

    ret = fbuf_create (&the_fbuf, fd, a_parentDirStatP->Length, FBUF_WRITE);
    if (ret) {
	close_ret = close (fd);
	assert (close_ret == 0);
	vld_free (volh);
	return ret;
    }

    /* XXX check name ! */
    ret = fdir_creat (&the_fbuf, a_newDirNameP, child);

    vld_update_size_fbuf (a_parentDirStatP, &the_fbuf);

    fbuf_end (&the_fbuf); /* XXX error ? */

    if (ret) {
	vld_free (volh);
	/* XXX remove new inode */
	abort();
	return ret;
    }

    ret = vld_open_vnode (volh, child.Vnode, a_dirFidStatP,
			  O_RDONLY, NULL, NULL);

    if (ret == 0)
	*a_newDirFidP = child;

    vld_vld2volsync (volh, a_volSyncP);
    vld_free (volh);

    ropa_getcallback (NULL, a_newDirFidP, a_newDirCallBackP);

    return ret;
}

/*
 *
 */

int
RXAFS_RemoveDir(struct rx_call *call,
		const struct AFSFid *a_parentDirStatP,
		const char *a_dirNameP,
		struct AFSFetchStatus *a_newParentDirStatP,
		struct AFSVolSync *a_volSyncP)
{
    return removenode (call, a_parentDirStatP, a_dirNameP, 
		       a_newParentDirStatP, a_volSyncP, TRUE);
}

/*
 *
 */

int
RXAFS_GiveUpCallBacks(struct rx_call *call,
		      const AFSCBFids *a_fidArrayP,
		      const AFSCBs *a_callBackArrayP)
{
    return 0;
}

/*
 *
 */

int
RXAFS_GetVolumeStatus(struct rx_call *call,
		      const int32_t a_volIDP,
		      struct AFSFetchVolumeStatus *a_volFetchStatP,
		      char *a_volNameP,
		      char *a_offLineMsgP,
		      char *a_motdP)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_SetVolumeStatus(struct rx_call *call,
		      const int32_t a_volIDP,
		      const struct AFSStoreVolumeStatus *a_volStoreStatP,
		      const char *a_volNameP,
		      const char *a_offLineMsgP,
		      const char *a_motdP)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_GetRootVolume(struct rx_call *call,
		    char *a_rootVolNameP)
{
    strncpy (a_rootVolNameP, "root.cell", AFSNAMEMAX);
    a_rootVolNameP[AFSNAMEMAX-1] = '\0';
    return 0;
}

/*
 * Get time, the poor mans ntp, used as probe by some clients
 */

int
RXAFS_GetTime(struct rx_call *call,
	      u_int32_t *a_secondsP,
	      u_int32_t *a_uSecondsP)
{
    struct timeval tv;

    gettimeofday (&tv, NULL);

    *a_secondsP = tv.tv_sec;
    *a_uSecondsP = tv.tv_usec;

    return 0;
}

/*
 *
 */

int
RXAFS_NGetVolumeInfo(struct rx_call *call,
		     const char *VolumeName,
		     struct AFSVolumeInfo *stuff)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_BulkStatus(struct rx_call *call,
		 const AFSCBFids *FidsArray,
		 AFSBulkStats *StatArray,
		 AFSCBs *CBArray,
		 struct AFSVolSync *Sync)
{
    struct volume_handle *volh = NULL;
    int ret, i = 0;
    int32_t oldvolume = -1;
    
    if (FidsArray->len == 0)
	return EINVAL;

    for (i = 0; FidsArray->len > i ; i++) {

	if (FidsArray->val[i].Volume != oldvolume) {
	    if (volh)
		vld_free (volh);

	    ret = vld_find_vol (FidsArray->val[i].Volume, &volh);
	    if (ret)
		return ret;
	    
	    ret = vld_db_uptodate (volh);
	    if (ret) {
		vld_free (volh);
		return ret;
	    }
	}
	
	ret = vld_open_vnode (volh, FidsArray->val[i].Vnode, 
			      &StatArray->val[i], O_RDONLY, NULL, NULL);
	
	if (ret) {
	    vld_free (volh);
	    return ret;
	}
	ropa_getcallback (NULL, &FidsArray->val[i], &CBArray->val[i]);
    }
    vld_free (volh);

    return 0;
}

/*
 *
 */

int
RXAFS_SetLock(struct rx_call *call,
	      const struct AFSFid *Fid,
	      const ViceLockType Type,
	      struct AFSVolSync *Sync)
{
    return EPERM;
}

/*
 *
 */

int
RXAFS_ExtendLock(struct rx_call *call,
		 const struct AFSFid *Fid,
		 struct AFSVolSync *Sync)
{
    return EPERM;
}


/*
 *
 */

int
RXAFS_ReleaseLock(struct rx_call *call,
		      const struct AFSFid *Fid,
		      struct AFSVolSync *Sync)
{
    return EPERM;
}
