/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>

RCSID("$Id: bosserver.c,v 1.6 1999/06/09 22:43:00 map Exp $");

#include <sys/types.h>
#include <sys/wait.h>

#include <rx/rx.h>
#include <rx/rx_null.h>

#include <ports.h>
#include <ko.h>

#ifdef KERBEROS
#include <des.h>
#include <krb.h>
#include <rxkad.h>
#endif

#include <netinit.h>
#include <msecurity.h>

#include <service.h>

#include <roken.h>
#include <err.h>
#include <getarg.h>

#include <bos.h>
#include <bos.ss.h>

typedef struct mtype {
    char *name;
    char *path;
    int state;
    pid_t pid;
} mtype;

struct mtype servers[] = {
    { "fs", "fileserver", 0, 0},
    { "vldb", "vldbserver", 0, 0},
    { NULL, NULL, 0}
} ;

static int debug = 1;
static char *bosserverprefix = NULL;

/*
 *
 */

static void
bosdebug (char *fmt, ...)
    __attribute__ ((format (printf, 1, 2)));

static void
bosdebug (char *fmt, ...)
{
    va_list args;
    if (!debug)
	return ;

    va_start (args, fmt);
    vfprintf (stderr, fmt, args);
    va_end(args);
}

/*
 *
 */

int
BOZO_CreateBnode(struct rx_call *call,
		 const char *type,
		 const char *instance,
		 const char *p1,
		 const char *p2,
		 const char *p3,
		 const char *p4,
		 const char *p5,
		 const char *p6)
{
    bosdebug ("BOZO_CreateNode: %s %s\n", type, instance);

    if (!sec_is_superuser(call))
	return VL_PERM;

    return 0;
}

/*
 *
 */

int
BOZO_DeleteBnode(struct rx_call *call,
		 const char *instance)
{
    bosdebug ("BOZO_DeleteBnode: %s\n", instance);

    return 0;
}


/*
 *
 */

int
BOZO_GetStatus(struct rx_call *call,
	       const char *instance,
	       int32_t *inStat,
	       char *statdescr)
{
    bosdebug ("BOZO_GetStatus: %s\n", instance);

    strcpy (statdescr, "foo");
    *inStat = 0;
    return 0;
}

/*
 *
 */

int
BOZO_EnumerateInstance(struct rx_call *call,
		       const int32_t instance,
		       char *iname)
{
    bosdebug ("BOZO_EnumerateInstance: %d\n", instance);

    return -1;
}

/*
 *
 */

int
BOZO_GetInstanceInfo(struct rx_call *call,
		     const char *instance,
		     char *type,
		     struct bozo_status *status)
{
    bosdebug ("BOZO_GetInstanceInfo: %s\n", instance);

    strcpy (type, "simple");
    memset (status, 0, sizeof(*status));
    return 0;
}


/*
 *
 */

int
BOZO_GetInstanceParm(struct rx_call *call,
		     const char *instance,
		     const int32_t num,
		     char *param)
{
    bosdebug ("BOZO_GetInstanceParm: %s %d\n", instance, num);

    strcpy (param, "foo");
    return 0;
}


/*
 * start the server `server'
 */

static int
start_server (struct mtype *server)
{
    int i;

    bosdebug ("starting server %s (%s)\n", server->name, 
	      server->path);
    
    i = fork();
    switch (i) {
    case 0: {
	char path[MAXPATHLEN];
	char *newargv[2];

	newargv[0] = server->name;
	newargv[1] = NULL;
	
	if (debug == 0) {
	    close (0);
	    close (1);
	    close (2);
	}

	snprintf (path, sizeof(path), "%s/%s", bosserverprefix, server->path);
	
	execvp (path, newargv);

	bosdebug ("server %s failed with %s (%d)\n", path,
		      strerror (errno), errno);
	exit (1);
	break;
    }
    case -1:
	bosdebug ("failed to fork with server %s with reson %s (%d)\n",
		  server->name, strerror (errno), errno);
	return errno;
	break;
    default:
	server->pid = i;
	bosdebug ("started server %s with pid %d\n", 
		  server->name, server->pid);
	break;
    }
    
    return 0;
}


/*
 * GC processes, and start new if appropriate.
 */

static int dead_children = 0;

PROCESS riperprocess;

void
theriper (char *no_arg)
{
    int status;
    struct mtype *server = &servers[0];
    pid_t pid;

    while (1) {
	if (dead_children == 0) {
	    bosdebug ("theriper: sleeping\n");
	    LWP_WaitProcess ((char *)theriper);
	}	

	bosdebug ("theriper: running\n");

	if (dead_children == 0) {
	    bosdebug ("theriper: called without a dead child\n");
	    continue;
	}
	
	if (dead_children < 0)
	    abort();

	dead_children--;
	    
	pid = wait (&status);

	if (WIFEXITED(status)) {
	    bosdebug ("normal exit: exit error %d", WEXITSTATUS(status));
	} else if (WIFSIGNALED(status)) {
	    bosdebug ("terminated by signal num %d", WTERMSIG(status));
	    if (WCOREDUMP(status))
		bosdebug (" coredumped\n");
	} else if (WIFSTOPPED(status)) {
	    bosdebug ("process stoped by signal %d", WSTOPSIG(status));
	} else {
	    bosdebug ("by unknown reason");
	}
		      
	while (server->name) {
	    if (server->pid == pid) {
		bosdebug ("%s was the killed, starting new", server->name);
		start_server (server);
		break;
	    }
	    server++;
	}
    }
}

/*
 * Signal-handler for SIGCHLD
 */

static void
sigchild (int foo)
{
    bosdebug ("child died\n");

    dead_children++;

    LWP_NoYieldSignal ((char *) theriper);
}

/*
 * bootstrap bosserver
 * Create riper thread and start all servers.
 */

static int
bosserver_init (void)
{
    struct mtype *server = &servers[0];
    int i;

    i = LWP_CreateProcess (theriper, 0, 1, NULL, "the-riper", &riperprocess);
    if (i)
	errx (1, "LWP_CreateProcess returned %d", i);

    while (server->name) {
	i = start_server(server);
	if (i)
	    bosdebug ("server_server: returned %s (%d)", strerror (i), i);
	server++;
    }
    
    return 0;
}


/*
 *
 */

static struct rx_service *bosservice = NULL;

static char *cell = NULL;
static char *realm = NULL;
static int do_help = 0;
static char *srvtab_file = NULL;

static struct getargs args[] = {
    {"cell",	0, arg_string,    &cell, "what cell to use", NULL},
    {"realm",  0, arg_string,    &realm, "what realm to use"},
    {"prefix",'p', arg_string,    &bosserverprefix, 
     "directory where servers is stored", NULL},
    {"debug", 'd', arg_flag,      &debug, "output debugging"},
    {"help",  'h', arg_flag,      &do_help, "help"},
    {"srvtab",'s', arg_string,    &srvtab_file, "what srvtab to use"},
    { NULL, 0, arg_end, NULL }
};

static void
usage(int exit_code)
{
    arg_printusage(args, NULL, "", ARG_AFSSTYLE);
    exit (exit_code);
}

/*
 *
 */

int
main (int argc, char **argv)
{
    int ret;
    int optind = 0;
    
    if (getarg (args, argc, argv, &optind, ARG_AFSSTYLE)) {
	usage (0);
    }

    argc -= optind;
    argv += optind;

    if (argc) {
	printf("unknown option %s\n", *argv);
	return 0;
    }
    
    if (do_help)
	usage(1);

    if (bosserverprefix == NULL)
	bosserverprefix = MILKO_LIBEXECDIR;

    ports_init();
    cell_init(0);

    if (cell)
	cell_setthiscell (cell);

    network_kerberos_init (srvtab_file);

    signal (SIGCHLD, sigchild);

    ret = network_init(htons(afsbosport), "bos", BOS_SERVICE_ID, 
		       BOZO_ExecuteRequest, &bosservice, realm);
    if (ret)
	errx (1, "network_init failed with %d", ret);
    
    ret = bosserver_init();
    if (ret)
	errx (1, "bosserver_init: error %d\n", ret);

    printf("Milko bosserver %s-%s started\n", PACKAGE, VERSION);

    rx_SetMaxProcs(bosservice,5) ;
    rx_StartServer(5) ;

    abort() ; /* should not get here */

    return 0;
}

