/*
 * Copyright (c) 1995, 1996, 1997, 1998, 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "arla_local.h"
RCSID("$Id: aix-subr.c,v 1.6 1999/07/24 19:48:30 lha Exp $");

static long blocksize = 1024;

struct args {
     int fd;
     off_t off;
     char *buf;
     char *ptr;
     struct dirent *last;
};

static void
flushbuf (struct args *args)
{
     unsigned inc = blocksize - (args->ptr - args->buf);

     args->last->d_reclen += inc;
     args->last->d_offset += inc;
     if (write (args->fd, args->buf, blocksize) != blocksize)
	  arla_warn (ADEBWARN, errno, "write");
     args->ptr = args->buf;
     args->last = NULL;
}

#undef DIRSIZ
#define DIRSIZ(name) \
        (((sizeof(struct dirent)+ (strlen(name)+1)) + \
	  3) & ~3)

static void
write_dirent(VenusFid *fid, const char *name, void *arg)
{
     struct dirent *real;
     struct args *args = (struct args *)arg;
     u_short reclen;

     reclen = DIRSIZ(name);

     if (args->ptr + reclen > args->buf + blocksize)
	  flushbuf (args);
     real = (struct dirent *)args->ptr;

     real->d_reclen = reclen;
     real->d_ino = afsfid2inode (fid);
     strcpy (real->d_name, name);
     args->ptr += real->d_reclen;
     args->off += real->d_reclen;
     real->d_offset = args->off;
     args->last = real;
}

Result
conv_dir (FCacheEntry *e, CredCacheEntry *ce, u_int tokens,
	  xfs_cache_handle *cache_handle,
	  char *cache_name, size_t cache_name_sz)
{
     struct args args;
     Result res;
     fbuf the_fbuf;
     int ret;

     e->flags.extradirp = TRUE;
     fcache_extra_file_name (e, cache_name, cache_name_sz);
     res.tokens = e->tokens |= XFS_DATA_R | XFS_OPEN_NR;

     args.fd = open (cache_name,
		     O_WRONLY | O_CREAT | O_TRUNC | O_BINARY, 0666);
     if (args.fd == -1) {
	  arla_warn (ADEBWARN, errno, "open %s", cache_name);
	  res.res = -1;
	  res.error = errno;
	  return res;
     }
     fcache_fhget (cache_name, cache_handle);

     args.off  = 0;
     args.buf  = (char *)malloc (blocksize);
     if (args.buf == NULL) {
	 arla_warn (ADEBWARN, errno, "malloc %lu", blocksize);
	 res.res = -1;
	 res.error = errno;
	 close (args.fd);
	 return res;
     }
     ret = fcache_get_fbuf (e, &the_fbuf, O_RDONLY, FBUF_READ);
     if (ret) {
	 res.res = -1;
	 res.error = ret;
	 close (args.fd);
	 free (args.buf);
	 return res;
     }
     
     args.ptr  = args.buf;
     args.last = NULL;
     
     fdir_readdir (&the_fbuf, write_dirent, (void *)&args, &e->fid);

     fbuf_end (&the_fbuf);

     if (args.last)
	  flushbuf (&args);
     free (args.buf);
     if ((res.res = close (args.fd)) != 0)
	  res.error = errno;
     return res;
}

/*
 * remove `filename` from the converted directory for `e'
 */

int
dir_remove_name (FCacheEntry *e, const char *filename,
		 xfs_cache_handle *cache_handle,
		 char *cache_name, size_t cache_name_sz)
{
    int ret;
    int fd;
    fbuf fb;
    struct stat sb;
    char *buf;
    char *p;
    size_t len;
    struct dirent *dp;
    struct dirent *last_dp;

    fcache_extra_file_name (e, cache_name, cache_name_sz);
    fd = open (cache_name, O_RDWR, 0);
    if (fd < 0)
	return errno;
    fcache_fhget (cache_name, cache_handle);
    if (fstat (fd, &sb) < 0) {
	ret = errno;
	close (fd);
	return ret;
    }
    len = sb.st_size;

    ret = fbuf_create (&fb, fd, len, FBUF_WRITE);
    if (ret)
	return ret;
    last_dp = NULL;
    for (p = buf = fbuf_buf (&fb); p < buf + len; p += dp->d_reclen) {
	dp = (struct dirent *)p;

	if (strcmp (filename, dp->d_name) == 0) {
	    if (last_dp != NULL)
		last_dp->d_reclen += dp->d_reclen;
	    dp->d_ino = 0;
	    fbuf_end (&fb);
	    return 0;
	}
	last_dp = dp;
    }
    fbuf_end (&fb);
    return ENOENT;
}
