/*
 * Copyright (c) 1999 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* 
   klog.c - a version of 'klog' for Arla

   Written by Chris Wing - wingc@engin.umich.edu
   based on examples of AFS code: pts.c (Arla) and kauth.c (KTH-KRB)

   This is a reimplementation of klog from AFS. The following new features
   have been added:
	-timeout	Number of seconds to wait for Kerberos operations
			to finish. By default, we will wait forever.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

RCSID("$Id: klog.c,v 1.6 1999/08/03 23:26:23 assar Exp $");

#include "appl_locl.h"
#include "klog.h"
#include "kafs.h"

/*
 * State variables
 */

/* Actually get a Kerberos ticket? i.e. -tmp */
static int get_tkt = 0;

/* Run in 'pipe mode' */
static int pipe_mode = 0;

/* Close stderr */
static int silent_mode = 0;

/* Get a new PAG and spawn a shell? */
static int set_pag = 0;

/* Number of seconds to wait for Kerberos commands to complete */
static int timeout = 0;

/* Lifetime in seconds. Default token lifetime is 720 hours (30 days) */
static int lifetime = 720 * 60 * 60;

/* AFS db servers */
static char **dbservers = NULL;
static int numservers = 0;

/* Kerberos ticket file */
static char *tkfile = NULL;

/* Kerberos principal, AFS cell, Kerberos realm */
static char *principal = NULL;
static char *cell = NULL;
static char *realm = NULL;

/* AFS ID */
static int afsid = 0;

/* Password */
static char *password = NULL;

/* Did the attempted network command time out? */
static int timed_out = 0;

/* Do we have a ticket file lying around we'd like to get rid of? */
static int have_useless_ticket_file = 0;

/* Did we get a token that will sit around unused if not destroyed? */
static int have_useless_token = 0;

/*
 * Various helper functions that we call
 */

/* Print out a help message and exit */

int do_help(void)
{
	printf("Usage: klog [-principal <username>]\n"
	       "            [-password <AFS password>]\n"
	       "            [-cell <cell name>]\n"
	       "            [-servers <list of servers to contact>]\n"
	       "            [-lifetime <token lifetime in hh[:mm[:ss]]>]\n"
	       "            [-timeout <network timeout in seconds>]\n"
	       "\n"
	       "            principal@cell\n"
	       "         or principal\n"
	       "         or @cell>\n"
	       "\n"
	       "Options:    -pipe (accept password from stdin, close stdout)\n"
	       "            -silent (close stderr)\n"
	       "            -setpag (spawn a new shell with new PAG "
	                    "and ticket file)\n"
	       "            -tmp (get a Kerberos 4 TGT)\n"
	       "            -help (print this message)\n");
	exit(1);
}

/*
 * Erase passwords from memory and remove tokens and tickets that
 * shouldn't be left laying around.
 */

static void
final_cleanup (void)
{
    /* Make sure to clear out the password in this process's memory image
       before exiting */
    if(password)
	memset(password, 0, PASSWD_MAX);

    /* Destroy any useless tokens that we acquired */
    if(have_useless_token)
	k_unlog();

    /* Destroy any temporary ticket files lying around. */
    if(have_useless_ticket_file)
	dest_tkt();
}

/* Death function that erases password before exiting */

void
die (int retcode)
{
    final_cleanup ();

    exit(retcode);
}

/* Die, print out an error message, and interpret errno
   Remember that *err*() from roken prints out a trailing newline. */

void
diet (int retcode, char *fmt, ...)
{
    va_list ap;

    final_cleanup ();

    va_start(ap, fmt);
    verr (retcode, fmt, ap);
    va_end(ap);
}

/* Die and print out an error message. Do not interpret errno. */

void
dietx (int retcode, char *fmt, ...)
{
    va_list ap;

    final_cleanup ();

    va_start(ap, fmt);
    verrx (retcode, fmt, ap);
    va_end(ap);
}


/* 
 * Figure out the AFS ID of a user name 
 */

int get_afs_id(void)
{
    int i=0;
    int connected = 0;
    int32_t returned_id;
    int32_t res;

/* FIXME: Should we use authorization when connecting to the dbserver?
	  noauth=0 sometimes gives warnings, e.g. if the realm name is
	  not the same as the name of the cell...
*/
    int noauth = 1;

    struct rx_connection *connptdb = NULL;
    namelist nlist;
    idlist ilist;

    prname pr_name_buf;

    /* set up necessary crap to use PR_NameToID */
    nlist.len = 1;
    nlist.val = &pr_name_buf;

    ilist.len = 1;
    ilist.val = &returned_id;

    strncpy(nlist.val[0], principal, sizeof(prname));

    /* null terminate the string just in case */
    pr_name_buf[sizeof(prname) - 1] = '\0';

    /* try all known servers :) */
    while(i < numservers) {
	connptdb = arlalib_getconnbyname(cell, dbservers[i], afsprport, 
					 PR_SERVICE_ID,
					 arlalib_getauthflag (noauth, 0,
					 0, 0));
	if (connptdb == NULL) {
	    warnx("Could not connect to dbserver %s",dbservers[i]);
	} else {
	    connected = TRUE;
	    break;
	}
	i++;
    }

    if(connected == 0) {
	warnx("Could not connect to any dbserver");
	return(-1);
    }

    res = PR_NameToID(connptdb, &nlist, &ilist);

    arlalib_destroyconn(connptdb);

    if(res)
	return(-1);
    else
	return(ilist.val[0]);
}

#ifndef KRB_TICKET_GRANTING_TICKET
#define KRB_TICKET_GRANTING_TICKET "krbtgt"
#endif

/* Get a Kerberos 4 TGT */

int get_k4_ticket(void)
{
	int rc;

	int krb_lifetime;

	/* The minimum lifetime is 5 minutes */
	if(lifetime < 5 * 60)
	    krb_lifetime = 1;
	else
	    krb_lifetime = krb_time_to_life(0, lifetime);

	/* If a ridiculously high lifetime is given to krb_time_to_life,
	   0 may be returned as a result... */
	if( (krb_lifetime > 255) || (krb_lifetime == 0) )
	    krb_lifetime = 255;

	rc = krb_get_pw_in_tkt(principal, "", realm, 
			       KRB_TICKET_GRANTING_TICKET, realm,
			       krb_lifetime, password);

	if(rc)
	    warnx("Unable to authenticate to Kerberos: %s",
		  krb_get_err_text(rc));

	return(rc);
}


/* Get an AFS token */

int get_afs_token(void)
{
	int rc;

	/* FIXME: This will happily store a token for one cell with the
		  name of another cell, and this makes Arla
		  misbehave :( */

	rc = krb_afslog_uid(cell, realm, afsid);

	if(rc)
	    warnx("Unable to get an AFS token: %s", krb_get_err_text(rc));

	return(rc);
}


/* Generalized machinery for performing a timeout on an arbitrary function
   returning an integer. Fun, eh?
*/

int do_timeout( int (*function)(void) )
{
    int pipearr[2];
    int reader, writer;
    pid_t pid;
    
    timed_out = 0;
    
    /* Don't bother with all this jibba jabba if we don't want to timeout */
    if(timeout == 0) {
	return(function());
    }

    if(pipe(pipearr) == -1)
	diet (1, "do_timeout(): Can't make a pipe");

    reader = pipearr[0];
    writer = pipearr[1];

    /* :) */
    fflush(stdout);
    fflush(stderr);

    pid = fork();

    if(pid == -1)
	diet (1, "do_timeout: Can't fork");

    if(pid) {
	/* parent's thread of execution */

	fd_set readfds;
	struct timeval tv;
	int retval;
	int result;
	ssize_t len;

	close(writer);

	/* this is how you set up a select call */
	FD_ZERO(&readfds);
	FD_SET(reader, &readfds);

	/* Wait as many as timeout seconds. */
	tv.tv_sec = timeout;
	tv.tv_usec = 0;

	retval = select(reader+1, &readfds, NULL, NULL, &tv);

	/* Kill the child process in any case */
	kill(pid, SIGKILL);

	if(!retval) {
	    timed_out = 1;
	} else {
	    /* okay, what happened */

	    len = read(reader, &result, sizeof(int));
	}

	/* no matter what, we must wait on the child or else we will
	   accumulate zombies */
	wait4(pid, NULL, 0, NULL);

	/* close the other end of the pipe */
	close(reader);

	if(timed_out)
	    return(1);
	else
	    return(result);

	/* PARENT ENDS HERE */

    } else {
	/* child's thread of execution */
	int retval;
	ssize_t len;

	close(reader);

	retval = function();

	len = write(writer, &retval, sizeof(int));

	/* Destroy the copy of the password in the child's memory image */
	memset(password, 0, PASSWD_MAX);

	exit(0);

	/* CHILD ENDS HERE */
    }
}

/*
 * randfilename()
 * Return a "random" file name, for use, e.g., as a ticket file
 * use umich compat basename of ticket.
 */

static char *tktbasename[] = {
    KLOG_TKT_ROOT,
    TKT_ROOT,
    NULL
};

char *
randfilename(void)
{
    char filename[MAXPATHLEN], **basepath;
    int fd;

    basepath = tktbasename;
    while (*basepath) {
	snprintf(filename, sizeof(filename), 
		 "%stkt_%d_XXXXXX", *basepath, getuid());
	
	fd = mkstemp(filename);
	if (fd >= 0)
	    break;

	basepath++;
    }
    if (*basepath == NULL)
	dietx (1, "could not create ticket file");

    close (fd);
    
    return estrdup(filename);
}


/* Is the string an option? */

int flagopt(const char *string)
{
	if(! strcmp(string, "-help"))
	    return TRUE;

	if(! strcmp(string, "-tmp"))
	    return TRUE;

	if(! strcmp(string, "-pipe"))
	    return TRUE;

	if(! strcmp(string, "-silent"))
	    return TRUE;

	if(! strcmp(string, "-setpag"))
	    return TRUE;

	if(! strcmp(string, "-principal"))
	    return TRUE;

	if(! strcmp(string, "-cell"))
	    return TRUE;

	if(! strcmp(string, "-realm"))
	    return TRUE;

	if(! strcmp(string, "-lifetime"))
	    return TRUE;

	if(! strcmp(string, "-timeout"))
	    return TRUE;

	if(! strcmp(string, "-password"))
	    return TRUE;

	if(! strcmp(string, "-servers"))
	    return TRUE;

	return FALSE;
}



/*
 * The core of this evil
 */

int
main(int argc, char **argv)
{
    int i,j;
    krb_principal princ;
    int firstserver;
    char prompt[PW_PROMPT_MAX];
    int rc;

    char *onepointer;

    char pwbuf[PASSWD_MAX];

    /* for Rx - Arlalib stuff */
    ports_init();
    cell_init(0); /* XXX */

    memset(&princ, 0, sizeof(princ));
    
    /* Parse command line options */
    for (i = 1; i < argc; i++) {
	if (! strcmp(argv[i], "-help")) {
	    do_help();
	} else if (! strcmp(argv[i], "-tmp")) {
	    get_tkt = TRUE;
	} else if (! strcmp(argv[i], "-pipe")) {
	    pipe_mode = TRUE;
	} else if (! strcmp(argv[i], "-silent")) {
	    silent_mode = TRUE;
	} else if (! strcmp(argv[i], "-setpag")) {
	    set_pag = TRUE;
	} else if (! strcmp(argv[i], "-principal")) {
	    i++;
	    if(i < argc) {
		principal = argv[i];
	    } else
 		dietx (1, "Must specify user name with -principal option");
	} else if (! strcmp(argv[i], "-realm")) {
	    i++;
	    if(i<argc) {
		realm = argv[i];
	    } else 
		dietx (1, "Must specify realm name with -realm option");

	} else if (! strcmp(argv[i], "-cell")) {
	    i++;
	    if(i<argc) {
		cell = argv[i];
	    } else
		dietx (1, "Must specify cell name with -cell option");
	} else if (! strcmp(argv[i], "-lifetime")) {
	    int h=0,m=0,s=0;
	    int matched;

	    i++;
	    if(i<argc) {
		matched = sscanf(argv[i], "%u:%u:%u", &h, &m, &s);

		if( (matched != 1) && (matched != 2) && (matched != 3) )
		    dietx (1, "Bad argument for -lifetime: %s", argv[i]);

		lifetime = h * 3600 + m * 60 + s;

	    } else 
		dietx (1, "Must specify hh[:mm[:ss]] for -lifetime option");
	} else if (! strcmp(argv[i], "-timeout")) {
	    i++;
	    if(i<argc) {
		if(sscanf(argv[i],"%u",&timeout) == 0)
		    dietx (1, "Bad option for -timeout: %s", argv[i]);

	    } else
		dietx (1, "Must specify number of seconds with -timeout");

	} else if (! strcmp(argv[i], "-password")) {
	    warnx ("WARNING: The use of -password is STRONGLY DISCOURAGED");

	    i++;
	    if(i<argc) {
		password = argv[i];
	    } else
		dietx (1, "Must specify password with -password option");
	} else if (! strcmp(argv[i], "-servers")) {
	    i++;
	    j=0;

	    firstserver=i;
	    while(i<argc) {
		if (flagopt(argv[i])) {
		    i--;	/* break out of parsing loop */
		    break;
		}
		i++;
		j++;
	    }
	    numservers=j;
	} else {
	    /* Not otherwise recognized argument assumed to be user@cell */
	    kname_parse (princ.name,
			 princ.instance,
			 princ.realm,
			 argv[i]);
	}
    }

    /* Simplest way to prevent any output from this command */
    if(pipe_mode)
	freopen("/dev/null", "r+", stdout);

    /* Simplest way to prevent any output from this command */
    if(silent_mode)
	freopen("/dev/null", "r+", stderr);

    if(!k_hasafs())
	dietx (1, "Hmm, your machine doesn't seem to have kernel support "
		  "for AFS");

    /* Try to get a new PAG, but don't abort if we can't */
    if(set_pag) {
	if (k_setpag() == -1)
	    warnx ("Couldn't get new PAG");
    }

    /* Figure out the AFS cell to use */
    if (cell == NULL) {
	if(princ.realm[0] == '\0') {
	    /* Get current cell if we weren't given one already */
	    cell = (char *)cell_getthiscell();

	    if(cell == NULL) {
		dietx (1, "Can't find local cell!");
	    }
	} else {
	    /* Use cell unparsed from last argument */
	    cell = princ.realm;
	}
    }

    /* FIXME: Figure out a way to automatically deal with setups where the
	      Kerberos realm is not just an uppercase version of the
	      AFS cell

	      if libkafs exported kafs_realm_of_cell(), we could use that */

    if(!realm) {
	char *p;

	realm = estrdup(cell);

	/* convert name to upper case */
	p = realm;
	while(*p != '\0') {
	    *p = toupper(*p);
	    p++;
	}
    }

    /* Figure out the Kerberos principal we are going to use */
    if (principal == NULL) {
	if(princ.name[0] == '\0') {
	    struct passwd *pwd = getpwuid (getuid ());

	    if (pwd == NULL)
		dietx (1, "Could not get default principal");
	    principal = pwd->pw_name;
	} else {
	    principal = princ.name;
	}
    }
    
    /* Figure out the db server we are going to contact */
    if(numservers == 0) {
	dbservers = &onepointer;
	dbservers[0] = (char *)cell_findnamedbbyname (cell);

	if(dbservers[0] == NULL)
	    dietx (1, "Can't find any db server for cell %s", cell);

	numservers = 1;
    } else {
	/* set dbserver pointer to argv[firstserver] */
	dbservers = &argv[firstserver];
    }

    /* Get the password */
    if(!password) {
	password = pwbuf;
	
	if(pipe_mode) {
	    /* PASSWD_MAX = 200 */
	    if(scanf("%198s", password) == 0) {
		dietx (1, "error reading password");
	    }
	} else {
	    snprintf(prompt, PW_PROMPT_MAX, 
		     "%s@%s's Password:", principal, cell);
	    
	    if (des_read_pw_string(password, PW_PROMPT_MAX-1, prompt, 0))
		dietx (1, "Unable to login because can't read password "
			  "from terminal.");
	    
	}
    } else {
	/* Truncate user-specified password to PASSWD_MAX */
	/* This also lets us use memset() to clear it later */
	
	strncpy(pwbuf, password, PASSWD_MAX);
	pwbuf[PASSWD_MAX - 1] = '\0';
	
	password = pwbuf;
    }
    
    /* A familiar AFS warning message */
    if (password == NULL || *password == '\0')
	dietx (1, "Unable to login because zero length password is illegal.");
    
    /* 
     * Create a secure random ticket file if we are running with -setpag,
     * because we can set the environment variable in the child shell.
     */

    if(set_pag) {
	tkfile = randfilename();
	have_useless_ticket_file = 1;
	setenv("KRBTKFILE", tkfile, 1);
    } else {
	if(get_tkt) {
	    tkfile = getenv("KRBTKFILE");
	    if(tkfile == NULL) {
		/* the insecure default ticket file :( */
		tkfile = TKT_FILE;
	    }
	} else {
	    /* Create a unique temporary file to get the temporary TGT */
	    tkfile = randfilename();
	    have_useless_ticket_file = 1;
	}
    }

    krb_set_tkt_string (tkfile);

    /* Get the Kerberos TGT */
    rc = do_timeout(&get_k4_ticket);
    if(timed_out)
	dietx (1, "Timed out trying to get Kerberos ticket for %s@%s", 
	       principal, realm);

    /* get_k4_ticket() will already have printed out an error message. */
    if(rc)
	die(rc);

    /* We can now clear the password from the memory image */
    memset(password, 0, PASSWD_MAX);

    /* Only keep this ticket file around if we were invoked as -tmp; this way,
       the user will still be able to get a Kerberos TGT even if he/she cannot
       obtain a token */
    if(get_tkt)
	have_useless_ticket_file = 0;
    else
	have_useless_ticket_file = 1;

    /* 
     * Figure out the AFS ID to store with the token.
     * Moved this after the TGT-gathering stage because it may want to
     * get a ticket to talk to the dbserver?
     */

    afsid = do_timeout(&get_afs_id);
    if(timed_out)
	warnx("Timed out trying to get AFS ID for %s@%s", principal, cell);

    if (afsid == -1) {
	/* Try various stupid means of guessing the AFS ID */
	struct passwd *pwd;

	pwd = getpwnam(principal);
	if(pwd == NULL) {
	    afsid = getuid();
	    warnx ("Couldn't get AFS ID for %s@%s, using current UID (%d)",
		   principal, cell, afsid);
	} else {
	    afsid = pwd->pw_uid;
	    warnx ("Couldn't get AFS ID for %s@%s, using %d from /etc/passwd",
		   principal, cell, afsid);
	}
    }

    /* Get the AFS token */
    rc = do_timeout(&get_afs_token);
    if(timed_out)
	dietx (1, "Timed out trying to get AFS token for %s@%s", 
	       principal, realm);

    /* get_afs_token() will already have printed out an error message. */
    if(rc)
	die(rc);

    /* Destroy the temporary Kerberos TGT if we didn't need it */
    if(!get_tkt) {
	dest_tkt();

	/* Avoid calling dest_tkt() twice-- may be security risk */
	have_useless_ticket_file = 0;
    }

    /* 
     * Exec a shell if the user specified -setpag, because otherwise the
     *  new PAG will magically disappear when the program exits 
     */

    if(set_pag) {
	/* 
	 * Get the default shell of the PRINCIPAL we are kloging as.
	 * This is the only thing that makes sense, as we are going to
	 * be opening up a session for this principal.
	 * Perhaps we should also change to this principal's home
	 * directory, as the shell will fail if the command was run in
	 * a restricted directory owned by a different AFS principal
	 */

	struct passwd *pwd;
	char *shell;

	pwd = getpwnam(principal);
	if(pwd == NULL) {
	    pwd = getpwuid(getuid());
	    if(pwd == NULL) {

		shell = getenv("SHELL");
		if (shell == NULL) {
		    warnx ("Can't get default shell for user %s, using "
			   "fallback (/bin/sh) shell.", principal);
		    shell = "/bin/sh";
		}

	    } else {
		warnx ("Can't get default shell for user %s, "
		       "using default shell for UID %d", 
		       principal, getuid());

		shell = pwd->pw_shell;
	    }
	} else {
	    shell = pwd->pw_shell;
	}

	execl(shell, shell, NULL);

	/* the AFS token is useless if the shell exec fails, because it
	   is in a PAG that will soon go away. */
	have_useless_token = 1;

	diet (1, "Can't exec shell: %s", shell);
    }

    return 0;
}
