#ifndef lint
static char rcsid[] = "$Id$";
#endif

/*
**  command-line process tools based on Solaris' /usr/proc/bin
**
**  pmap - print an address space map for the given process ids.
**
**  Version 1.0
**
**  This program is property of Netscape Communications Corporation.
**  This entire comment MUST remain intact.
**
**  Copyright 1999 by Chuck Lever (chuckl@netscape.com)
*/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <ctype.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <errno.h>

/* these are copied from vmstat.c.  sigh. */
#define BUFFSIZE 1024
#define PROCFILENAMELEN 80

/* the following was copied from fs/proc/array.c */
#define MAPS_LINE_LENGTH        4096

unsigned char rflag, xflag;

/*
 *  Get the real process id for the process identified by argument
 */
unsigned int verify_pid(char *arg)
{
	unsigned int pid;
	FILE *f;
	char filename[PROCFILENAMELEN];
	char buff[BUFFSIZE];

	sprintf(filename, "/proc/%s/status", arg);

	if (!(f = fopen(filename, "r"))) {
		sprintf(filename, "/proc/%s", arg);
		perror(filename);
		return 0;
	}
	while (fgets(buff, BUFFSIZE, f))
		if (!strncmp(buff, "Pid:", 4))
			sscanf(buff, "%*s %d", &pid);
	fclose(f);
	return pid;
}

/*
 *  Retrieve the pathname of the executable
 */
char *get_exec_path(char *arg)
{
	char filename[PROCFILENAMELEN];
	char buff[BUFFSIZE];
	char *link_text;

	sprintf(filename, "/proc/%s/exe", arg);
	if ((readlink(filename, buff, BUFFSIZE)) == -1) {
		sprintf(filename, "/proc/%s", arg);
		perror(filename);
		return NULL;
	}

	link_text = malloc(strlen(buff));
	strcpy(link_text, buff);
	return link_text;
}

/*
 * Print region protection flags, spelled out
*/
void
print_region_flags(char *a)
{
	if (!strncmp(a, "rwxp", 4))
		printf("read/write/exec");
	if (!strncmp(a, "rw-p", 4))
		printf("read/write     ");
	if (!strncmp(a, "r-xp", 4))
		printf("read/exec      ");
	if (!strncmp(a, "r--p", 4))
		printf("read           ");
	if (!strncmp(a, "-wxp", 4))
		printf("write/exec     ");
	if (!strncmp(a, "-w-p", 4))
		printf("write          ");
	if (!strncmp(a, "--xp", 4))
		printf("exec           ");
	if (!strncmp(a, "---p", 4))
		printf("               ");
	if (!strncmp(a, "rwxs", 4))
		printf("read/write/exec/shared");
	if (!strncmp(a, "rw-s", 4))
		printf("read/write/shared");
	if (!strncmp(a, "r-xs", 4))
		printf("read/exec/shared");
	if (!strncmp(a, "r--s", 4))
		printf("read/shared    ");
	if (!strncmp(a, "-wxs", 4))
		printf("write/exec/shared");
	if (!strncmp(a, "-w-s", 4))
		printf("write/shared   ");
	if (!strncmp(a, "--xs", 4))
		printf("exec/shared    ");
	if (!strncmp(a, "---s", 4))
		printf("shared         ");
	printf("   ");

	return;
}

/*
 *  Find the start of the stack
 */
unsigned long
get_stack(char *arg)
{
	unsigned long stack;
	FILE *f;
	char filename[PROCFILENAMELEN];

	sprintf(filename, "/proc/%s/stat", arg);

	if (!(f = fopen(filename, "r"))) {
		sprintf(filename, "/proc/%s", arg);
		perror(filename);
		return 0;
	}

	fscanf(f, "%*d %*s %*c %*d %*d %*d %*d %*d %*u %*u %*u %*u %*u %*d %*d %*d %*d %*d %*d %*u %*u %*d %*u %*u %*u %*u %*u %*u %lu", &stack);
	fclose(f);
	return stack;
}

/*
 *  Walk through the process's maps file, and parse the lines.
 */
void
do_next_arg(char *arg)
{
	unsigned int pid;
	unsigned long tsize, stack;
	char *executable;
	FILE *f;
	char filename[PROCFILENAMELEN];
	char buff[MAPS_LINE_LENGTH];

	if ((pid = verify_pid(arg)) == 0)
		return;
	if ((executable = get_exec_path(arg)) == NULL)
		return;
	if ((stack = get_stack(arg)) == 0)
		return;
	printf("%d:\t%s\n", pid, executable);
	free(executable);

	sprintf(filename, "/proc/%s/maps", arg);

	if (!(f = fopen(filename, "r"))) {
		sprintf(filename, "/proc/%s", arg);
		perror(filename);
		return;
	}
	tsize = 0;
	while (fgets(buff, MAPS_LINE_LENGTH, f)) {
		unsigned long x, y, size;
 		char *a, *start, *end;

		/* region starting address, in hex, and size, in kilobytes */
		start = buff;
		end = strchr(start, '-');
		*end = '\0';
		for (a=start; *a!='\0'; a++) *a = toupper(*a);
		printf("%s", start);
		end++;
		a = strchr(end, ' ');
		*a = '\0';

		x = strtoul(start, NULL, 16);
		y = strtoul(end, NULL, 16);
		size = (y - x) / 1024;
		printf("%7ldK ", size);
		tsize += size;

		print_region_flags(++a);

		/* mapped filename, "heap", "stack", or "anon" */
		/* Solaris knows where heap is, Linux doesn't */
		/* XXX should also print shmid when known */
		a = strrchr(a, ' ');
		a++;
		if (*a == '/')
			printf("%s", a);
		else if ((x<=stack)&&(stack<=y))
			printf("  [ stack ]\n");
		else
			printf("  [ anon ]\n");
	}
	fclose(f);

	/* print total memory displayed in the map, in kilobytes */
	printf(" total %8ldK\n", tsize);

	return;
}

/*
 *  Walk through the process's maps file, and parse the lines.
 */
void
do_extended_arg(char *arg)
{
	unsigned int pid;
	unsigned long tsize, ssize, psize, stack;
	char *executable;
	FILE *f;
	char filename[PROCFILENAMELEN];
	char buff[MAPS_LINE_LENGTH];

	if ((pid = verify_pid(arg)) == 0)
		return;
	if ((executable = get_exec_path(arg)) == NULL)
		return;
	if ((stack = get_stack(arg)) == 0)
		return;
	printf("%d:\t%s\n", pid, executable);
	free(executable);

	sprintf(filename, "/proc/%s/maps", arg);

	if (!(f = fopen(filename, "r"))) {
		sprintf(filename, "/proc/%s", arg);
		perror(filename);
		return;
	}
	printf("Address   Kbytes Resident Shared Private Permissions       Mapped File\n");
	tsize = ssize = psize = 0;
	while (fgets(buff, MAPS_LINE_LENGTH, f)) {
		unsigned long x, y, size;
 		char *a, *start, *end;

		/* region starting address, in hex, and size, in kilobytes */
		start = buff;
		end = strchr(start, '-');
		*end = '\0';
		for (a=start; *a!='\0'; a++) *a = toupper(*a);
		printf("%s", start);
		end++;
		a = strchr(end, ' ');
		*a = '\0';

		x = strtoul(start, NULL, 16);
		y = strtoul(end, NULL, 16);
		size = (y - x) / 1024;
		printf("%8ld", size);
		tsize += size;

		/* for now, simulate these values */
		printf("%8ld", size);

		a++;
		if (a[3] == 'p') {
			printf("       - %7ld ", size);
			psize += size;
		} else {
			printf("%7ld        - ", size);
			ssize += size;
		}

		print_region_flags(a);

		/* mapped filename, "heap", "stack", or "anon" */
		/* Solaris knows where the heap is, Linux doesn't */
		/* XXX should also print shmid when known */
		a = strrchr(a, '/');
		if (a != NULL)
			printf("%s", ++a);
		else if ((x<=stack)&&(stack<=y))
			printf(" [ stack ]\n");
		else
			printf(" [ anon ]\n");
	}
	fclose(f);

	/* print total memory displayed in the map, in kilobytes */
	printf("--------  ------  ------  ------  ------\n");
	printf("total Kb %7ld %7ld %7ld %7ld\n",
		tsize, tsize, ssize, psize);

	return;
}

int
main(int argc, char **argv)
{
	if (argc < 2) {
		fprintf(stderr, "usage:\tpmap [-r] [-x] pid ...\n");
		fprintf(stderr, "  (report process address maps)\n");
		fprintf(stderr, "  -r: show reserved address maps\n");
		fprintf(stderr, "  -x: show resident/shared/private mapping details\n");
		exit(0);
	}

	/*
	 *  We don't convert argv to an integer; "self" is a legitimate
	 *  thing to look up with these tools, and there may be others
	 *  in the future.
	 */
	rflag = 0;
	xflag = 0;
	while (--argc)
	{
		argv++;
		if (!strcmp(argv[0], "-r")) {
			fprintf(stderr, "pmap: -r not yet supported\n");
			exit(1);
		}
		if (!strcmp(argv[0], "-x")) {
			if (rflag) {
				fprintf(stderr, "pmap: -r and -x are mutually exclusive\n");
				exit(1);
			}
			xflag = 1;
			continue;
		}

		if (xflag)
			do_extended_arg(argv[0]);
		else
			do_next_arg(argv[0]);
	}

	exit(0);
}
