/*
 * Copyright (c) 2002,2005
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of the university of
 * michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software
 * without specific, written prior authorization.  if the
 * above copyright notice or any other identification of the
 * university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.  
 *
 * this software is provided as is, without representation
 * from the university of michigan as to its fitness for any
 * purpose, and without warranty by the university of
 * michigan of any kind, either express or implied, including
 * without limitation the implied warranties of
 * merchantability and fitness for a particular purpose. the
 * regents of the university of michigan shall not be liable
 * for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising
 * out of or in connection with the use of the software, even
 * if it has been or is hereafter advised of the possibility of
 * such damages.
 */

/* TODO:
 * dont save web_server credentials in file.
 * should this be configurable?: TGS_VERSION "kct version 1"
 * better logic to hook into save_handshake (affects this & mod_ssl)
 * com_err -- replace with ap_log_printf.
 * should be able to get more than one service ticket on behalf of
 *	a client.
 *
 * FIXED:
 * better name for ticket than /tmp/mdw [This is now somewhat configurable]
 * kct_init -- calls exit.  (?!). [removed]
 * eliminate silly debugging messages (ap_user_id etc.) [Change log-level...]
 */

/* apache includes */
#include "ap_config.h"
#include "ap_alloc.h"
#include "httpd.h"
#include "http_config.h"
#include "http_conf_globals.h"
#include "http_protocol.h"
#include "http_request.h"
#include "http_main.h"
#include "http_core.h"
#include "http_log.h"
  
/* krb5 includes */
#include <krb5.h>
#include <com_err.h>

/* openssl includes */
#include <openssl/ssl.h>
#include "mod_ssl.h"

#ifdef KCT_CC_LOCKS
#include "common_util.c"
#endif

typedef struct saved_handshake_st {

	unsigned char *c_h;
	unsigned int c_h_l;

	unsigned char *s_h;
	unsigned int s_h_l;

	unsigned char *s_c;
	unsigned int s_c_l;

	unsigned char *s_c_r;
	unsigned int s_c_r_l;

	unsigned char *s_k_e;
	unsigned int s_k_e_l;

	unsigned char *c_c;
	unsigned int c_c_l;

	unsigned char *c_k_e;
	unsigned int c_k_e_l;

	unsigned char *c_v;
	unsigned int c_v_l;

} SSL_HANDSHAKE;

/* socket library */
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/tcp.h>
#include <signal.h>
#include <unistd.h>
#include <assert.h>


#define REALM_SZ      40
#define BUFF_SIZE     1024
#define CACHE_SIZE    512
#define KEY_SIZE      9
#define CLIENTID_SIZE KEY_SIZE
#define EMAIL_SIZE    100
#define HEADER_SIZE   2
#define TKT_CACHE_LEN MD5_DIGEST_LENGTH+11

#define MAX_NUM_TKT_ENTRIES 10

#define NO_ERROR               0
#define SOCKET_FAILED          239
#define GETHOSTBYNAME_FAILED   240
#define CONNECT_FAILED         241
#define NETWORK_FAILED         242
#define MEMORY_FAILED          243
#define NO_TICKET              244
#define COMMUNICATION_ERROR    246

#define TGS_VERSION "kct version 2"

/* kct error codes */
#define KCT_ERR_NO_ERROR                0
#define KCT_ERR_INTERNAL_ERROR          250
#define KCT_ERR_RD_PRIV                 251
#define KCT_ERR_INVALID_HANDSHAKE       252
#define KCT_ERR_OLD_HANDSHAKE           253
#define KCT_ERR_INVALID_ID              254
#define KCT_ERR_REALM_MISMATCH          255
#define KCT_ERR_COMMUNICATION_ERROR     256
#define KCT_ERR_UNAUTHORIZED            257
#define KCT_ERR_INTERNAL_KERBEROS_ERROR 258
#define KCT_ERR_UNKNOWN_PRINCIPAL       259
#define KCT_ERR_BAD_PRINCIPAL           260
#define KCT_ERR_MALFORMED_LIFETIME      261

#define ERR_NO_CLIENT_CERTIFICATE       300
#define ERR_BAD_IDENTIFIER		301
#define ERR_SSL_RENEG_FAILED            302
#define ERR_GET_TGT_FAILED		303
#define ERR_HTTPS_EXPECTED              304

#define MODKCT_ERR_SENDAUTH_FAILED      400
#define MODKCT_ERR_INTERNAL_KERBEROS_ERROR      401

typedef struct auth_data_struct {
  int               sock;
  unsigned char     *client_id;
  krb5_context      context;
  krb5_ccache       ccache;
  krb5_keytab       keytab;
  krb5_principal    tgs_server;
  krb5_principal    web_server;
  krb5_creds        ws_creds;
  krb5_auth_context auth_context;
  krb5_creds        *tgs_creds;
  krb5_creds        *service_creds;
} AUTH_DATA;

typedef struct {
  int    kct_err;
  int    nPostRead;
  int    nTranslate;
  int    nFixups;
  module *ssl_mod;
} KCT_MOD_CONFIG_REC;

typedef struct {
  char *service_name;
  char *lifetime;
} KCT_SERVICES;

typedef struct {
  char *kct_version;
  char *kct_keytab;
  char *kct_ws_name;
  int num_services;
  KCT_SERVICES **kct_services;
  char *kct_tgs_service_name;
  char *kct_kdc_fullname;
  char *kct_tgs_server_hostname;	/* kct daemon host */
  int kct_tgs_server_port;		/* kct daemon port */
  char *kct_tkt_cache_name;		/* ccache used to talk to kct daemon */
  krb5_deltat kct_tgs_lifetime;		/* kct service ticket lifetime */
  char *kct_default_realm;
  char *kct_client_prefix;
  int  bEnabled;
  int  bVerifyClient;
  int  bCaching;
} KCT_SRV_CONFIG_REC;

typedef struct {
  char *kct_keytab;
  char *kct_ws_name;
  int num_services;
  KCT_SERVICES **kct_services;
  char *kct_tgs_service_name;
  char *kct_kdc_fullname;
  char *kct_tgs_server_hostname;	/* kct daemon host */
  int kct_tgs_server_port;		/* kct daemon port */
  char *kct_tkt_cache_name;		/* ccache used to talk to kct daemon */
  krb5_deltat kct_tgs_lifetime;		/* kct service ticket lifetime */
  char *kct_default_realm;
  char *kct_client_prefix;
  int  bVerifyClient;
} KCT_DIR_CONFIG_REC;

static int kct_set_default_config(server_rec *, pool *);
static void print_buffer(request_rec *r, unsigned char *buffer, size_t len, 
			 unsigned char *comment);
static size_t recvblock(request_rec *r, int sd, unsigned char *buf, 
                        size_t total);
static size_t sendblock(request_rec *r, int sd, unsigned char *buf, 
                        size_t total);
static void create_message(request_rec *, unsigned int, unsigned char **, 
                           unsigned char *);
static int krb5_kdcrep2creds_local(krb5_context, krb5_kdc_rep*, 
				   krb5_address **,
				   krb5_data*, krb5_creds**, pool *);

static int do_kct(request_rec *r, KCT_SERVICES **, int, int);
static int get_tgt(request_rec *);
static int get_client_identity(pool *p, X509 *, unsigned char **);
static int check_if_ticket_cached(request_rec *r, unsigned char *, 
				  KCT_SERVICES **, int, 
				  KCT_SERVICES ***, int *);
static int check_if_expired(request_rec *r, krb5_creds v5_creds);
static int connect_to_kct(request_rec *r, AUTH_DATA *);
static int send_handshake(AUTH_DATA *, SSL *, SSL_HANDSHAKE *sh, request_rec *);
static int send_sessiondata(AUTH_DATA *, SSL *,  KCT_SERVICES **, int, 
			    request_rec *);
static int receive_service_ticket(AUTH_DATA *, request_rec *);

static int renegotiate(request_rec *, SSL *);
static void kct_hook_NewConnection(conn_rec *conn);

static void send_error(request_rec *r, int retval);
static int check_if_verify_client_enabled(server_rec *r, KCT_DIR_CONFIG_REC *);
static int check_if_ssl_enabled(conn_rec *);
static void initialize_handshake(SSL_HANDSHAKE *sh);

#ifdef PRINT_HANDSHAKE
static void print_handshake(SSL_HANDSHAKE *sh);
#endif

#ifdef DEBUG_SSL_ID
static void print_ssl_info(SSL *, server_rec *, char *);
#endif

char kct_module_name[] = "kct_module";
extern module kct_module;	/* forward dcl */

#define kctModConfig() (KCT_MOD_CONFIG_REC *)ap_ctx_get(ap_global_ctx, "kct_module")
#define kctSrvConfig(srv) (KCT_SRV_CONFIG_REC *)ap_get_module_config(srv->module_config, &kct_module) 
#define kctDirConfig(req) (KCT_DIR_CONFIG_REC *)ap_get_module_config(req->per_dir_config, &kct_module)

void kct_init(server_rec *s, pool *p) {

  pool               *pPool;
  KCT_MOD_CONFIG_REC *mod;
  
  if((mod = ap_ctx_get(ap_global_ctx, "kct_module")) == NULL) {
    pPool = ap_make_sub_pool(NULL);
    mod = (KCT_MOD_CONFIG_REC *)ap_palloc(pPool, sizeof(KCT_MOD_CONFIG_REC));
    ap_ctx_set(ap_global_ctx, "kct_module", mod);
  }
  mod->kct_err    = NO_ERROR;
  mod->nPostRead  = 0;
  mod->nTranslate = 0;
  mod->nFixups    = 0; 
  mod->ssl_mod    = NULL;
  if((mod->ssl_mod = ap_find_linked_module("mod_ssl.c")) == NULL) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, s,
                 "mod_kct: ssl module is not linked in");
  }
  return;
}

void *kct_config_perserver_create (pool *p, server_rec *s) {
  KCT_SRV_CONFIG_REC *srv = NULL;
  int i = 0;

  srv = ap_pcalloc (p, sizeof(KCT_SRV_CONFIG_REC));

  srv->kct_keytab = NULL;
  srv->kct_ws_name = NULL;
  srv->num_services = 0;
  srv->kct_services = (KCT_SERVICES **)
    ap_palloc(p, MAX_NUM_TKT_ENTRIES*sizeof(KCT_SERVICES));
  for(i = 0; i < MAX_NUM_TKT_ENTRIES; i++) {
    srv->kct_services[i] = NULL;
  }
  srv->kct_tgs_service_name = NULL;
  srv->kct_kdc_fullname = NULL;
  srv->kct_tgs_server_hostname = NULL;
  srv->kct_tgs_server_port = -1;
  srv->kct_tkt_cache_name = NULL;
  srv->kct_tgs_lifetime = -1;
  srv->kct_default_realm = NULL;
  srv->bEnabled = -1;
  srv->bVerifyClient = -1;
  srv->kct_client_prefix = NULL;
  srv->bCaching = 1;
  srv->kct_version = ap_pstrdup(p, TGS_VERSION);
  return srv;
}

void *kct_config_perserver_merge(pool *p, void *basev, void *addv) {
  KCT_SRV_CONFIG_REC *base = (KCT_SRV_CONFIG_REC *)basev;
  KCT_SRV_CONFIG_REC *add = (KCT_SRV_CONFIG_REC *)addv;
  KCT_SRV_CONFIG_REC *new = 
  (KCT_SRV_CONFIG_REC *)ap_palloc(p, sizeof(KCT_SRV_CONFIG_REC));
  int i = 0, j = 0;

  cfgMergeString(kct_keytab);  
  cfgMergeString(kct_ws_name);  
  new->kct_services = (KCT_SERVICES **)
    ap_palloc(p, MAX_NUM_TKT_ENTRIES*sizeof(KCT_SERVICES));
  new->num_services = 0;
  for(i = 0; i < base->num_services; i++) {
    new->kct_services[i] = (KCT_SERVICES *) ap_palloc(p, sizeof(KCT_SERVICES));
    new->kct_services[i]->service_name = 
      ap_pstrdup(p, base->kct_services[i]->service_name);
    new->kct_services[i]->lifetime = 
      ap_pstrdup(p, base->kct_services[i]->lifetime);
    new->num_services++;
  }

  for(j = 0; j < add->num_services; j++, i++) {
    new->kct_services[i] = (KCT_SERVICES *) ap_palloc(p, sizeof(KCT_SERVICES));
    new->kct_services[i]->service_name = 
      ap_pstrdup(p, add->kct_services[j]->service_name);
    new->kct_services[i]->lifetime = 
      ap_pstrdup(p, add->kct_services[j]->lifetime);
    new->num_services++;
  }

  cfgMergeString(kct_tgs_service_name);  
  cfgMergeString(kct_kdc_fullname);  
  cfgMergeString(kct_tgs_server_hostname);
  cfgMergeInt(kct_tgs_server_port);  
  cfgMergeString(kct_tkt_cache_name);
  cfgMergeInt(kct_tgs_lifetime);  
  cfgMergeString(kct_default_realm);
  cfgMergeInt(bEnabled);  
  cfgMergeInt(bVerifyClient);
  cfgMergeString(kct_client_prefix);
  cfgMergeInt(bCaching);
  cfgMergeString(kct_version);
  return new;  
}

void *kct_config_perdir_create (pool *p, char *d) {
  KCT_DIR_CONFIG_REC *dir = NULL;
  int i = 0;

  dir = (KCT_DIR_CONFIG_REC *)ap_palloc(p, sizeof(KCT_DIR_CONFIG_REC));

  dir->kct_keytab = NULL;
  dir->kct_ws_name = NULL;
  dir->num_services = 0;
  dir->kct_services = (KCT_SERVICES **)
    ap_palloc(p, MAX_NUM_TKT_ENTRIES*sizeof(KCT_SERVICES));
  for(i = 0; i < MAX_NUM_TKT_ENTRIES; i++) {
    dir->kct_services[i] = NULL;
  }
  dir->kct_tgs_service_name = NULL;
  dir->kct_kdc_fullname = NULL;
  dir->kct_tgs_server_hostname = NULL;
  dir->kct_tgs_server_port = -1;
  dir->kct_tkt_cache_name = NULL;
  dir->kct_tgs_lifetime = -1;
  dir->kct_default_realm = NULL;
  dir->bVerifyClient = -1;
  dir->kct_client_prefix = NULL;
  
  return dir;
}

void *kct_config_perdir_merge(pool *p, void *basev, void *addv) {
  KCT_DIR_CONFIG_REC *base = (KCT_DIR_CONFIG_REC *)basev;
  KCT_DIR_CONFIG_REC *add = (KCT_DIR_CONFIG_REC *)addv;
  KCT_DIR_CONFIG_REC *new = 
  (KCT_DIR_CONFIG_REC *)ap_palloc(p, sizeof(KCT_DIR_CONFIG_REC));
  int i = 0, j = 0;

  cfgMergeString(kct_keytab);  
  cfgMergeString(kct_ws_name);  
  // kct_services COMEBACK
  new->kct_services = (KCT_SERVICES **)
    ap_palloc(p, MAX_NUM_TKT_ENTRIES*sizeof(KCT_SERVICES));
  new->num_services = 0;
  for(i = 0; i < base->num_services; i++) {
    new->kct_services[i] = (KCT_SERVICES *) ap_palloc(p, sizeof(KCT_SERVICES));
    new->kct_services[i]->service_name = 
      ap_pstrdup(p, base->kct_services[i]->service_name);
    new->kct_services[i]->lifetime = 
      ap_pstrdup(p, base->kct_services[i]->lifetime);
    new->num_services++;
  }

  for(j = 0; j < add->num_services; j++, i++) {
    new->kct_services[i] = (KCT_SERVICES *) ap_palloc(p, sizeof(KCT_SERVICES));
    new->kct_services[i]->service_name = 
      ap_pstrdup(p, add->kct_services[j]->service_name);
    new->kct_services[i]->lifetime = 
      ap_pstrdup(p, add->kct_services[j]->lifetime);
    new->num_services++;
  }

  cfgMergeString(kct_tgs_service_name);  
  cfgMergeString(kct_kdc_fullname);  
  cfgMergeString(kct_tgs_server_hostname);
  cfgMergeInt(kct_tgs_server_port);  
  cfgMergeString(kct_tkt_cache_name);
  cfgMergeInt(kct_tgs_lifetime);  
  cfgMergeString(kct_default_realm);
  cfgMergeInt(bVerifyClient);
  cfgMergeString(kct_client_prefix);

  return new;  
}

/**
 ** Implement the 'kctkeytab' command (which should appear
 **  in SERVER_ROOT/conf/srm.conf)
 **/
const char *kct_set_version(cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir,
                            char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);
                                                                                
  if (!srv) return "Help! no srv in kct_set_version.";
  if (!arg) return "Must supply KCT Kerberos version string here.";
                                                                                
  srv->kct_version = ap_pstrdup(cmd->pool, arg);
                                                                                
  return NULL;
}

const char *kct_set_keytab (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
			    char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no srv in kct_set_keytab.";
  if (!arg) return "Must supply a keytab here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_keytab = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_keytab = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_ws_name (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
			     char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_ws_name.";
  if (!arg) return "Must supply a ws_name here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_ws_name = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_ws_name = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_service_name (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				  char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);
  char *service_name = NULL;
  char *lifetime = NULL;

  if (!srv) return "Help! no sec in kct_set_service_name.";
  if (!arg) return "Must supply a name of a kerberized service and "
    "its ticket lifetime where format is <kca_service:lifetime>.";

  service_name = ap_getword_nc(cmd->pool, &arg, ':');
  if((lifetime = ap_get_token(cmd->pool, (const char **)&arg, 1)) == NULL ||
     !strcmp(lifetime, "")) 
    return "Must specify lifetime for a service eg. kca_service:10h";

  if(cmd->path == NULL || dir == NULL) {
    srv->kct_services[srv->num_services] = 
      (KCT_SERVICES *)ap_palloc(cmd->pool, sizeof(KCT_SERVICES));
    srv->kct_services[srv->num_services]->service_name = 
      ap_pstrdup(cmd->pool, service_name);
    srv->kct_services[srv->num_services]->lifetime = 
      ap_pstrdup(cmd->pool, lifetime);
    srv->num_services++;
  }
  else {
    dir->kct_services[dir->num_services] = 
      (KCT_SERVICES *)ap_palloc(cmd->pool, sizeof(KCT_SERVICES));
    dir->kct_services[dir->num_services]->service_name = 
      ap_pstrdup(cmd->pool, service_name);
    dir->kct_services[dir->num_services]->lifetime = 
      ap_pstrdup(cmd->pool, lifetime);
    dir->num_services++;
  }
  return NULL;
}

const char *kct_set_tgs_service_name (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				      char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_tgs_service_name.";
  if (!arg) return "Must supply a tgs_service_name here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_tgs_service_name = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_tgs_service_name = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_kdc_fullname (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				  char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_kdc_fullname.";
  if (!arg) return "Must supply a kdc_fullname here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_kdc_fullname = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_kdc_fullname = ap_pstrdup(cmd->pool, arg);
    
  return NULL;
}

const char *kct_set_tgs_server_hostname (cmd_parms *cmd, 
					 KCT_DIR_CONFIG_REC *dir, char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_tgs_server_hostname.";
  if (!arg) return "Must supply a tgs_server_hostname here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_tgs_server_hostname = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_tgs_server_hostname = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_tgs_server_port (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				     char *arg) {
  server_rec *s = cmd->server;
  int x;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_tgs_server_port.";
  if (!arg) return "Must supply a tgs_server_port here.";

  if (sscanf(arg, "%d", &x) != 1)
      return "TGS port number must be a number.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_tgs_server_port = x;
  else
    dir->kct_tgs_server_port = x;

  return NULL;
}

const char *kct_set_tkt_cache_name (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				    char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_tkt_cache_name.";
  if (!arg) return "Must supply a tkt_cache_name here.";

  if(cmd->path == NULL || dir == NULL)
    srv->kct_tkt_cache_name = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_tkt_cache_name = ap_pstrdup(cmd->pool, arg);
    
  return NULL;
}

const char *kct_set_tgs_lifetime (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				  char *arg) {
  server_rec *s = cmd->server;
  int retval;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_tgs_lifetime.";
  if (!arg) return "Must supply a tgs_lifetime here.";

  if(cmd->path == NULL || dir == NULL) {
    if((retval = krb5_string_to_deltat(arg, &srv->kct_tgs_lifetime)) ||
       srv->kct_tgs_lifetime == 0)
      return "Bad TGS lifetime";
  }
  else {
    if((retval = krb5_string_to_deltat(arg, &dir->kct_tgs_lifetime)) ||
       dir->kct_tgs_lifetime == 0)
      return "Bad TGS lifetime";
  }
  return NULL;
}

const char *kct_set_default_realm (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				   char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_default_realm.";
  if (!arg) return "Must supply a default_realm here.";

  if(cmd->path == NULL || dir == NULL) 
    srv->kct_default_realm = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_default_realm = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_enableEngine (cmd_parms *cmd, void *dir, int flag) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_enableEngine.";

  srv->bEnabled = (flag ? 1: 0);

  return NULL;
}

const char *kct_set_verifyClient (cmd_parms *cmd, KCT_DIR_CONFIG_REC *dir, 
				  int flag) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_verifyClient.";

  if(cmd->path == NULL || dir == NULL) 
    srv->bVerifyClient = (flag ? 1: 0);
  else
    dir->bVerifyClient = (flag ? 1: 0);

  return NULL;
}

const char *kct_set_client_cache_prefix (cmd_parms *cmd, 
				       KCT_DIR_CONFIG_REC *dir, char *arg) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_client_cache_name.";

  if(cmd->path == NULL || dir == NULL) 
    srv->kct_client_prefix = ap_pstrdup(cmd->pool, arg);
  else
    dir->kct_client_prefix = ap_pstrdup(cmd->pool, arg);

  return NULL;
}

const char *kct_set_cred_caching (cmd_parms *cmd, void *d, int flag) {
  server_rec *s = cmd->server;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);

  if (!srv) return "Help! no sec in kct_set_credscaching.";

  srv->bCaching = (flag ? 1: 0);

  return NULL;
}
/**
 ** "glue" for our commands.
 **/

command_rec kct_cmds[] = {
{"kctversionstr", kct_set_version,  NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify Kerberos version string"},
{"kctkeytab", kct_set_keytab, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify keytab file" },
{ "kctwsname", kct_set_ws_name, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify web server's name" },
{ "kctservicename", kct_set_service_name, NULL, RSRC_CONF|OR_AUTHCFG, ITERATE, "specify a list of kerberized services with their lifetimes, separated by a colon" },
{ "kcttgsservicename", kct_set_tgs_service_name, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify tgs_service_name principal" },
{ "kctkdcfullname", kct_set_kdc_fullname, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify kdc_fullname hostname" },
{ "kcttgsserverhostname", kct_set_tgs_server_hostname, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify tgs_server_hostname" },
{ "kcttgsserverport", kct_set_tgs_server_port, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify tgs_server_port number" },
{ "kcttktcachename", kct_set_tkt_cache_name, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify tkt_cache_name cc-name" },
{ "kcttgslifetime", kct_set_tgs_lifetime, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify tgs_lifetime" },
{ "kctrealm", kct_set_default_realm, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify default_realm" },
{ "kctengine", kct_set_enableEngine, NULL, RSRC_CONF, FLAG, "KCT switch for the KCT module" },
{ "kctverifyclient", kct_set_verifyClient, NULL, RSRC_CONF|OR_AUTHCFG, FLAG, "KCT client verify" },
{ "kctclientcacheprefix", kct_set_client_cache_prefix, NULL, RSRC_CONF|OR_AUTHCFG, TAKE1, "specify prefix for the client's ticket cache"},
{ "kctcaching", kct_set_cred_caching, NULL, RSRC_CONF, FLAG, "switch for credential caching" },
{ NULL }
};

static int do_kct(request_rec *r, KCT_SERVICES **services, int num_services, 
		  int cache) {
  int retval = NO_ERROR; 
  AUTH_DATA *data = (AUTH_DATA *)ap_pcalloc(r->pool, sizeof(AUTH_DATA));
  SSL *s = ap_ctx_get(r->connection->client->ctx, "ssl");
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  char *service_name = "kca_service/yoga.citi.umich.edu";
  KCT_SERVICES **non_cached_services;
  int num_non_cached_services = 0, i = 0;
  
#ifdef DEBUG_SSL_ID
  print_ssl_info(s, r->server, "do_kct");
#endif

  if (!s->session->peer) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: No client certificate");
    mod->kct_err = ERR_NO_CLIENT_CERTIFICATE;
    return OK; 
  }

  s->msg_callback(0, s->version, SSL3_RT_HANDSHAKE,
                     s->s3->handshake_fragment, 4,
                     s, s->msg_callback_arg);

  /** retrieve client's identity from the certificate inside of the
   ** of the SSL context
   **
   ** function returns -1 if it's unable to locate an email field within
   ** the certificate
   **/
  if(get_client_identity(r->pool, s->session->peer, &data->client_id) < 0) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: wasn't able to get client's identity from the cert");
    mod->kct_err = ERR_BAD_IDENTIFIER;
    return OK;
  }
  if(data->client_id)
    ap_table_setn(r->subprocess_env, "CLIENT_ID", data->client_id);

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
	       "request %s from %s (caching=%d)", 
	       r->uri, data->client_id, cache);
  
  /** need to check if there already a ticket for the requestor
   ** if so, 
   **  return control
   ** if not,
   **  check for the partial handshake
   **  if so,
   **    request a renegotiation (so that we get a full handshake to get the
   **    the necessary ticket
   **  if not,
   **    proceed with acquiring the ticket
   **/
  if(cache) {
    switch (check_if_ticket_cached(r, data->client_id, services, num_services,
				   &non_cached_services, 
				   &num_non_cached_services)) {
    case NO_ERROR:
      return DECLINED;
    default:
      if(s->msg_callback_arg == NULL) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		     "about to renegotiate SSL handshake");
	if(renegotiate(r, s) == FORBIDDEN) {
	  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		       "unable to renegotiate");
	  mod->kct_err = ERR_SSL_RENEG_FAILED;
	  return OK;
	}
	s = ap_ctx_get(r->connection->client->ctx, "ssl");
      }
    }
  }
  else {
    if(s->msg_callback_arg == NULL) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
        "[%d]: mod_kct: partial handshake (non caching)", getpid());
      if(renegotiate(r, s) == FORBIDDEN) {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
        "unable to renegotiate");
	mod->kct_err = ERR_SSL_RENEG_FAILED;
        return OK;
      }
      s = ap_ctx_get(r->connection->client->ctx, "ssl"); 
    }
  }

  /** connect to kct and get a service ticket
   ** 
   ** web server's tgt could expire. if so, then get a new tgt and 
   ** resubmit the request
   **/
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
    "mod_kct: calling kct for '%s' from '%s'", r->uri, data->client_id);
  
  retval = connect_to_kct(r, data);
  switch (retval) {
  case NO_ERROR: break;
  case HTTP_INTERNAL_SERVER_ERROR: return HTTP_INTERNAL_SERVER_ERROR;

  case KRB5KRB_AP_ERR_TKT_EXPIRED:
  case KRB5_FCC_NOFILE: 
  case KRB5_CC_NOTFOUND:

    mod->kct_err = NO_ERROR;

    switch(get_tgt(r)) {
    case NO_ERROR: break;
    case HTTP_INTERNAL_SERVER_ERROR: return HTTP_INTERNAL_SERVER_ERROR;
    default: 
      /**
       ** if a web server is unable to get a tgt, then it will not be able
       ** to service client's request. we need to return some kind of
       ** SERVER INTERNAL error message here
       **/
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
        "mod_kct: unable to get a tgt after encountering expired tgt");
      mod->kct_err = ERR_GET_TGT_FAILED;
      return OK;
    }

    retval = connect_to_kct(r, data);
    switch(retval) {
    case NO_ERROR: break;      
    case HTTP_INTERNAL_SERVER_ERROR: return HTTP_INTERNAL_SERVER_ERROR;
    default: 
      /**
       ** after getting a tgt, if connect_to_kct returned an error
       ** let the client know
       **/
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
        "mod_kct: unable to get a service ticket after obtaining new tgt");
      mod->kct_err = retval;
      return OK;
    }
    break;

  case SOCKET_FAILED:
  case GETHOSTBYNAME_FAILED:
  case CONNECT_FAILED:
  default: 
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: failed to get a service ticket");
    if(data->sock) ap_pclosesocket(r->pool, data->sock);
    mod->kct_err = retval;
    return OK;
  }
  /** after authenticating to kct, send the handshake
   **/
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
    "mod_kct: sending the SSL handshake");

  retval = send_handshake(data, s, s->msg_callback_arg, r);
  switch (retval) {
  case NO_ERROR: break;
  default: 
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: failed while sending SSL handshake");
    if(data->sock) ap_pclosesocket(r->pool, data->sock);
    mod->kct_err = retval;
    return OK;
  }
  
  /** send the master session key, so that kct can verify the
   ** handshake. this will no longer be needed in TLS
   **/
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
    "mod_kct: sending the key");
  retval = send_sessiondata(data, s, non_cached_services, 
			    num_non_cached_services, r);
  switch (retval) {
  case NO_ERROR: break;
  default: 
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: failed while sending SSL master session key"); 
    if(data->sock) ap_pclosesocket(r->pool, data->sock);
    mod->kct_err = retval;
    return OK;
  }

  /** receive the service ticket from kct
   **/
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
	       "mod_kct: receiving %d service ticket(s)", 
	       num_non_cached_services);
  for(i = 0; i < num_non_cached_services; i++)
    retval = receive_service_ticket(data, r);
  
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: closes socket = %d\n", data->sock);

  if(data->sock) ap_pclosesocket(r->pool, data->sock);
  krb5_free_cred_contents(data->context, data->tgs_creds);
  if(!retval) krb5_free_cred_contents(data->context, data->service_creds);
  if(data->ccache) krb5_cc_close(data->context, data->ccache);
  if(data->web_server) krb5_free_principal(data->context, data->web_server);
  if(data->tgs_server) krb5_free_principal(data->context, data->tgs_server);
  if(data->auth_context) krb5_auth_con_free(data->context, data->auth_context);
  if(data->context) krb5_free_context(data->context);

  switch (retval) {
  case NO_ERROR: break;
  case HTTP_INTERNAL_SERVER_ERROR: return HTTP_INTERNAL_SERVER_ERROR;
  default: 
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "mod_kct: failed to receive service ticket");
    mod->kct_err = retval;
    return OK;
  }

#ifdef SSLv3_RENEGOTIATE
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
    "mod_kct: SSLv3_RENEGOTIATE since master session key has been exposted");
  if(renegotiate(r, s) == FORBIDDEN) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
      "unable to renegotiate");
    mod->kct_err = ERR_SSL_RENEG_FAILED;
    return OK;
  }
#endif

  return DECLINED;
}



static int get_client_identity(pool *p, X509 *cert, unsigned char **name) {

  X509_NAME *client=NULL;
  X509_NAME_ENTRY *entry=NULL;
  int i, n, entries;
  char *tmp;
  char *q;

  tmp = (unsigned char *)ap_pcalloc(p, EMAIL_SIZE);
  (*name) = (unsigned char *)ap_pcalloc(p, CLIENTID_SIZE);

  client = X509_get_subject_name(cert);
  X509_NAME_get_text_by_NID(client, NID_pkcs9_emailAddress, tmp, 
			    EMAIL_SIZE);
  if(!(q = memccpy(*name, tmp, '@', EMAIL_SIZE)))
    return -1;	/* XXX */
  *--q = 0;
  return OK;
}

/* XXX dont need pool if have r */
static int check_if_ticket_cached(request_rec *r,
                                  unsigned char *client_id,
				  KCT_SERVICES **services,
				  int num_services,
				  KCT_SERVICES ***non_cached_services,
				  int *num_non_cached_services) {
  krb5_context context;
  krb5_ccache ccache;
  krb5_creds in_creds, v5creds;
  krb5_error_code retval = 0;
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  KCT_DIR_CONFIG_REC *dir = kctDirConfig(r);
  char *default_realm = NULL;
  char *cache_prefix = NULL;
  char *tkt_cache_name = NULL;
#ifdef KCT_CC_LOCKS
  int lockfd;
#endif

  //COME BACK
  int i = 0;

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: checking if tickets are cached START\n");

  if(dir->kct_default_realm != NULL)
    default_realm = dir->kct_default_realm;
  else if(srv->kct_default_realm != NULL)
    default_realm = srv->kct_default_realm;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		 "mod_kct: kerberos realm was not supplied");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_client_prefix != NULL)
    cache_prefix = dir->kct_client_prefix;
  else if(srv->kct_client_prefix != NULL)
    cache_prefix = srv->kct_client_prefix;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		 "mod_kct: client cache prefix was not supplied");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  (*non_cached_services) = (KCT_SERVICES **)ap_palloc(r->pool, 
						      MAX_NUM_TKT_ENTRIES);
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: will checking %d services\n", num_services);

  if((retval = krb5_init_context(&context)) != 0) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		 "krb5_init_context() failed (%d)\n", retval);
    com_err(kct_module_name, retval, "while initializing krb5");
    goto cleanup1;
  }
    
  for(i = 0; i < num_services; i++) {
    char *service_name = NULL, *token = "/", *replace = NULL;
    int valid_cache = 0, valid_creds = 0, tkt_cache_size = 0;

    /* change the /'s in the Kerberos service name to _'s */
    service_name = ap_pstrdup(r->pool, services[i]->service_name);    
    if((replace = strpbrk(service_name, token)) != NULL)
	 replace[0] = '_';

    tkt_cache_size = strlen(client_id) + strlen(cache_prefix) + 
      strlen(service_name) + 2;

    tkt_cache_name = (unsigned char *)ap_pcalloc(r->pool, tkt_cache_size);
            
    sprintf(tkt_cache_name, "%s%s_%s", cache_prefix, client_id, service_name);
    ap_table_setn(r->subprocess_env, "KRB5CCNAME", tkt_cache_name);

#ifdef KCT_CC_LOCKS
    if ((lockfd = open_and_lock(tkt_cache_name, kct_module_name)) == -1) {
      retval = MODKCT_ERR_INTERNAL_KERBEROS_ERROR;
      goto cleanup2;
    }
#endif
    
    if((retval = krb5_cc_resolve(context, tkt_cache_name, &ccache)) != 0) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		   "krb5_cc_resolve(%s) failed (%d)", tkt_cache_name, retval);
      com_err(kct_module_name, retval, "unable to resolve krb cache");
      goto cleanup3;
    } else
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		   "mod_kct: resolved cache %s for service", tkt_cache_name, 
		   service_name);    

    if((retval = krb5_cc_get_principal(context, ccache, 
				       &in_creds.client)) != 0) {
      if(retval != KRB5_CC_FORMAT && retval != KRB5_FCC_NOFILE) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		     "krb5_cc_get_principal() failed (%d)", retval);
	com_err(kct_module_name, retval, 
		"unable to get the default principal");
	krb5_cc_close(context, ccache);
	goto cleanup3;
      } else 
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		     "krb5_cc_get_principal() returned (%d)\n", retval);
    } else {
      valid_cache = 1;
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		   "mod_kct: verified that cache belongs to %s", client_id);
    }

    if(valid_cache) {
      if((retval = krb5_parse_name(context, services[i]->service_name,
				   &in_creds.server)) != 0) {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		     "krb5_parse_name() failed (%d)\n", retval);
	com_err(kct_module_name, retval, 
		"unable to convert %s to a principal", service_name);
	krb5_cc_close(context, ccache);
	goto cleanup3;
      }
    
      if ((retval = krb5_cc_retrieve_cred(context, ccache, 
					  KRB5_TC_MATCH_SRV_NAMEONLY, 
					  &in_creds, &v5creds)) != 0) {
	if(retval != KRB5_CC_NOTFOUND && retval != KRB5_FCC_NOFILE) {
	  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		       "krb5_cc_retrieve_cred() failed (%d)\n", retval);
	  com_err(kct_module_name, retval, "while retrieving creds: %s", 
		  in_creds.server->data->data);
	  krb5_cc_close(context, ccache);
	  goto cleanup3;
	} else 
	  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		       "krb5_cc_retrieve_cred() returned (%d)\n", retval);
      } else {
	valid_creds = 1;
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		     "mod_kct: found an entry for service %s", service_name);
      }

      if(valid_creds) retval = check_if_expired(r, v5creds);
    }

    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server, 
		 "mod_kct: check_if_ticket_cached: ccache=%s service=%s "
		 "client=%s status=%s", tkt_cache_name, service_name, 
		 client_id, !retval ? "valid" : "invalid");

    if(retval || !valid_cache) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
		   "mod_kct: adding to non_cached services");
      (*non_cached_services)[(*num_non_cached_services)] = 
	(KCT_SERVICES *)ap_palloc(r->pool, sizeof(KCT_SERVICES));
      (*non_cached_services)[(*num_non_cached_services)]->service_name = 
	ap_pstrdup(r->pool, services[i]->service_name);
      (*non_cached_services)[(*num_non_cached_services)]->lifetime = 
	ap_pstrdup(r->pool, services[i]->lifetime);
      (*num_non_cached_services)++;
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, "done");
    }

    if(valid_cache)
      krb5_cc_close(context, ccache);
  }

  // this is set so the mod_kx509 can lookup creds
  ap_table_setn(r->subprocess_env, "TKT_CC_PREFIX", cache_prefix);

 cleanup3:
#ifdef KCT_CC_LOCKS
  unlock_and_close(lockfd, kct_module_name);
#endif

 cleanup2:
  krb5_free_context(context);

 cleanup1:
  if(retval != NO_ERROR && retval != KRB5_FCC_NOFILE && 
     retval != KRB5_CC_NOTFOUND && retval != KRB5_CC_FORMAT &&
     retval != NO_TICKET ) {    
    mod->kct_err = retval;
  }

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: checking if tickets are cached END\n");

  return (*num_non_cached_services);
}

static int check_if_expired(request_rec *r, krb5_creds v5creds) {

  int retval = NO_ERROR;
  struct timeval cTime;

  gettimeofday(&cTime, (struct timezone *)0);
  if(((cTime.tv_sec - v5creds.times.endtime) >= 0) || 
     (abs(cTime.tv_sec - v5creds.times.endtime) <= 300)) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server, 
		 "c=%ld e=%ld ticket expired", cTime.tv_sec,
	    v5creds.times.endtime);
    retval = NO_TICKET;
  }

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
	       "check_if_expired() retval %d", retval);
  return retval;
}

static int get_tgt(request_rec *r) {

  krb5_get_init_creds_opt opts;
  krb5_principal client = 0;
  krb5_error_code retval = NO_ERROR;
  krb5_context context = 0;
  krb5_keytab keytab = 0;
  krb5_ccache ccache = 0;
  krb5_creds ws_creds;
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  KCT_DIR_CONFIG_REC *dir = kctDirConfig(r);
  krb5_deltat lifetime;
  char *default_realm = NULL;
  char *conf_keytab = NULL;
  char *ws_name = NULL;
  char *tkt_cache_name = NULL;
  char *kdc_fullname = NULL;

  char *fn = NULL;
  char *princ_name = NULL;
  char *tmp = NULL;

#ifdef KCT_CC_LOCKS
  int lockfd = -1;
#endif

  if(dir->kct_tgs_lifetime > 0)
    lifetime = dir->kct_tgs_lifetime;
  else if(srv->kct_tgs_lifetime > 0)
    lifetime = srv->kct_tgs_lifetime;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: default web server tgt lifetime is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_default_realm != NULL)
    default_realm = dir->kct_default_realm;
  else if(srv->kct_default_realm != NULL)
    default_realm = srv->kct_default_realm;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: default realm is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_keytab != NULL)
    conf_keytab = dir->kct_keytab;
  else if(srv->kct_keytab != NULL)
    conf_keytab = srv->kct_keytab;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: web server keytab filename is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_ws_name != NULL)
    ws_name = dir->kct_ws_name;
  else if(srv->kct_ws_name != NULL)
    ws_name = srv->kct_ws_name;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: web server principal name is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_tkt_cache_name != NULL)
    tkt_cache_name = dir->kct_tkt_cache_name;
  else if(srv->kct_tkt_cache_name != NULL)
    tkt_cache_name = srv->kct_tkt_cache_name;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: web server ticket cache filename is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_kdc_fullname != NULL)
    kdc_fullname = dir->kct_kdc_fullname;
  else if(srv->kct_kdc_fullname != NULL)
    kdc_fullname = srv->kct_kdc_fullname;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: kdc fullname is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
	       "mod_kct: getting a tgt");

  memset((char*)&ws_creds, 0, sizeof ws_creds);
  memset((char*)&client, 0, sizeof client);

  krb5_get_init_creds_opt_init(&opts);

  krb5_get_init_creds_opt_set_tkt_life(&opts, lifetime);


  if((retval = krb5_init_context(&context)) != 0) {
    com_err(kct_module_name, retval, "while initializing krb5");
    goto cleanup;
  }

  if((retval = krb5_kt_resolve(context, conf_keytab, &keytab)) != 0) {
    com_err(kct_module_name, retval, "unable to resolve keytable file %s", 
	    conf_keytab);
    goto cleanup;
  }

  tmp = ap_pcalloc(r->pool, strlen(ws_name)+strlen(default_realm)+2);
  sprintf(tmp, "%s@%s", ws_name, default_realm);
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
	       "mod_kct: ws_principal = %s", tmp);
  
  if((retval = krb5_parse_name(context, tmp, &client))){
    com_err(kct_module_name, retval, "unable to parse principal <%s>",
	    ws_name);
    goto cleanup;
  }

  if((retval = krb5_unparse_name(context, client, &princ_name))){
    com_err(kct_module_name, retval, 
	    "unable to unparse a principal made from <%s>", ws_name);
    goto cleanup;
  }

#ifdef KCT_CC_LOCKS
  if ((lockfd = open_and_lock(tkt_cache_name, kct_module_name)) == -1) {
    retval = MODKCT_ERR_INTERNAL_KERBEROS_ERROR;
    goto cleanup;
  }
#endif

  if((retval = krb5_cc_resolve(context, tkt_cache_name, &ccache)) != 0) {
    com_err(kct_module_name, retval, "unable to resolve krb cache <%s>", 
	    tkt_cache_name);
    goto cleanup;
  }

  if((retval = krb5_cc_initialize(context, ccache, client)) != 0) {
    com_err(kct_module_name, retval, "unable to initialize krb cache <%s>", 
	    tkt_cache_name);
    goto cleanup;
  }

  if((retval = krb5_get_init_creds_keytab(context, &ws_creds, client,
      keytab, 0, kdc_fullname, &opts)) != 0) {
    com_err(kct_module_name, retval, 
	    "[%ld] unable to get a TGT for <%s> using keytab <%s>",
	    retval, princ_name, conf_keytab);
    goto cleanup;
  }

  if((retval = krb5_cc_store_cred(context, ccache, &ws_creds)) != 0) {
    com_err(kct_module_name, retval, "unable to store creds in <%s>",
	    tkt_cache_name);
    goto cleanup;
  }

  /* this is a hack but it's needed when the apache is started by root */
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,  
	       "ap_user_id %d", ap_user_id);
  {
      char *fn = tkt_cache_name, *cp;

      if(!(cp = strchr(fn, ':')) || (!strncmp(fn, "FILE:", 5) && 
				     !!(fn = cp+1))) {
	if(chown(fn, ap_user_id, -1 /* no gid change */) == -1) {
	  com_err(kct_module_name, errno, "error doing chown %d <%s>", ap_user_id,
		  fn);
	  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server, 
		       "error doing chown %d <%s>", ap_user_id, fn);
	}
      } else {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
		     "tkt_cache_name = <%s>",  tkt_cache_name);
      }
  }

 cleanup:
  if(princ_name)	free(princ_name);
  if(ccache)   krb5_cc_close(context, ccache);

#ifdef KCT_CC_LOCKS
  if(lockfd != -1) unlock_and_close(lockfd, kct_module_name);
#endif

  if(keytab)   krb5_kt_close(context, keytab);
  if(client)   krb5_free_principal(context, client);
  krb5_free_cred_contents(context, &ws_creds);
  if(context)  krb5_free_context(context);
  if(retval != NO_ERROR && retval != KRB5_FCC_NOFILE && 
     retval != KRB5_CC_NOTFOUND) 
    mod->kct_err = retval;

  return retval;
}

int connect_to_kct(request_rec *r, AUTH_DATA *data) {

  struct hostent *hp;
  struct sockaddr_in sin;
  krb5_error_code retval = 0;
  krb5_error *err_ret;
  krb5_creds in_creds, *out_creds, *tmp_creds;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  KCT_DIR_CONFIG_REC *dir = kctDirConfig(r);

  char *default_realm;
  char *tkt_cache_name;
  char *tgs_service_name;
  char *tgs_server_hostname;
  int  tgs_server_port;
  char *tmp;

  if(dir->kct_default_realm != NULL)
    default_realm = dir->kct_default_realm;
  else if(srv->kct_default_realm != NULL)
    default_realm = srv->kct_default_realm;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: default realm is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_tgs_service_name != NULL)
    tgs_service_name = dir->kct_tgs_service_name;
  else if(srv->kct_tgs_service_name != NULL)
    tgs_service_name = srv->kct_tgs_service_name;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: kct principal name is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_tgs_server_port > 0)
    tgs_server_port = dir->kct_tgs_server_port;
  else if(srv->kct_tgs_server_port > 0)
    tgs_server_port = srv->kct_tgs_server_port;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: kct port is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_tkt_cache_name != NULL)
    tkt_cache_name = dir->kct_tkt_cache_name;
  else if(srv->kct_tkt_cache_name != NULL)
    tkt_cache_name = srv->kct_tkt_cache_name;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: web server ticket cache filename is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_tgs_server_hostname != NULL)
    tgs_server_hostname = dir->kct_tgs_server_hostname;
  else if(srv->kct_tgs_server_hostname != NULL)
    tgs_server_hostname = srv->kct_tgs_server_hostname;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: kct server hostname is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if((retval = krb5_init_context(&data->context)) != 0) {
    com_err(kct_module_name, retval, "while initializing krb5");
    goto cleanup;
  }

  if((retval = krb5_auth_con_init(data->context, &data->auth_context)) != 0) {
    com_err(kct_module_name, retval, "unable to initialize auth_context");
    goto cleanup;
  }

  if((retval = krb5_cc_resolve(data->context, tkt_cache_name, 
			       &data->ccache)) != 0) {
    com_err(kct_module_name, retval, "unable to resolve krb cache <%s>",
	    tkt_cache_name);
    goto cleanup;
  }

  if((retval = krb5_cc_get_principal(data->context, data->ccache, 
				     &data->web_server)) != 0) {
    com_err(kct_module_name, retval, 
	    "while getting client principal name from cache <%s>",
	    tkt_cache_name);
    goto cleanup;
  }
 
  tmp = ap_pcalloc(r->pool, strlen(tgs_service_name)+strlen(default_realm)+2);
  sprintf(tmp, "%s@%s", tgs_service_name, default_realm);
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,  
	       "tgs_principal=%s", tmp); 

  if((retval = krb5_parse_name(data->context, tmp, 
			       &data->tgs_server)) != 0) {
    com_err(kct_module_name, retval, "while parsing servername <%s>",
	    tgs_service_name);
    goto cleanup;
  }

  if ((data->sock = ap_psocket(r->pool, AF_INET, SOCK_STREAM, 0)) < 0)  { 
    ap_log_error(APLOG_MARK, APLOG_ERR, r->server,  "Failed creating socket"); 
    retval = SOCKET_FAILED;
    goto cleanup;
  } else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
                 "mod_kct: connect_to_kct: socket = %d\n", data->sock);
  } 
  
  memset((char *)&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(tgs_server_port);
  
  if((hp = gethostbyname(tgs_server_hostname)) == NULL) {  
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		 "unknown host <%s>", tgs_server_hostname);
    retval = GETHOSTBYNAME_FAILED;
    goto cleanup;
  }
  
  sin.sin_family = hp->h_addrtype;
  memcpy((char *)&sin.sin_addr, (char *)hp->h_addr, hp->h_length);
  
  if(connect(data->sock, (struct sockaddr *) &sin, 
	     sizeof(struct sockaddr_in)) == -1) { 
    ap_log_error(APLOG_MARK, APLOG_ERR, r->server,
		 "Failed connecting socket to %s %d.%d.%d.%d:%d",
		 hp->h_name,
		 (unsigned char)(ntohl(sin.sin_addr.s_addr)>>24),
		 (unsigned char)(ntohl(sin.sin_addr.s_addr)>>16),
		 (unsigned char)(ntohl(sin.sin_addr.s_addr)>>8),
		 (unsigned char)(ntohl(sin.sin_addr.s_addr)),
		 ntohs(sin.sin_port));
    retval = CONNECT_FAILED; 
    goto cleanup;
  } 

  if((retval = krb5_sendauth(data->context, &data->auth_context, 
			     (krb5_pointer)&data->sock, TGS_VERSION, 
			     data->web_server, data->tgs_server, 0, NULL, 0, 
			     data->ccache, &err_ret, NULL, 
			     &data->tgs_creds)) != 0) {
    com_err(kct_module_name, retval, 
	    "while sending client info (sendauth failed) retval = %d", retval);
    if(errno == ECONNRESET)
      retval = MODKCT_ERR_SENDAUTH_FAILED;
    goto cleanup;
  }

  return NO_ERROR;

 cleanup:
  ap_pclosesocket(r->pool, data->sock);
  krb5_cc_close(data->context, data->ccache);
  krb5_free_principal(data->context, data->web_server);
  krb5_free_principal(data->context, data->tgs_server);
  krb5_auth_con_free(data->context, data->auth_context);
  krb5_free_context(data->context);

  return retval;
}

static int send_handshake
(
  AUTH_DATA *data,
  SSL *s,
  SSL_HANDSHAKE *sh,
  request_rec *r
)
{
  unsigned int len, total;
  unsigned char *buf, *p;
  int ilen = sizeof(unsigned long);
  unsigned long header;
  int retval;

  len = sh->c_h_l+sh->s_h_l+
    sh->s_c_l+sh->s_c_r_l+
    sh->s_k_e_l+sh->c_c_l+
    sh->c_k_e_l+sh->c_v_l;

  total = len*sizeof(unsigned char)+9*sizeof(unsigned long);

  p = buf = (unsigned char *) ap_pcalloc(r->pool, total);

  header = htonl(s->version);
  memcpy(p, (char *)&header, ilen);
  p += ilen;

  create_message(r, sh->c_h_l, &p, sh->c_h);
  create_message(r, sh->s_h_l, &p, sh->s_h);
  create_message(r, sh->s_c_l, &p, sh->s_c);
  create_message(r, sh->s_c_r_l, &p, sh->s_c_r);
  create_message(r, sh->s_k_e_l, &p, sh->s_k_e);
  create_message(r, sh->c_c_l, &p, sh->c_c);
  create_message(r, sh->c_k_e_l, &p, sh->c_k_e);
  create_message(r, sh->c_v_l, &p, sh->c_v);

  header = htonl(total);
  if((retval = sendblock(r, data->sock, (char *)&header, 
			 sizeof(header))) < 0) {
    perror("socket closed");
    return COMMUNICATION_ERROR;
  }
  if((retval = sendblock(r, data->sock, buf, total)) < 0) {
    perror("socket closed");
    return COMMUNICATION_ERROR;
  }

  return NO_ERROR;
}

static int send_sessiondata(AUTH_DATA *data, SSL *s, KCT_SERVICES **services, 
			    int num_services, request_rec *r) {

  krb5_error_code retval = 0;
  krb5_data send_data, send_enc_data;
  krb5_address local_kaddr;
  struct sockaddr_in local_addr;
  char *service_name = NULL; //"kca_service/yoga.citi.umich.edu"; //COME BACK
  int addrlen, slen = 0; //strlen(service_name)+1;
  long header;
  char *p;

  int i = 0;
  for(i = 0; i < num_services; i++) {
    if(i) 
    service_name = ap_pstrcat(r->pool, service_name, 
			      services[i]->service_name, ":",
    			services[i]->lifetime, NULL);
    else
      service_name = ap_pstrcat(r->pool,
				services[i]->service_name, ":",
				services[i]->lifetime, NULL);
    if(i+1 != num_services)
      service_name = ap_pstrcat(r->pool, service_name, " ", NULL);
  }

  slen = strlen(service_name) + 1;
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
	       "mod_kct: services %s %d", service_name, slen);

  if((retval = krb5_auth_con_setflags(data->context, data->auth_context,
				      KRB5_AUTH_CONTEXT_DO_SEQUENCE)) != 0) {
    com_err(kct_module_name, retval, "unable to set auth_con flags");
    goto cleanup;
  }

  send_data.length = 3*sizeof(header) + s->session->master_key_length + slen;
  p = send_data.data = (char *)ap_pcalloc(r->pool, send_data.length);

  header = htonl(s->session->master_key_length);
  memcpy(p, (char *)&header, sizeof(header));
  p += sizeof(header);
  memcpy(p, s->session->master_key, s->session->master_key_length);
  p += s->session->master_key_length;
  header = htonl(num_services);
  memcpy(p, (char *)&header, sizeof(header));
  p += sizeof(header);
  header = htonl(slen);
  memcpy(p, (char *)&header, sizeof(header));
  p += sizeof(header);
  memcpy(p, service_name, slen);

  memset((char *)&local_addr, 0, sizeof(local_addr));
  addrlen = sizeof(local_addr);

  if (getsockname(data->sock, (struct sockaddr *)&local_addr, &addrlen) < 0) {
    retval = NETWORK_FAILED;
    goto cleanup;
  }

  local_kaddr.addrtype = ADDRTYPE_IPPORT;
  local_kaddr.length = sizeof(local_addr.sin_port);
  local_kaddr.contents = (krb5_octet *)&local_addr.sin_port;

  if((retval = krb5_auth_con_setports(data->context, data->auth_context,
				      &local_kaddr, NULL)) != 0) {
    com_err(kct_module_name, retval, "in krb5_auth_con_setports");
    goto cleanup;
  }

  local_kaddr.addrtype = ADDRTYPE_INET;
  local_kaddr.length = sizeof(local_addr.sin_addr);
  local_kaddr.contents = (krb5_octet *) &local_addr.sin_addr;

  if((retval = krb5_auth_con_setaddrs(data->context, data->auth_context, 
				      &local_kaddr, NULL)) != 0) {
    com_err(kct_module_name, retval, "in krb5_auth_con_setaddrs");
    goto cleanup;
  }
   
  if((retval = krb5_mk_priv(data->context, data->auth_context, &send_data, 
			    &send_enc_data, NULL)) != 0) {
    com_err(kct_module_name, retval, "while making a KRB_PRIV message");
    goto cleanup;
  }

  header = htonl(send_enc_data.length);
  if((retval = sendblock(r, data->sock, (char *)&header, 
			 sizeof(header))) < 0) {
    perror("socket closed");
    retval = COMMUNICATION_ERROR;
    goto cleanup;
  }
  if((retval = sendblock(r, data->sock, (char *)send_enc_data.data, 
	       send_enc_data.length)) < 0) {
    perror("socket closed");
    retval = COMMUNICATION_ERROR;
    goto cleanup;
  }

  krb5_free_data_contents(data->context, &send_enc_data);

  return NO_ERROR;

 cleanup:
  krb5_cc_close(data->context, data->ccache);
  krb5_free_principal(data->context, data->web_server);
  krb5_free_principal(data->context, data->tgs_server);
  krb5_free_cred_contents(data->context, data->tgs_creds);
  krb5_auth_con_free(data->context, data->auth_context);
  krb5_free_context(data->context);

  return retval;
}

static int receive_service_ticket(AUTH_DATA *data, request_rec *r) {

  struct sockaddr saddr;
  long header = 0;
  unsigned char *recv_buf;
  long error_code = 0;
  krb5_response reply;
  krb5_kdc_rep *dec_rep;
  krb5_error_code retval = 0;
  krb5_enctype enctype;
  krb5_ccache user_cache;
  krb5_principal client = 0;
  char *cache_file_name = NULL;
  char *cache_prefix = NULL;
  char *default_realm = NULL;
  char *service_fqname = NULL;
  char *service_name = NULL;
  char *token = "/";
  char *replace = NULL;
  char *tmp;
  KCT_MOD_CONFIG_REC *mod;
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  KCT_DIR_CONFIG_REC *dir = kctDirConfig(r);
#ifdef KCT_CC_LOCKS
  int lockfd;
#endif

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: receive_ticket begin");

  if(dir->kct_client_prefix != NULL)
    cache_prefix = dir->kct_client_prefix;
  else if(srv->kct_client_prefix != NULL)
    cache_prefix = srv->kct_client_prefix;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: path to the client's tkt cache is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if(dir->kct_default_realm != NULL)
    default_realm = dir->kct_default_realm;
  else if(srv->kct_default_realm != NULL)
    default_realm = srv->kct_default_realm;
  else {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "mod_kct: default realm is not specified");
    return HTTP_INTERNAL_SERVER_ERROR;
  }

  if((retval = recvblock(r, data->sock, (char *)&header, 
			 sizeof(header))) < 0) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "receiving header: socket closed");
    retval = COMMUNICATION_ERROR;
    goto cleanup1;
  }
  header = ntohl(header);
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
	"received header: %d", header);

  recv_buf = (unsigned char *)ap_pcalloc(r->pool, header);
  if((retval = recvblock(r, data->sock, (char *)recv_buf, header)) < 0) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "receiving content: socket closed");
    retval = COMMUNICATION_ERROR;
    goto cleanup1;
  }
  memcpy((char *)&retval, recv_buf, sizeof(long));
  retval = ntohl(retval);

  switch(retval) {
  case NO_ERROR: break;
  default: 
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
                 "mod_kct: error code returned by kct = %d", retval); 
    goto cleanup1;
  }

  reply.response.length = header - sizeof(long);
  reply.response.data = (char *)ap_pcalloc(r->pool, reply.response.length);
  memcpy(reply.response.data, recv_buf+sizeof(long), reply.response.length);

  if((retval = krb5_decode_kdc_rep(data->context, &reply.response,
				   &data->tgs_creds->keyblock, 
				   &dec_rep)) != 0) {
    com_err(kct_module_name, retval, 
	    "error occurred while decrypting kdc reply");
    goto cleanup1;
  }

  if((retval = krb5_kdcrep2creds_local(data->context, dec_rep, 
				       data->tgs_creds->addresses,
				       &data->tgs_creds->second_ticket, 
				       &data->service_creds, r->pool))) {
    com_err(kct_module_name, retval, "while transferring creds");
    goto cleanup1;
  }

  krb5_unparse_name(data->context, data->service_creds->server, 
		    &service_fqname);
  service_name = (char *)ap_palloc(r->pool, strlen(service_fqname)+1);
  sscanf(service_fqname, "%[^@]", service_name);
  if(service_fqname) free(service_fqname);

  if((replace = strpbrk(service_name, token)) != NULL)
	 replace[0] = '_';
  
  cache_file_name = (char *) 
    ap_pcalloc(r->pool, strlen(data->client_id) + strlen(cache_prefix) + 
	       strlen(service_name) + 2);
  sprintf(cache_file_name, "%s%s_%s", cache_prefix, data->client_id, 
	  service_name);
  
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "cache name %s", cache_file_name);

#ifdef KCT_CC_LOCKS
  if ((lockfd = open_and_lock(cache_file_name, kct_module_name)) == -1) {
    retval = MODKCT_ERR_INTERNAL_KERBEROS_ERROR;
    goto cleanup1;
  }
#endif

  /*
   * We've got the cc file locked at this point.  From here
   * on out, we need to free the lock before returning!
   */

  if((retval = krb5_cc_resolve(data->context, cache_file_name, 
			       &user_cache)) != 0) {
    com_err(kct_module_name, retval, "cc_resolve");
    goto cleanup2;
  }

  tmp = ap_pcalloc(r->pool, strlen(data->client_id)+strlen(default_realm)+2);
  sprintf(tmp, "%s@%s", data->client_id, default_realm);
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: client_principal %s", tmp);

  if((retval = krb5_parse_name(data->context, tmp,
                               &client)) != 0) {
    com_err(kct_module_name, retval, "parse_name");
    goto cleanup2;
  }


  if((retval = krb5_cc_initialize(data->context, user_cache, client)) != 0) {
    com_err(kct_module_name, retval, "cc_initialize for: %s", data->client_id);
    krb5_free_principal(data->context, client);
    goto cleanup2;
  }

  if (client) krb5_free_principal(data->context, client);


  if((retval = krb5_cc_store_cred(data->context, user_cache, 
				  data->service_creds)) != 0) {
    com_err(kct_module_name, retval, "cc_store_cred");
    goto cleanup2;
  }

  /* remember cache_file_name for later here. */
  ap_table_setn(r->subprocess_env, "TKT_CC_PREFIX", cache_prefix);

 cleanup2:
  krb5_cc_close(data->context, user_cache);
#ifdef KCT_CC_LOCKS
  unlock_and_close(lockfd, kct_module_name);
#endif

 cleanup1:
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "mod_kct: done with receive_ticket reval = %d", retval);
  return retval;
}

static int
krb5_kdcrep2creds_local(context, pkdcrep, address, psectkt, ppcreds, p)
    krb5_context          context;
    krb5_kdc_rep        * pkdcrep;
    krb5_address ** address;
    krb5_data           * psectkt;
    krb5_creds         ** ppcreds;
    pool *p;
{
    krb5_error_code retval;
    krb5_data *pdata;

    if ((*ppcreds = (krb5_creds *)ap_pcalloc(p, sizeof(krb5_creds))) == NULL) {
        return ENOMEM;
    }

    memset(*ppcreds, 0, sizeof(krb5_creds));

    if ((retval = krb5_copy_principal(context, pkdcrep->client,
                                     &(*ppcreds)->client)))
        goto cleanup;

    if ((retval = krb5_copy_principal(context, pkdcrep->enc_part2->server,
                                     &(*ppcreds)->server)))
        goto cleanup;

    if ((retval = krb5_copy_keyblock_contents(context,
                                              pkdcrep->enc_part2->session,
                                              &(*ppcreds)->keyblock)))
        goto cleanup;

    if ((retval = krb5_copy_data(context, psectkt, &pdata)))
        goto cleanup;
    (*ppcreds)->second_ticket = *pdata;
    free(pdata);

    (*ppcreds)->ticket_flags = pkdcrep->enc_part2->flags;
    (*ppcreds)->times = pkdcrep->enc_part2->times;
    (*ppcreds)->magic = KV5M_CREDS;

    (*ppcreds)->authdata = NULL;                        /* not used */
    (*ppcreds)->is_skey = psectkt->length != 0;

    if (pkdcrep->enc_part2->caddrs) {
        if ((retval = krb5_copy_addresses(context, pkdcrep->enc_part2->caddrs,
                                          &(*ppcreds)->addresses)))
            goto cleanup_keyblock;
    } else {
        /* no addresses in the list means we got what we had */
        if ((retval = krb5_copy_addresses(context, address,
                                          &(*ppcreds)->addresses)))
            goto cleanup_keyblock;
    }

    if ((retval = encode_krb5_ticket(pkdcrep->ticket, &pdata)))
        goto cleanup_keyblock;

    (*ppcreds)->ticket = *pdata;
    free(pdata);
    return NO_ERROR;

cleanup_keyblock:
    krb5_free_keyblock(context, &(*ppcreds)->keyblock);

cleanup:
    free (*ppcreds);
    return retval;
}

static int renegotiate(request_rec *r, SSL *ssl) {
  
  SSL_renegotiate(ssl);
  SSL_do_handshake(ssl);
  if (SSL_get_state(ssl) != SSL_ST_OK) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
		 "Re-negotiation request failed");
    return FORBIDDEN;
  }

#ifdef RD_RENO
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server, 
	       "Awaiting re-negotiation handshake");
#endif

  ssl->state = SSL_ST_ACCEPT;
  SSL_do_handshake(ssl);
  if (SSL_get_state(ssl) != SSL_ST_OK) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
	    "Re-negotiation handshake failed: Not accepted by client!?");
    return FORBIDDEN;
  }

  return NO_ERROR;
}

static void create_message(request_rec *r, unsigned int len, unsigned char **p, 
			   unsigned char *msg) {
  unsigned long header;
  int ilen = sizeof(header);
 
  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, 
		       "sending: %d", len);
 
  header = htonl(len);
  memcpy((*p), (char *)&header, ilen);
  (*p) += ilen;
  if(msg != NULL) memcpy((*p), msg, len);
  (*p) += len;
}

static size_t recvblock(request_rec *r, int sd, unsigned char *buf, 
			size_t total) {

  int len = 0;
  unsigned int recvd = 0;

  while(recvd < total) {
    len = recv(sd, (char *)buf+recvd, total-recvd, 0);
    if(len <= 0) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
                   "mod_kct: recvblock: error while receiving data len = %d"
		   " %s", len, strerror(errno));
      return (-1);
    } 
    recvd+=len;
  }
  return (recvd);
}

static size_t sendblock(request_rec *r, int sd, unsigned char *buf, 
			size_t total) {

  int len;
  unsigned int sent = 0;
  int i;

  while(sent < total) {
    len = send(sd, (char *)buf+sent, total-sent, 0);
    if(len <= 0) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server, 
                   "mod_kct: sendblock: error while sending data");
      return (-1);
    }
    sent+=len;
  }
  return (sent);
}

static void print_buffer(request_rec *r, unsigned char *buffer, size_t len, 
			 unsigned char *comment) {

  int i;
  char *temp, *t2;

  temp = ap_palloc(r->pool, 5 + strlen(comment) + 3*len);
  if (!temp) {
    ap_log_error(APLOG_MARK, APLOG_ERR, r->server, 
                 "%s: can't allocate room to print %d bytes of data",
		 comment, len);
  }

  sprintf(t2 = temp, "%s", comment);
  t2 += strlen(t2);
  for(i = 0; i < len; i++) {
    sprintf(t2, " %02X", buffer[i]);
    t2 += strlen(t2);
    if((i+1) % 22 == 0 && i != 0)
    {
	ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, temp);
	t2 = temp;
    }
  }
  if(t2 != temp)
     ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server, temp);
  free(temp);
}

#ifdef DEBUG_SSL_ID
static void print_ssl_info(SSL *s, server_rec *server, char *msg) {
  int i;

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, server,
	       "mod_kct: %s: ssl id = ", msg);
  for(i = 0; i < s->session->session_id_length; i++) {
    fprintf(stderr, "%02x", s->session->session_id[i]);
  }
  fprintf(stderr, "\n");
}
#endif

#ifdef PRINT_HANDSHAKE
static void print_handshake(SSL_HANDSHAKE *sh) {
    int i;
    FILE *f = fopen("/tmp/ssl.log", "w");

    setbuf(f, NULL);	/* XXX */

    if(sh->c_h != NULL) {
      fprintf(f, "      READ: CLIENT HELLO: %d\n      ", sh->c_h_l);
      for(i = 0; i < sh->c_h_l; i++) {
	fprintf(f, "%02x ", sh->c_h[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->s_h != NULL) {
      fprintf(f, "      WRITE: SERVER HELLO: %d\n      ", sh->s_h_l);
      for(i = 0; i < sh->s_h_l; i++) {
	fprintf(f, "%02x ", sh->s_h[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->s_c != NULL) {
      fprintf(f, "      WRITE: SERVER CERTIFICATE: %d\n      ", 
	      sh->s_c_l);
      for(i = 0; i < sh->s_c_l; i++) {
	fprintf(f, "%02x ", sh->s_c[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->s_k_e != NULL) {
      fprintf(f, "      WRITE: SERVER KEY EXCHANGE: %d\n      ", 
	      sh->s_k_e_l);
      for(i = 0; i < sh->s_k_e_l; i++) {
	fprintf(f, "%02x ", sh->s_k_e[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->s_c_r != NULL) {
      fprintf(f, "      WRITE: SERVER CERTIFICATE REQUEST: %d\n      ", 
	      sh->s_c_r_l );
      for(i = 0; i < sh->s_c_r_l; i++) {
	fprintf(f, "%02x ", sh->s_c_r[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->c_c != NULL) {
      fprintf(f, "      READ: CLIENT CERTIFICATE: %d\n      ", 
	      sh->c_c_l );
      for(i = 0; i < sh->c_c_l; i++) {
	fprintf(f, "%02x ", sh->c_c[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->c_k_e != NULL) {
      fprintf(f, "      READ: CLIENT KEY EXCHANGE: %d\n      ", 
	      sh->c_k_e_l);
      for(i = 0; i < sh->c_k_e_l; i++) {
	fprintf(f, "%02x ", sh->c_k_e[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

    if(sh->c_v != NULL) {
      fprintf(f, "      READ: CLIENT VERIFY: %d\n      ",
	      sh->c_v_l);
      for(i = 0; i < sh->c_v_l; i++) {
	fprintf(f, "%02x ", sh->c_v[i]);
	if((i+1) % 22 == 0 && i != 0) fprintf(f, "\n      ");
      }
      fprintf(f, "\n");
    }

Done:
    fflush(f);
    fclose(f);
}
#endif


/*
 * This callback function is executed once for each of the messages
 * in an SSL handshake.
 */

void kct_ssl_callback
(
    int			write_p,
    int			version,
    int			content_type,
    const void		*buf,
    size_t		len,
    SSL			*ssl,
    void		*arg
)
{
    conn_rec		*c = NULL;
    server_rec		*s = NULL;
    module		*m = NULL;
    SSLSrvConfigRec 	*sc = NULL;
    char		*str = NULL;
    const unsigned char	*b = (const unsigned char *)buf;
    SSL_HANDSHAKE	*sh = NULL;


    /*
     * only interested in HANDSHAKE messages
     */
    if (content_type != SSL3_RT_HANDSHAKE)
	return;

    /*
     * Not interested in messages during renegotiation nor after handshake
     */
    if (ssl->s3->renegotiate || !ssl->in_handshake)
	return;

    /*
     * find corresponding server
     */
    if ((c = (conn_rec *)SSL_get_app_data(ssl)) == NULL)
        return;
    s = c->server;
    m = (module *)ap_find_linked_module("mod_ssl.c");
    sc = (SSLSrvConfigRec *)ap_get_module_config(s->module_config, m);
    if (sc == NULL)
        return;

    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"ssl_callback_Msg: arg=0x%08X", arg);

    /*
     * ensure this connection has a SSL_HANDSHAKE allocated from the conn pool
     * and that it is set as the msg_callback_arg
     */
    if (arg == NULL)
    {
	sh = ap_pcalloc(c->pool, sizeof(SSL_HANDSHAKE));
	if (sh == NULL)
	{
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, s,
            "mod_kct: ssl_callback_Msg: ap_pcalloc of SSL_HANDSHAKE failed!");
	    return;
	}
	initialize_handshake(sh);
	if (!SSL_set_msg_callback_arg(ssl, sh))
	{
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, s,
            "mod_kct: ssl_callback_Msg: SSL_set_msg_callback_arg of SSL_HANDSHAKE failed!");
	    return;
	}
    }
    else
	sh = arg;

    /*
     * copy the contents of the handshake messages to the SSL_HANDSHAKE struct
     */
    switch (b[0])
    {
	case  1: /* CLIENT HELLO */
	    if (sh->c_h_l == 0)
		if ( (sh->c_h = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of c_h failed!");
		    return;
		}
		else
		{
		    memcpy(sh->c_h, b, len);
		    sh->c_h_l = len;
		}
	    break;
	case  2: /* SERVER HELLO */
	    if (sh->s_h_l == 0)
		if ( (sh->s_h = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of s_h failed!");
		    return;
		}
		else
		{
		    memcpy(sh->s_h, b, len);
		    sh->s_h_l = len;
		}
	    break;
	case 11: /* CLIENT|SERVER CERTIFICATE */
	    if (write_p)
	    {
		if (sh->s_c_l == 0)
		    if ( (sh->s_c = ap_pcalloc(c->pool, len)) == NULL)
		    {
			ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        		     "ssl_callback_Msg: ap_pcalloc of s_c failed!");
			return;
		    }
		    else
		    {
			memcpy(sh->s_c, b, len);
			sh->s_c_l = len;
		    }
	    }
	    else
	    {
		if (sh->c_c_l == 0)
		    if ( (sh->c_c = ap_pcalloc(c->pool, len)) == NULL)
		    {
			ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        		    "ssl_callback_Msg: ap_pcalloc of c_c failed!");
			return;
		    }
		    else
		    {
			memcpy(sh->c_c, b, len);
			sh->c_c_l = len;
		    }
	    }
	    break;
	case  12: /* SERVER KEY EXCHANGE */
	    if (sh->s_k_e_l == 0)
		if ( (sh->s_k_e = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of s_k_e failed!");
		    return;
		}
		else
		{
		    memcpy(sh->s_k_e, b, len);
		    sh->s_k_e_l = len;
		}
	    break;
	case 13: /* SERVER CERTIFICATE REQUEST */
	    if (sh->s_c_r_l == 0)
		if ( (sh->s_c_r = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of s_c_r failed!");
		    return;
		}
		else
		{
		    memcpy(sh->s_c_r, b, len);
		    sh->s_c_r_l = len;
		}
	    break;
	case 15: /* CLIENT VERIFY */
	    if (sh->c_v_l == 0)
		if ( (sh->c_v = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of c_v failed!");
		    return;
		}
		else
		{
		    memcpy(sh->c_v, b, len);
		    sh->c_v_l = len;
		}
	    break;
	case 16: /* CLIENT KEY EXCHANGE */
	    if (sh->c_k_e_l == 0)
		if ( (sh->c_k_e = ap_pcalloc(c->pool, len)) == NULL)
		{
		    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
        	        "ssl_callback_Msg: ap_pcalloc of c_k_e failed!");
		    return;
		}
		else
		{
		    memcpy(sh->c_k_e, b, len);
		    sh->c_k_e_l = len;
		}
	    break;
        case 0:  /* HELLO REQUEST */
        case 14: /* DONE */
        case 20: /* FINISHED */
            break;
	default:
	    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
                "ssl_callback_Msg: UNKNOWN MSG: 0x%02X", b[0]);
	    break;
    }
}

void kct_hook_NewConnection(conn_rec *conn) {
  SSL *s = ap_ctx_get(conn->client->ctx, "ssl");
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(conn->server);

  if(s == NULL && srv->bEnabled == 1) {    
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, conn->server,
		 "mod_kct: SSL context is NULL");
    mod->kct_err = ERR_HTTPS_EXPECTED;
    return;
  }

  if((mod->ssl_mod = ap_find_linked_module("mod_ssl.c")) == NULL) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, conn->server,
                 "mod_kct: ssl module is not linked in");
    mod->kct_err = ERR_HTTPS_EXPECTED;
    return;
  }

  if(check_if_verify_client_enabled(conn->server, NULL) != OK) {
    return;
  }

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, conn->server,
	       "kct_hook_NewConnection");

#ifdef DEBUG_SSL_ID
  print_ssl_info(s, conn->server, "kct_hook_NewConnection");
#endif

  // Cause handshake-transcription to happen for this connection
  
  s->msg_callback = kct_ssl_callback;
  mod->nPostRead  = 0;
  mod->nTranslate = 0;
  mod->nFixups    = 0; 
  mod->kct_err    = NO_ERROR;

}

int kct_post_read(request_rec *r) {
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  SSLSrvConfigRec *ssl_srv;
  int retval = NO_ERROR;

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "(%d) kct_post_read: %s uri=%s", mod->nPostRead, 
	       r->method, r->uri);

  mod->nPostRead++;
  if(mod->nPostRead > 1) return DECLINED;

  ap_table_setn(r->subprocess_env, "AFSFILEACCESS", "error");

  if(check_if_verify_client_enabled(r->server, NULL) != OK) {
    return DECLINED;
  }

  if(check_if_ssl_enabled(r->connection) != OK) {
    return DECLINED;
  }

  ssl_srv = (SSLSrvConfigRec *)ap_get_module_config(r->server->module_config, 
						    mod->ssl_mod);
  
  if(ssl_srv->nVerifyClient == SSL_CVERIFY_REQUIRE) {
    if(!srv->num_services) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		   "mod_kct: kct_post_read kct service name was not supplied");
      return HTTP_INTERNAL_SERVER_ERROR;
    }

    retval = do_kct(r, srv->kct_services, srv->num_services, srv->bCaching); 
    if(mod->kct_err == NO_ERROR) 
      ap_table_setn(r->subprocess_env, "AFSFILEACCESS", "on");

    return retval;
  }
  else {
    ap_table_setn(r->subprocess_env, "AFSFILEACCESS", "off");
    return DECLINED;
  }
}

static int kct_translate(request_rec *r) {
  KCT_MOD_CONFIG_REC *mod = kctModConfig();

  if(mod->nTranslate < 2)
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		 "(%d) kct_translate: %s uri=%s", mod->nTranslate, 
		 r->method, r->uri);

  mod->nTranslate++;
  if(mod->nTranslate > 1) return DECLINED;

  if(check_if_verify_client_enabled(r->server, NULL) != OK) {
    return DECLINED;
  }

  if(check_if_ssl_enabled(r->connection) != OK) {
    return DECLINED;
  }

  if(mod->kct_err != NO_ERROR) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		 "mod_kct: translate: error");
    return OK;
  }
  else 
    return DECLINED;

}

int kct_fixups(request_rec *r) {
  int retval = NO_ERROR;
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(r->server);
  KCT_DIR_CONFIG_REC *dir = kctDirConfig(r);
  SSLSrvConfigRec *ssl_srv;

  if(mod->nFixups < 2)
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		 "(%d) kct_fixups: %s uri=%s", mod->nFixups, 
		 r->method, r->uri);

  mod->nFixups++;
  if(mod->nFixups > 1) return DECLINED;

  if(check_if_verify_client_enabled(r->server, dir) != OK) {
    return DECLINED;
  }

  if(check_if_ssl_enabled(r->connection) != OK) {
    return DECLINED;
  }

  ssl_srv = (SSLSrvConfigRec *)ap_get_module_config(r->server->module_config, 
						    mod->ssl_mod);

  if(ssl_srv->nVerifyClient == SSL_CVERIFY_REQUIRE) {
    if(mod->kct_err != NO_ERROR) {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
		   "mod_kct: fixups: error");
      return OK;
    }
    else
      return DECLINED;
  }
  else {
    //COME BACK NEED TO ITERATE THROUGH THE LIST OF SERVICES
    if(srv->num_services) 
      retval = do_kct(r, srv->kct_services, srv->num_services, srv->bCaching); 
    else if(dir->kct_services)
      retval = do_kct(r, dir->kct_services, dir->num_services, srv->bCaching); 
    else {
      ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r->server,
		   "mod_kct: kct_post_read kct service name was not supplied");
      return HTTP_INTERNAL_SERVER_ERROR;
    }
    if(mod->kct_err != NO_ERROR) 
      ap_table_setn(r->subprocess_env, "AFSFILEACCESS", "error");
    return retval;
  }
}

static int kct_handler(request_rec *r) {
  KCT_MOD_CONFIG_REC *mod = kctModConfig();

  ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
	       "kct_handler: %s uri=%s", r->method, r->uri);

  if(check_if_verify_client_enabled(r->server, NULL) != OK) {
    return DECLINED;
  }

  if(check_if_ssl_enabled(r->connection) != OK) {
    return DECLINED;
  }

  mod->nPostRead = 0;
  mod->nTranslate = 0;
  mod->nFixups = 0;

  if(mod->kct_err != NO_ERROR) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, r->server,
      "mod_kct: handler: printing error %d", mod->kct_err);
    send_error(r, mod->kct_err);
    return OK;
  }
  else {
    return DECLINED;
  }


}

static int check_if_ssl_enabled(conn_rec *conn) {
  SSL *s = NULL;

  if((s = ap_ctx_get(conn->client->ctx, "ssl")) == NULL) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, conn->server,
		 "mod_kct: SSL context is NULL");
    return DECLINED;
  }
  return OK;
}

static int check_if_verify_client_enabled(server_rec *s, 
                                          KCT_DIR_CONFIG_REC *dir) {
  KCT_MOD_CONFIG_REC *mod = kctModConfig();
  KCT_SRV_CONFIG_REC *srv = kctSrvConfig(s);
  SSLSrvConfigRec *ssl_srv = 
  (SSLSrvConfigRec *)ap_get_module_config(s->module_config, mod->ssl_mod);

  if(ssl_srv == NULL || srv == NULL || srv->bEnabled < 1 || 
    ((ssl_srv->nVerifyClient == SSL_CVERIFY_REQUIRE && 
      srv->bVerifyClient < 1) ||
     (srv->bVerifyClient < 1 && dir != NULL && dir->bVerifyClient < 1))) {
    ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_INFO, s,
		 "mod_kct: kct is not enabled for this server");
    return DECLINED;
  }

  return OK;
}

static void send_error(request_rec *r, int retval) {
  KCT_MOD_CONFIG_REC *mod = kctModConfig();

  mod->kct_err = NO_ERROR;
  r->content_type = "text/html";
  ap_rprintf(r, "<HTML><HEAD><TITLE>mod_kct: error</TITLE></HEAD>"
	     "<BODY BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\">"
	     "<H3>mod_kct error:</H3>\n"
	     "Your request for '%s' returned: <P><B>", r->uri);

  switch(retval) {
  case CONNECT_FAILED:
    ap_rputs("Server failed to establish connection with KCT", r);
    break;
  case MODKCT_ERR_SENDAUTH_FAILED:
    ap_rputs("Server failed to establish secure connection with KCT", r);
    break;
  case KCT_ERR_INTERNAL_ERROR:
    ap_rputs("Miscellaneous internal error",r);
    break;
  case KCT_ERR_RD_PRIV:            
    ap_rputs("Unable to decrypt private message",r);
    break;
  case KCT_ERR_INVALID_HANDSHAKE: 
    ap_rputs("Invalid handshake",r);
    break;      
  case KCT_ERR_OLD_HANDSHAKE:
    ap_rputs("Time skew in handshake",r);
    break;           
  case KCT_ERR_INVALID_ID:
    ap_rputs("Invalid ID found in handshake",r);
    break;              
  case KCT_ERR_REALM_MISMATCH: 
    ap_rputs("Realm mismatch",r);
    break;         
  case KCT_ERR_COMMUNICATION_ERROR:
    ap_rputs("Communications error",r);
    break;     
  case KCT_ERR_UNAUTHORIZED: 
    ap_rputs("Unauthorized web server request",r);
    break;           
  case MODKCT_ERR_INTERNAL_KERBEROS_ERROR:
    ap_rputs("Web Server-Internal Kerberos-related error",r);
    break; 
  case KCT_ERR_INTERNAL_KERBEROS_ERROR:
    ap_rputs("KCT-Internal Kerberos-related error",r);
    break; 
  case KCT_ERR_BAD_PRINCIPAL:
  case KCT_ERR_UNKNOWN_PRINCIPAL:
    ap_rputs("Requested service principal does not exist",r);
    break; 
  case ERR_NO_CLIENT_CERTIFICATE:
    ap_rputs("No client certificate", r);
    break; 
  case ERR_BAD_IDENTIFIER:
    ap_rputs("No or bad Kerberos identifier in the certificate",r);
    break;  
  case ERR_SSL_RENEG_FAILED:
    ap_rputs("SSL Renegotiation failed", r);
    break;  
  case ERR_GET_TGT_FAILED:
    ap_rputs("A Web Server was not able to get a TGT",r);
    break;  
  case COMMUNICATION_ERROR:
     ap_rputs("Connection problems",r);
     break;
  case ERR_HTTPS_EXPECTED:
    ap_rputs("Expected an HTTPS connection",r);
    break;
  default:
    ap_rprintf(r, "Unknown error %d", retval);
  }
  ap_rputs("</B></BODY></HTML>\n\n", r);
}

static void initialize_handshake(SSL_HANDSHAKE *sh) {
        sh->c_h = NULL;
        sh->c_h_l = 0;
        sh->s_h = NULL;
        sh->s_h_l = 0;
        sh->s_c = NULL;
        sh->s_c_l = 0;
        sh->s_c_r = NULL;
        sh->s_c_r_l = 0;
        sh->s_k_e = NULL;
        sh->s_k_e_l = 0;
        sh->c_c = NULL;
        sh->c_c_l = 0;
        sh->c_k_e = NULL;
        sh->c_k_e_l = 0;
        sh->c_v = NULL;
        sh->c_v_l = 0;
}

static handler_rec kct_handlers[] = {
  {DIR_MAGIC_TYPE, kct_handler},
  {CGI_MAGIC_TYPE, kct_handler},
  {INCLUDES_MAGIC_TYPE, kct_handler},
  {INCLUDES_MAGIC_TYPE3, kct_handler},
  {MAP_FILE_MAGIC_TYPE, kct_handler},
  {STATUS_MAGIC_TYPE, kct_handler},
  {ASIS_MAGIC_TYPE, kct_handler},
  {"*/*", kct_handler},
  {"application/x-jakarta-servlet", kct_handler},
  {"jakarta-servlet", kct_handler},
  {"application/x-httpd-php", kct_handler},
  {NULL}
};

module MODULE_VAR_EXPORT kct_module = {
  STANDARD_MODULE_STUFF,
  kct_init,			/* initializer */
  kct_config_perdir_create,	/* create per-dir config */
  kct_config_perdir_merge,	/* merge per-dir config */
  kct_config_perserver_create,	/* create per-server config */
  kct_config_perserver_merge,	/* merge per-server config */
  kct_cmds,			/* command table */
  kct_handlers,			/* [#8] handlers */
  kct_translate,		/* [#1] file name translation */
  NULL,				/* [#4] check user id */
  NULL,				/* [#5] check auth */
  NULL,				/* [#3] check access */
  NULL,				/* [#6] type checker */ 
  kct_fixups,			/* [#7] prerun fixups */ 
  NULL,				/* [#9] logger */
  NULL,				/* [#2] header parser */
  NULL,				/* child init */
  NULL,				/* child exit */
  kct_post_read			/* [#0] post read request */
#ifdef EAPI
  ,NULL,			/* EAPI: add_module */
  NULL,				/* EAPI: remove_module */
  NULL,				/* EAPI: rewrite_command */
  kct_hook_NewConnection	/* EAPI: new_connection */
#endif
};
