/*
 * Copyright    2000
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of the university of
 * michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software
 * without specific, written prior authorization.  if the
 * above copyright notice or any other identification of the
 * university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * this software is provided as is, without representation
 * from the university of michigan as to its fitness for any
 * purpose, and without warranty by the university of
 * michigan of any kind, either express or implied, including
 * without limitation the implied warranties of
 * merchantability and fitness for a particular purpose. the
 * regents of the university of michigan shall not be liable
 * for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising
 * out of or in connection with the use of the software, even
 * if it has been or is hereafter advised of the possibility of
 * such damages.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#ifndef WIN32
#ifdef macintosh
#include <Sockets.h>
#include <Kerberos/Kerberos.h>		// Needed on Mac for Kerberos Framework
#else /* !macintosh */
# include <sys/time.h>
# include <sys/socket.h>
# include <netinet/in.h>
#endif /* macintosh */
#endif

#ifdef WIN32
# define WSHELPER
#endif /* WIN32 */

#ifdef WSHELPER
# include <wshelper.h>
#else /* !WSHELPER */
# include <arpa/inet.h>
# include <arpa/nameser.h>
# include <resolv.h>
#endif /* !WSHELPER */

#ifdef HAVE_SYS_FILIO_H
# include <sys/filio.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#ifdef HAVE_SYS_SELECT_H
# ifndef FD_SET
#  include <sys/select.h>
# endif
#endif

#ifndef WIN32
# include <netdb.h>
#endif

#include <memory.h>

#ifdef WIN32
# define __WINCRYPT_H__		// PREVENT windows.h from including wincrypt.h
				// since wincrypt.h and openssl namepsaces collide
				//  ex. X509_NAME is #define'd and typedef'd ...
# include <winsock.h>		// Must be included before <windows.h> !!!
# include <windows.h>
# include <openssl/pem.h>
#endif  /* WIN32 */


#include <stdlib.h>
#include <openssl/x509v3.h>

#ifdef USE_KRB5
# include <krb5.h>
# include <com_err.h>
#else /* !USE_KRB5 */
# ifndef WIN32
#  ifndef linux
#   define DES_DEFS		// Prevent collision with K5 DES delarations
#  endif /* !linux */
#  ifdef macintosh
#   include <KClient.h>
#  else /* !macintosh */
#   include "des-openssl-hack.h"
#   include <krb.h>
#  endif /* macintosh */
# else /* !WIN32 */
#  include "des-openssl-hack.h"
#  include <krb.h>
# endif /* !WIN32 */
#endif /* USE_KRB5 */

#include "msg.h"
#include "udp_nb.h"
#include "kx509.h"
#include "doauth.h"
#include "debug.h"

#include <openssl/crypto.h>
#include <openssl/err.h>
#include <openssl/asn1_mac.h>
#include "kx509_asn.h"
#include <openssl/rand.h>

char version_2_0_string[4]={0,0,2,0};

#define	MAX_MSG_LEN	2048
#define RECV_TIMEOUT	5
#define SEND_TIMEOUT	5
#define MAX_KCA_HOSTS	16

#if defined(WIN32) && !defined(USE_KRB5)
/* I don't know if WIN32 defines this or not, but if not, here it is... */
# ifndef MAX_KTXT_LEN
#  define MAX_KTXT_LEN 1250
# endif
# ifndef ANAME_SZ
#  define ANAME_SZ	40
# endif
# ifndef REALM_SZ
#  define REALM_SZ	40
# endif
# ifndef SNAME_SZ
#  define SNAME_SZ	40
# endif
# ifndef INST_SZ
#  define INST_SZ	40
# endif
# ifndef KSUCCESS
#  define KSUCCESS	0
# endif
#endif	/* WIN32 && !USE_KRB5 */


#ifdef DEBUG
void print_response(KX509_RESPONSE *);
void print_request(KX509_REQUEST *);
#endif

extern int debugPrint;	/* XXX TEMPORARY TAKE THIS OUT */

void print_response(KX509_RESPONSE *client_response);
void fill_in_octet_string( ASN1_OCTET_STRING *osp, char *st, int len);

#ifdef WIN32
 void clean_cert_store(char *realm);
# ifdef USE_KRB5
  int get_krb5_realm(krb5_context k5_context, char *realm, 
		     char *tkt_cache_name,char **err_msg);
# else
  int get_krb4_realm(char *realm, char **err_msg);
#endif /* USE_KRB5 */
// int /*PASCAL*/ krb_mk_req(KTEXT,char *,char *,char *,long);
#endif /* WIN32 */
int get_kca_list(char *base_realm, char ***dns_hostlist);

#define KRBCHK_PORT     (u_short)9878
#define BUF_LEN		2048


#if defined(USE_KRB5)
#define CA_SERVICE	"kca_service"
#else
#define CA_PRINC	"cert"
#define CA_INST		"x509"
#endif

#if defined(USE_KRB5)
char ca_service[256] = CA_SERVICE;
#else
char ca_princ[ANAME_SZ] = CA_PRINC, ca_inst[INST_SZ] = CA_INST;
#endif

#define	DEFBITS	512 /* first get MS stuff working, then do 1024 */

/* Make "buffer" static since it's sometimes used for returned error messages */
#if defined(USE_KRB5)
static char buffer[2048];
#else
static char buffer[MAX_KTXT_LEN+1];
#endif

#if defined(USE_KRB5)

/*
 *=========================================================================*
 *
 * get_cert_authent_K5()
 *
 *=========================================================================*
 */
int
get_cert_authent_K5(krb5_context k5_context,
		    char *ca_hostname,
		    krb5_data *k5_authent,
		    char sess_key_result[],
		    int *sess_len_ptr,
		    char *realm,
		    char *tkt_cache_name,
		    char **err_ptr)
{
#ifndef USE_MSK5
	krb5_auth_context k5_auth_context;
	krb5_ccache cc;
	krb5_principal me;
	krb5_creds mcreds, outcreds;
	krb5_error_code result;
	char **realms_of_host;

	/* Without this, I get a bus error within krb5_mk_req() in krb5_copy_keyblock() */
	memset(&k5_auth_context, '\0', sizeof(k5_auth_context));

	/* DETERMINE USER'S PRINCIPAL NAME FROM TICKET FILE */

	if(tkt_cache_name) 
	{
	  if (result = krb5_cc_resolve(k5_context, tkt_cache_name, &cc))
	  {
		msg_printf("get_cert_authent_K5: krb5_cc_resolve: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Unable to determine default credentials cache.";
		return KX509_STATUS_CLNT_FIX;
	  }
	}
        else
	{
	  if (result = krb5_cc_default(k5_context, &cc))
	  {
		msg_printf("get_cert_authent_K5: krb5_cc_default: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Unable to determine default credentials cache.";
		return KX509_STATUS_CLNT_FIX;
	  }
	}

	if (result = krb5_cc_get_principal(k5_context, cc, &me))
	{
		msg_printf("get_cert_authent_K5: krb5_cc_get_principal: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Unable to determine principal from credentials cache.";
		return KX509_STATUS_CLNT_FIX;
	}

	/* GENERATE KRB5 AUTHENTICATOR FOR CA SERVER */

	if (result = krb5_mk_req(k5_context, &k5_auth_context, AP_OPTS_MUTUAL_REQUIRED,
			ca_service, ca_hostname, 0L, cc, k5_authent))

	{
		msg_printf("get_cert_authent_K5: krb5_mk_req: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Unable to use your tickets to build the necessary authenticator.";
		return KX509_STATUS_CLNT_FIX;
	}

	/* EXTRACT THE SESSION KEY FROM THE RESULTING CREDENTIALS (TICKET) */

	memset(&mcreds, 0, sizeof(mcreds));
	memset(&outcreds, 0, sizeof(outcreds));
	mcreds.client = me;
	if (result = krb5_get_host_realm(k5_context, ca_hostname, &realms_of_host))
	{
		msg_printf("get_cert_authent_K5: krb5_get_host_realm: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Error determining realm of KCA host.";
		return KX509_STATUS_CLNT_FIX;
	}

	if (realms_of_host[0] == NULL)
	{
		msg_printf("get_cert_authent_K5: krb5_get_host_realm returned empty list\n");
		*err_ptr = "Check your Kerberos 5 configuration.  "
			"Could not determine the realm of the KCA host.";
		return KX509_STATUS_CLNT_FIX;
	}

	if (result = krb5_build_principal_ext(k5_context, &mcreds.server,
				strlen(realms_of_host[0]),
				realms_of_host[0],
				strlen(ca_service), ca_service,
				strlen(ca_hostname), ca_hostname,
				0))
	{
		msg_printf("get_cert_authent_K5: krb5_build_principal_ext: %s\n",
			error_message(result));
		*err_ptr = "Try re-authenticating(K5).  "
			"Error while creating principal from retrieved principal.";
		krb5_free_host_realm(k5_context, realms_of_host);
		return KX509_STATUS_CLNT_FIX;
	}

	krb5_free_host_realm(k5_context, realms_of_host);

	if (result = krb5_cc_retrieve_cred(k5_context, cc, 0, &mcreds, &outcreds))
	{
		msg_printf("get_cert_authent_K5: krb5_cc_retrieve_cred: %s\n",
			error_message(result));
		*err_ptr, "Try re-authenticating(K5).  "
			"Unable to find correct credentials to get session key.";
		return KX509_STATUS_CLNT_FIX;
	}

	/* Verify caller can hold session key, and return it */
	if (*sess_len_ptr < outcreds.keyblock.length)
	{
		*err_ptr = "get_cert_authent_K5: Internal error; not enough room to hold session key.";
		krb5_free_creds(k5_context, &outcreds);
		return KX509_STATUS_CLNT_FIX;
	}
	*sess_len_ptr = outcreds.keyblock.length;
	memcpy(sess_key_result, outcreds.keyblock.contents, outcreds.keyblock.length);

	krb5_free_principal(k5_context, me);
	krb5_free_cred_contents(k5_context, &outcreds);
	krb5_cc_close(k5_context, cc);
	return 0;
#else /* USE_MSK5 */

	/* Place to receive the MS K5 authenticator data */
	static	char	authent_dat[1024];


	k5_authent->data = &authent_dat[0];
	return !MSK5_get_authent_and_sesskey(CA_SERVICE, ca_hostname, realm,
						&k5_authent->data[0],
						&k5_authent->length,
						sess_key_result,
						sess_len_ptr);

#endif /* USE_MSK5 */
}

#else	/* !USE_KRB5 */

/*
 *=========================================================================*
 *
 * get_cert_authent()
 *
 *=========================================================================*
 */
int
get_cert_authent(KTEXT authent,
		 char sess_key_result[],
		 int *sess_len,
		 char *realm,
		 char **err_ptr)
{
#ifdef WIN32
	char		guard_rail_1[2048];	/* BILLDO */
	CREDENTIALS	cr;
	char		guard_rail_2[2048];	/* BILLDO */
# else /* !WIN32 */
	char		dummy[MAX_K_NAME_SZ+1];
#  ifndef macintosh
	CREDENTIALS	cr;
	char		*sess_key;
#  endif /* macintosh */
# endif /* WIN32 */

#ifdef macintosh
	KClientSessionInfo session;
	int		err;
	static KClientKey sessionKey;
#endif /* macintosh */


#ifdef WIN32
	memset(guard_rail_1, 0, 2048);	/* BILLDO */
	memset(guard_rail_2, 0, 2048);	/* BILLDO */
#endif /* WIN32 */

	if (*sess_len < 8)
	{
		*err_ptr = "Internal error; bad *sess_len";
		return KX509_STATUS_CLNT_FIX;
	}
	*sess_len = 8;

	/* GENERATE KRB4 AUTHENTICATOR FOR CA SERVER */

#if defined(macintosh)
	if ((err = KClientNewSession(&session, 0, 0, 0, 0)) != noErr)
	{
		*err_ptr = "Nope, you needed to pass KClientNewSession something else.";
		return KX509_STATUS_CLNT_FIX;
	}
	sprintf(dummy, "%s.%s@%s", ca_princ, ca_inst, realm);
	authent->length = sizeof(authent->dat);
	if ((err = KClientGetTicketForService(&session, dummy, authent,
					(unsigned long *)&authent->length)) != noErr)
	{
		*err_ptr = "Try re-authenticating.  "
			"Unable to use your tickets to build the necessary authenticator.";
		KClientDisposeSession(&session);
		return KX509_STATUS_CLNT_FIX;
	}
	if ((err = KClientGetSessionKey(&session, &sessionKey)) != noErr) {
		*err_ptr = "Try re-authenticating.  "
			"Unable to use your tickets to get session key.";
		KClientDisposeSession(&session);
		return KX509_STATUS_CLNT_FIX;
	}
	/* XXX need to get the session key */
	memcpy(sess_key_result, &sessionKey, 8);
	KClientDisposeSession(&session);
#else   /* MIT V4 on Unix or WIN32 */
	if (krb_mk_req(authent, ca_princ, ca_inst, realm, 0L))
	{
		*err_ptr = "Try re-authenticating.  "
			"Unable to use your tickets to build the necessary authenticator.";
		return KX509_STATUS_CLNT_FIX;
	}
	if (krb_get_cred(ca_princ, ca_inst, realm, &cr))
	{
		*err_ptr = "Try re-authenticating.  "
			"Unable to use your tickets to get session key.";
		return KX509_STATUS_CLNT_FIX;
	}
	memcpy(sess_key_result, cr.session, 8);
#endif  /* MIT V4 on Unix or WIN32 */
	return 0;
}

#endif	/* USE_KRB5 */

/*
 *=========================================================================*
 *
 * try_ca()
 *
 * Request a certificate from a particular KCA.
 * If we haven't already generated a key-pair, do that now.
 * If using K5, we need a different authenticator for each
 * CA we contact.  If using K4, then we can use the same one
 * for each CA.  We use the session key to seed the generation
 * of the key-pair.
 *
 *=========================================================================*
 */
int try_ca(
#if defined(USE_KRB5)
		krb5_context k5_context,
#endif
		int		socket,				/* IN Socket to be used to communicate with CA */
		char	*ca_hostname,		/* IN Host name of the the CA to try */
		char 	*realm,				/* IN Realm name */
		RSA		**rsa,				/* IN/OUT key-pair information */
		X509	**certp,			/* OUT certificate information */
		int (*verify_recvd_packet)(),/*IN routine to call to verify the CA response */
		void	*arg,				/* IN Arguments passed to verification routine */
		char	sess_key[],			/* IN/OUT session key holder */
		int		*sess_len_ptr,		/* IN/OUT length of session key */
#if defined(USE_KRB5)
		char    *tkt_cache_name,		/* IN credential cache file name */
#endif
		char	**emsg,				/* IN/OUT error string buffer */
		int		*err_num_ptr		/* OUT Error value recipient */
)
{
	int				keybits=DEFBITS;	/* Number of bits in the public key / private key */
	fd_set		readfds;
	struct hostent	*ca_hostent;
	struct sockaddr_in ca_addr;
	struct timeval	timeout;
	DWORD		i;
	KX_MSG		pkt_to_send;
	KX_MSG		pkt_recvd;
	KX509_REQUEST	*request = 0;
	char		*pubkey_ptr;
	unsigned char	*tmp_ptr;
	int		pubkey_len;
/*
 * require 128 bytes of randomness  N.B. This is a moving target.
 * This is defined in OpenSSL's crypto/rand/rand_lcl.h.  But that
 * is an internal header.  As of OpenSSL 0.9.6 the value was 20 bytes.
 * As of OpenSSL 0.9.7 the value is 32 bytes.
 */
#define ENTROPY_NEEDED 128

	char		entropy_pool[ENTROPY_NEEDED];
	int		entropy_still_needed = ENTROPY_NEEDED;
	int		entropy_to_copy = 0;
	int		len;
	static int	triedAuthent = 0;
#ifdef macintosh
	OSErr		err;
#endif /* macintosh */
	int		rc;

#if defined(USE_KRB5)
	krb5_data	k5_authent;
#else
	static KTEXT_ST	authent;	/* BILLDO 2001.0330 -- make static so that it's preserved across try_ca calls */
#endif

	*err_num_ptr = 0;


#if defined(USE_KRB5)
	/* For K5, we always generate a new authenticator for the host we are contacting */
	if (rc = get_cert_authent_K5(k5_context, ca_hostname, &k5_authent, sess_key,
				sess_len_ptr, realm, tkt_cache_name, emsg))
	{
		return rc;
	}
#endif

	/*
	 * If this is the first host we've tried
	 * {
	 *	If using K4
	 *	{
	 *	  generate authenticator
	 *	}
	 *	generate the key-pair
	 * }
	 */

	if ( NULL == *rsa)
	{
#if !defined(USE_KRB5)
		if (triedAuthent == 0) 
		{
			triedAuthent = 1;
			if (rc = get_cert_authent(&authent, sess_key, sess_len_ptr, realm, emsg))
			{
				return rc;
			}
		}
		else
		{
			return(*err_num_ptr = KX509_STATUS_CLNT_TMP);

		}
#endif
		/*
		 * THIS COULD BE **BETTER** -- Starting with Openssl-0.9.6, the
		 * RAND functions insist that ENTROPY_NEEDED (20) bytes of seed
		 * material be provided before they will work at all.  As a really
		 * cheesy work-around, the code below simply copies the 8-byte
		 * kerberos session-key a couple times to generate 24-bytes of
		 * entropy...
		 */
		
		/*
		 * Note that with 3DES, the session key is 24 bytes.
		 * Don't copy too much!
		 */
		if (*sess_len_ptr > entropy_still_needed)
			entropy_to_copy = entropy_still_needed;
		else
			entropy_to_copy = *sess_len_ptr;

		memcpy(entropy_pool, sess_key, entropy_to_copy);
		entropy_still_needed -= entropy_to_copy;
		while (entropy_still_needed > 0)
		{
			if (entropy_still_needed < *sess_len_ptr)
			{
				memcpy(&entropy_pool[ENTROPY_NEEDED-entropy_still_needed],
					sess_key, entropy_still_needed);
				entropy_still_needed = 0;
			}
			else
			{
				memcpy(&entropy_pool[ENTROPY_NEEDED-entropy_still_needed],
					sess_key, *sess_len_ptr);
				entropy_still_needed -= *sess_len_ptr;
			}
		}

		/* GENERATE PUBLIC KEY PAIR  */

		RAND_seed(entropy_pool, ENTROPY_NEEDED);

		*rsa=client_genkey(keybits); 
		if (*rsa == NULL) {		/* Verify that key generation succeeded.  If not, bail out now! */
			*emsg = "Error generating RSA key pair.";
			return(*err_num_ptr = KX509_STATUS_CLNT_BAD);
		}
	 
		log_printf("try_ca: sending authentication request (len %d) to KCA\n",
#if defined(USE_KRB5)
			k5_authent.length);
#else
			authent.length);
#endif

	}

	
	/* CONVERT KEY-PAIR INFO AND AUTHENT TO REQUEST */

	memset(buffer, 0, sizeof(buffer));	/* BILLDO 2001.0330 -- something causes 1st try_ca failures to make later try_ca's fail... */
	pubkey_ptr		   = buffer;
	tmp_ptr = (unsigned char *)pubkey_ptr;
	pubkey_len = i2d_RSAPublicKey (*rsa, (unsigned char **)&tmp_ptr);

	log_printf("try_ca: sending pubkey_len=%d bytes of public key\n", pubkey_len);

	request = KX509_REQUEST_new();
	fill_in_octet_string(request->authenticator,
#if defined(USE_KRB5)
		k5_authent.data, k5_authent.length);
#else
		(char *)authent.dat, authent.length);
#endif
	fill_in_octet_string(request->pkey, pubkey_ptr, pubkey_len);
	KX509_REQUEST_compute_checksum((unsigned char *)version_2_0_string, request,
		request->hash, sess_key, *sess_len_ptr);

	/* CONVERT REQUEST STRUCTURE TO WIRE-VERSION MSG */

	log_printf("try_ca: authent.length is %d, and pubkey_len is %d\n",
#if defined(USE_KRB5)
		k5_authent.length, pubkey_len);
#else
		authent.length, pubkey_len);
#endif

	len = i2d_KX509_REQUEST(request, 0) + 4;
	log_printf("try_ca: Checking len %d against MAX_UDP_PAYLOAD_LEN %d\n",
			len, MAX_UDP_PAYLOAD_LEN);
	if (len > MAX_UDP_PAYLOAD_LEN)
	{
		log_printf("try_ca: len=%d MAX_UDP_PAYLOAD_LEN=%d\n",
			len, MAX_UDP_PAYLOAD_LEN);
		*emsg = "Weird!  KX509 transmit packet is too large!";
		return(*err_num_ptr = KX509_STATUS_CLNT_BAD);
	}

	if (MSG_ALLOC(&pkt_to_send, len))
	{
		log_printf("try_ca: could not allocate %d bytes?\n", len);
		*emsg = "Try again.  (Hopefully) transient client-side malloc problem";
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	memcpy(pkt_to_send.m_data, version_2_0_string, 4);
	tmp_ptr = pkt_to_send.m_data+4;
	i2d_KX509_REQUEST(request, &tmp_ptr);
	pkt_to_send.m_curlen = tmp_ptr - pkt_to_send.m_data;

	/* XXX This won't work on macintosh */
	if (debugPrint) {
		PEM_write(stderr, "kx509 request", ca_hostname,
			pkt_to_send.m_data+4, pkt_to_send.m_curlen-4);
	}


	/* DETERMINE IP ADDRESS OF KCA SERVER */

	if (!(ca_hostent = gethostbyname(ca_hostname)))
	{
#ifdef macintosh
		err = GetMITLibError();
		if (GetErrorLongFormat(err, buffer, sizeof(buffer)) == noErr) {
		    log_printf("try_ca: gethostbyname of CA (%s) failed ('%s')\n",
			ca_hostname, buffer);
		}
#else /* !macintosh */
		log_printf("try_ca: gethostbyname of CA (%s) failed ('%s')\n",
			ca_hostname, strerror(errno));
#endif /* macintosh */
		*emsg = "try_ca: gethostbyname failed";
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	memset(&ca_addr, 0, sizeof(ca_addr));
	ca_addr.sin_family	= AF_INET;
	ca_addr.sin_port	= htons(KRBCHK_PORT);
	ca_addr.sin_addr.s_addr	= *(int *)(ca_hostent->h_addr_list[0]);

	/* "CONNECT" TO IT (ICMP RESPONSE INDICATES HOST ISN'T LISTENING ON THAT PORT) */

	log_printf("try_ca: About to connect to KCA at %s:%d\n",
		inet_ntoa(ca_addr.sin_addr), KRBCHK_PORT);
	if (udp_nb_connect(socket, &ca_addr) == -1)
	{
#ifdef macintosh
		err = GetMITLibError();
		if (GetErrorLongFormat(err, buffer, sizeof(buffer)) == noErr) {
		    log_printf("try_ca: udp_nb_connect failed with err %d ('%s')\n",
			err, buffer);
		}
#else /* !macintosh */
		log_printf("try_ca: udp_nb_connect failed with errno %d ('%s')\n",
			errno, strerror(errno));
#endif /* macintosh */
		*emsg = "try_ca: udp_nb_connect failed";
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	/* SOMETHINGS LISTENING -- SEND PACKET */

	i = udp_nb_send(socket, &pkt_to_send);
	log_printf("try_ca: sent KX_CLNT_PKT of %0d bytes (rc = %d) \n",
			pkt_to_send.m_curlen, i);

	/* RECV WIRE-VERSION OF KX_SRVR_PKT FROM CA SERVER */

	if (MSG_ALLOC(&pkt_recvd, MAX_KSP_LEN))
	{
		log_printf("try_ca: failed to allocate %d bytes for recv pkt?\n", MAX_KSP_LEN);
		*emsg = "Try again.  (Hopefully) transient client-side malloc problem";
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	/* WAIT UP TO "KX509_CLIENT_TIMEOUT" SECONDS FOR RESPONSE */

	FD_ZERO(&readfds);
	FD_SET((WORD)socket, &readfds);
	timeout.tv_sec = KX509_CLIENT_TIMEOUT;
	timeout.tv_usec = 0;
	i = udp_nb_select(&readfds, NULL, NULL, &timeout);
	if (i<0)
	{
#ifdef macintosh
		err = GetMITLibError();
		if (GetErrorLongFormat(err, buffer, sizeof(buffer)) == noErr) {
		    log_printf("try_ca: udp_nb_select failed with code %d, errno %d ('%s')\n",
			i, err, buffer);
		}
#else /* !macintosh */
		log_printf("try_ca: udp_nb_select failed with code %d, errno %d ('%s')\n",
			i, errno, strerror(errno));
#endif /* macintosh */
		*emsg = "Error return waiting for response.";
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}
	else if (i==0)
	{
		log_printf("try_ca: timeout during udp_nb_select\n");
		*emsg = "Timed out waiting for response from a Kerberized Certificate Authority";
		KX509_REQUEST_free(request);
		MSG_FREE(&pkt_recvd);
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	if (udp_nb_recv(socket, &pkt_recvd) == -1)
	{
#ifdef macintosh
		err = GetMITLibError();
		if (GetErrorLongFormat(err, buffer, sizeof(buffer)) == noErr) {
			log_printf("try_ca: udp_nb_recv failed with err %d ('%s')\n",
				err, buffer);
		}
#else /* !macintosh */
		log_printf("try_ca: udp_nb_recv failed with errno %d ('%s')\n",
			errno, strerror(errno));
#endif /* macintosh */
#ifdef WIN32
		if (WSAGetLastError() == WSAECONNREFUSED)
#elif defined(macintosh)
		if (0)
#else /* !WIN32 && !macintosh */
		if (errno == ECONNREFUSED)
#endif
			*emsg = "Try again later.  None of the Kerberized Certificate Authorities are currently available.";
		else
			*emsg = "Strange.  Unexpected error on receive.";
		KX509_REQUEST_free(request);
		return(*err_num_ptr = KX509_STATUS_CLNT_TMP);
	}

	if (pkt_to_send.m_data)
		MSG_FREE(&pkt_to_send);


	*emsg = NULL;
	rc = (*verify_recvd_packet)(&pkt_recvd, arg);

	if (pkt_recvd.m_data)
		MSG_FREE(&pkt_recvd);
	KX509_REQUEST_free(request);

	return(rc);
}


void fill_in_octet_string(
	ASN1_OCTET_STRING *osp,
	char *st,
	int len)
{
	char *c;
	if (osp->data && osp->length)
		Free(osp->data);

	if (len <= 0)
		return;

	c = Malloc(len);
	memcpy(c, st, len);
	osp->data = (unsigned char *)c;
	osp->length = len;
}

struct verify_arg {
	KX509_RESPONSE	*response;
	char sess_key[32];
	int sess_len;
	char *emsg;
};


int
verify_recvd_packet(
	KX_MSG		*pkt_recvd,
	void 		*arg
)
{
	unsigned char *p;
#if defined(DEBUG) && !defined(WIN32)
	unsigned char *op;
#endif
	int length;
	ASN1_OCTET_STRING *hash;
	int result;
	struct verify_arg *varg = (struct verify_arg *)arg;

	if (pkt_recvd->m_curlen < 4)
	{
		sprintf(buffer,"verify_recvd_packet: received runt of length %d",
			pkt_recvd->m_curlen);
		varg->emsg = buffer;
		return KX509_STATUS_CLNT_BAD;
	}
	if (2[(unsigned char *)pkt_recvd->m_data] != version_2_0_string[2])
	{
		sprintf(buffer,
			"verify_recvd_packet: rec'd version %d.%d does not match %d.*",
			2[(unsigned char *)pkt_recvd->m_data],
			3[(unsigned char *)pkt_recvd->m_data],
			version_2_0_string[2]);
		varg->emsg = buffer;
		return KX509_STATUS_CLNT_BAD;
	}
	p = pkt_recvd->m_data+4;
	length = pkt_recvd->m_curlen-4;
#if defined(DEBUG) && !defined(WIN32)
	/* XXX this wont work on macintosh */
	if (debugPrint) {
		PEM_write(stderr, "kx509 response", "", p, length);
		bin_dump((char *)p, length);
	}
	op = p;
#endif
	if (!(varg->response = d2i_KX509_RESPONSE(NULL, &p, length)))
	{
		varg->emsg = "verify_recvd_packet: d2i_X509_RESPONSE failed";
		return KX509_STATUS_CLNT_BAD;
	}
#if defined(DEBUG) && !defined(WIN32)
	log_printf ("Decoded %d bytes of %d\n", p-op, length);
	print_response(varg->response);
#endif
	if (!varg->response->hash)
	{
		if (varg->response->error_message)
		{
			int xlen;
			xlen = sizeof buffer-1;
			if (xlen > varg->response->error_message->length)
				xlen=varg->response->error_message->length;
			memcpy(buffer, varg->response->error_message->data, xlen);
			buffer[xlen] = 0;
			varg->emsg = buffer;
		}
		return varg->response->status ?
			varg->response->status : KX509_STATUS_CLNT_BAD;
	}
	if (!(hash = ASN1_OCTET_STRING_new()))
	{
		varg->emsg = "verify_recvd_packet: out of memory";
		return KX509_STATUS_CLNT_BAD;
	}
	KX509_RESPONSE_compute_checksum(pkt_recvd->m_data,
		varg->response,
		hash,
		varg->sess_key, varg->sess_len);
	result = 0;
	if (hash->length != varg->response->hash->length
		|| memcmp(hash->data, varg->response->hash->data, hash->length))
	{
		varg->emsg = "verify_recvd_packet: generated hash did not compare";
		result = KX509_STATUS_CLNT_BAD;
	}
	ASN1_OCTET_STRING_free(hash);
	return result;
}

/*
 *=========================================================================*
 *
 * getcert()
 *
 * Attempt to obtain a certificate
 *
 *=========================================================================*
 */
int getcert(
	RSA	**rsa,
	X509	**certp,
	char	*emsg,
	int	elen,
	char	*realm,
	char	*tkt_cache_name
)
{
#if defined(USE_KRB5)
	krb5_error_code	k5_result;
	krb5_context	k5_context;
#endif
	char	**dns_hostlist = NULL;
	char	**kca_hostlist;
	char	ca_hostname_to_try[256];
	char	*base_realm;
	char	*env_host_list;
	int	rc;
	int	n;
	struct verify_arg arg[1]; 
	int	socket;
	unsigned char	*tmp_ptr;
#ifdef macintosh
	OSErr		err;
#endif /* macintosh */

#ifdef WIN32
{
	WORD			wVersionRequested;
	WSADATA			wsaData;
	int			err;



	/* GET SOCKET */


	wVersionRequested = MAKEWORD( 2, 2 );
 
	err = WSAStartup( wVersionRequested, &wsaData );
	if ( err != 0 )
		return FALSE;
 
	/* Confirm that the WinSock DLL supports 2.2.*/
	/* Note that if the DLL supports versions greater    */
	/* than 2.2 in addition to 2.2, it will still return */
	/* 2.2 in wVersion since that is the version we      */
	/* requested.                                        */
 
	if ( LOBYTE( wsaData.wVersion ) != 2 ||
			HIBYTE( wsaData.wVersion ) != 2 ) 
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		WSACleanup( );
		return FALSE;
	}
 }
#endif /* WIN32 */


	*certp = 0;
	*emsg = 0;
	memset((char*)arg, 0, sizeof *arg);
	arg->sess_len = sizeof(arg->sess_key);

	/* CREATE SOCKET TO BIND TO CA SERVER */

	if ((socket=udp_nb_socket(0)) == -1)
	{
#ifdef macintosh
		err = GetMITLibError();
		if (GetErrorLongFormat(err, buffer, sizeof(buffer)) == noErr) {
			log_printf("try_ca: udp_nb_socket failed to obtain a socket ('%s')\n",
				buffer);
		}
#else /* !macintosh */
		log_printf("try_ca: udp_nb_socket failed to obtain a socket ('%s')\n",
			strerror(errno));
#endif /* macintosh */
		arg->emsg = "Failed to create a socket.\n";
		rc = KX509_STATUS_CLNT_TMP;
		goto Failed;
	}

#if defined(USE_KRB5)
#if !defined(USE_MSK5)
	if ((k5_result = krb5_init_context(&k5_context))) 
	{
		msg_printf("getcert: unable to initialize Kerberos 5 context: %s\n",
			error_message(k5_result));
		arg->emsg = "Verify KRB5 configuration. Failed to initialize Kerberos 5 context.\n";
		rc = KX509_STATUS_CLNT_BAD;
		goto Failed;
	}
#else /* USE_MSK5 */
	k5_context = NULL;
#endif /* USE_MSK5 */
#endif /* USE_KRB5 */

	/* Determine the realm */

#if defined(USE_KRB5)
	if (rc = get_krb5_realm(k5_context, realm, tkt_cache_name, &arg->emsg))
#else
	if (rc = get_krb4_realm(realm, &arg->emsg))
#endif
	{
/*		log_printf("getcert: failed to determine kerberos realm information (%d)\n", rc); */
/*		arg->emsg = "Failed to determine kerberos realm information.\n"; */
/*		rc = KX509_STATUS_CLNT_BAD; */
		goto Failed;
	}

	/* CLEAN OUT OLD CERTS  */
 
#ifdef WIN32
	clean_cert_store(realm); 
#endif /* WIN32 */

	/*
	 * We use one of two ways to determine the hostname(s) of the KCA server(s):
	 *
	 *
	 *	1.  if the KCA_HOST_LIST environment variable is defined,
	 *		we simply use the list of hostname(s) defined there
	 *
	 *	2.  otherwise, we *assume* that the KCA servers can be
	 *		reached by resolving the hostname(s) that Kerberos
	 *		expects the KDC to be at for a given base_realm as
	 *		specified in /etc/krb.conf
	 *
	 *		(note: for the "ENGIN.UMICH.EDU" realm,
	 *		       "UMICH.EDU" is used for base_realm)
	 */

	/* DNS SRV records should obviate need for ENGIN->UMICH mapping */
	base_realm = realm;

	/* Use environment variable first, otherwise use list from DNS */

	if ((env_host_list = getenv("KCA_HOST_LIST")) != NULL)
	{
		char *host;
		int hostcount = 0;
		char *hostlist = NULL;
		char **hostarray = NULL;

		/* Make a copy of the environment string */
		if (strlen(env_host_list)) hostlist = malloc(strlen(env_host_list + 1));
		if (hostlist)
			strcpy(hostlist, env_host_list);
		else
		{
			rc = KX509_STATUS_CLNT_BAD;
			arg->emsg = "Empty KCA_HOST_LIST environment variable or malloc error";
			goto Failed;
		}

		hostarray = calloc(MAX_KCA_HOSTS + 1, sizeof(char *));

		if (hostarray)
		{

			/* Separate the hosts in the list and keep an array of pointers */
			host = strtok(hostlist, " ");
			while (host != NULL && *host != '\0')
			{
				hostarray[hostcount++] = host;
				host = strtok(NULL, " ");
			}
		}
		else
		{
			rc = KX509_STATUS_CLNT_BAD;
			arg->emsg = "Error allocating array for KCA_HOST_LIST";
			goto Failed;
		}
		if (hostcount <= 0)
		{
			rc = KX509_STATUS_CLNT_BAD;
			arg->emsg = "Empty KCA_HOST_LIST environment variable or tokenize error";
			goto Failed;
		}
		kca_hostlist = hostarray;
	} else {
		if (get_kca_list(base_realm, &dns_hostlist))
		{
			rc = KX509_STATUS_CLNT_BAD;
			arg->emsg = "DNS SRV lookup of KCA hostname(s) failed!";
			goto Failed;
		}
		else
			kca_hostlist = dns_hostlist;
	}

	/* ITERATE THROUGH LIST OF KCA HOSTNAMES */

	for (n=0; kca_hostlist[n]; )
	{
		int e;

		strcpy(ca_hostname_to_try, kca_hostlist[n++]);

		/* Exit the loop as soon as we get a good response */
#if defined(USE_KRB5)
		if (!(rc = try_ca(k5_context, socket, ca_hostname_to_try,
				realm, rsa, certp, verify_recvd_packet,
				(void*)arg, arg->sess_key, &arg->sess_len,
				tkt_cache_name,
				&arg->emsg, &e)))
#else
		if (!(rc = try_ca(socket, ca_hostname_to_try,
				realm, rsa, certp, verify_recvd_packet,
				(void*)arg, arg->sess_key, &arg->sess_len, &arg->emsg, &e)))
#endif
			break;
		else
		{
			log_printf("try_ca to '%s' returned rc %d, ecode %d, emsg '%s'\n",
				ca_hostname_to_try, rc, e, arg->emsg);
		}
	}
	if (!n)
	{
		rc = KX509_STATUS_CLNT_BAD;
		arg->emsg = "Error!  Unable to determine KCA hostname(s)!";
	}
	if (arg->emsg)
		log_printf("%s\n", arg->emsg);
Failed:
	if (dns_hostlist != NULL)
	{
#if 0
//		for (n=0; dns_hostlist[n]; n++)
//			Free(dns_hostlist[n]);
#endif
		Free(dns_hostlist);
	}

	if (rc)
	{
		if (!arg->emsg || !*arg->emsg)
			arg->emsg = "Missing error message #1";
		strncpy(emsg, arg->emsg, elen);
		emsg[elen-1] = '\0';
	}
	else if (rc = arg->response->status)
	{
		if (arg->response->error_message)
		{
			log_printf ("status %d; response had error message; contents were:\n", rc);
#ifdef DEBUG
			bin_dump((char *)arg->response->error_message->data, arg->response->error_message->length);
#endif
		} else {
			log_printf ("status %d; response no longer has error message\n", rc);
		}
		if (!arg->response->error_message
		|| !arg->response->error_message->length)
			strncpy(emsg, "Missing error message #2", elen);
		else {
			if (arg->response->error_message->length > (elen-1))
				arg->response->error_message->length = elen-1;
			memcpy(emsg,
				arg->response->error_message->data,
				arg->response->error_message->length);
			emsg[arg->response->error_message->length] = 0;
		}
	} else if (arg->response->certificate) {
		tmp_ptr = arg->response->certificate->data;
		if (!(*certp = d2i_X509(NULL, &tmp_ptr,
			arg->response->certificate->length)))
		{
			strncpy(emsg, "getcert: d2i_X509 failed", elen);
			rc = KX509_STATUS_CLNT_BAD;
		}
	} else {
		strncpy(emsg, "getcert: missing certificate", elen);
		rc = KX509_STATUS_CLNT_BAD;
	}

	KX509_RESPONSE_free(arg->response);

	return rc;
}

#ifdef DEBUG

/*
 *=========================================================================*
 *
 * print_response()
 *
 *=========================================================================*
 */
void print_response(KX509_RESPONSE *client_response)
{
	log_printf ("response status %d\n", client_response->status);
	if (client_response->certificate)
	{
		log_printf ("response certificate:\n");
		bin_dump((char *)client_response->certificate->data,
		client_response->certificate->length);
	} else log_printf ("no response certificate\n");
		if (client_response->hash)
	{
		log_printf ("response hash:\n");
		bin_dump((char *)client_response->hash->data,
		client_response->hash->length);
	} else log_printf ("no response hash\n");
	if (client_response->error_message)
	{
		log_printf ("response error_message:\n");
		bin_dump((char *)client_response->error_message->data,
		client_response->error_message->length);
	} else log_printf ("no response error_message\n");
	return;
}

/*
 *=========================================================================*
 *
 * print_request()
 *
 *=========================================================================*
 */
void print_request(KX509_REQUEST *server_request)
{
	log_printf ("request Authenticator:\n");
	bin_dump((char *)server_request->authenticator->data,
		server_request->authenticator->length);
	log_printf ("request hash:\n");
	bin_dump((char *)server_request->hash->data,
		server_request->hash->length);
	log_printf ("request pkey:\n");
	bin_dump((char *)server_request->pkey->data,
		server_request->pkey->length);
}

/*
 *=========================================================================*
 *
 * bin_dump()
 *
 *=========================================================================*
 */
bin_dump(char *cp, int s)
{
	char *buffer;
	char c;
	int w;
	int i;
	long o;

	o = 0;
	buffer = cp;
	while (s > 0)
	{
		c = 16;
		if (c > s) c = s;
		log_printf ("%06lx:", o);
		w = 0;
		for (i = 0; i < c/2; ++i)
			w += 5, log_printf (" %4x", ((unsigned short *)buffer)[i]);
		if (c & 1)
			w += 3, log_printf (" %2x", buffer[c-1]);
		while (w < 41)
			++w, log_printf(" ");
		for (i = 0; i < c; ++i)
			if (isprint(buffer[i]))
				log_printf("%c", buffer[i]);
			else
				log_printf(".");
		log_printf("\n");
		o += c;
		buffer += c;
		s -= c;
	}
	log_printf ("%06lx:\n", o);
	return 1;
}
#endif
