/*
 * Copyright    2000
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of the university of
 * michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software
 * without specific, written prior authorization.  if the
 * above copyright notice or any other identification of the
 * university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * this software is provided as is, without representation
 * from the university of michigan as to its fitness for any
 * purpose, and without warranty by the university of
 * michigan of any kind, either express or implied, including
 * without limitation the implied warranties of
 * merchantability and fitness for a particular purpose. the
 * regents of the university of michigan shall not be liable
 * for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising
 * out of or in connection with the use of the software, even
 * if it has been or is hereafter advised of the possibility of
 * such damages.
 */

/* get_realm.c -- gather into one file all code related to determining
 *			the realm of the user's (currently active)
 *			Kerberos tickets, irrespective of the client's
 *			architecture or kerberos implementation
 *
 * CHANGE HISTORY:
 *	2000.1213 -- billdo -- created
 *
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <errno.h>
#include <string.h>

#ifdef WIN32
# define __WINCRYPT_H__		// PREVENT windows.h from including wincrypt.h
				// since wincrypt.h and openssl namepsaces collide
				//  ex. X509_NAME is #define'd and typedef'd ...
# include <winsock.h>		// Must be included before <windows.h> !!!
# include <windows.h>
# include <openssl/pem.h>
#endif  /* WIN32 */


#include <stdlib.h>
#include <openssl/x509v3.h>

#ifdef USE_KRB5
# include <krb5.h>
# include <com_err.h>
#else /* !USE_KRB5 */
# ifndef WIN32
#  ifndef linux
#   define DES_DEFS			// Prevent collision with K5 DES delarations
#  endif /* !linux */
#  ifdef macintosh
#   include <KClient.h>
#  else /* !macintosh */
#   include "des-openssl-hack.h"
#   include <krb.h>
#  endif /* macintosh */
# else /* !WIN32 */
#  include "des-openssl-hack.h"
#  include <krb.h>
# endif /* !WIN32 */
#endif /* USE_KRB5 */


#include "kx509.h"
#include "debug.h"

#if defined(USE_KRB5)
/*
 *=========================================================================*
 *
 * get_krb5_realm()
 *
 *=========================================================================*
 */
int get_krb5_realm(krb5_context k5_context, char *realm, 
		   char *tkt_cache_name, char **err_msg)
{
#if !defined(USE_MSK5)
	krb5_ccache cc;
	krb5_principal me;
	krb5_error_code result;

	*realm = 0;

	if(tkt_cache_name)
	{
	  if (result = krb5_cc_resolve(k5_context, tkt_cache_name, &cc))
	  {
	    msg_printf("get_cert_authent_K5: krb5_cc_resolve: %s\n",
	               error_message(result));
	    *err_msg = "Try re-authenticating(K5).  "
	               "Unable to determine default credentials cache.";
	    return KX509_STATUS_CLNT_FIX;
	  }
	}
	else
	{
	  if (result = krb5_cc_default(k5_context, &cc))
	  {
	    msg_printf("get_krb5_realm: krb5_cc_default: %s\n", 
		       error_message(result));
	    *err_msg = "Try re-authenticating(K5).  "
			"Unable to determine default credentials cache.";
	    return KX509_STATUS_CLNT_FIX;
	  }
	}

	if (result = krb5_cc_get_principal(k5_context, cc, &me))
	{
		msg_printf("get_krb5_realm: krb5_cc_get_principal: %s\n", error_message(result));
		*err_msg = "Try re-authenticating(K5).  "
			"Unable to determine principal from credentials cache.";
		return KX509_STATUS_CLNT_FIX;
	}

	strcpy(realm, krb5_princ_realm(k5_context, me)->data);
	
	krb5_free_principal(k5_context, me);

	return 0;
#else /* USE_MSK5 */

	char	user[256];
	int	rc = 0;


	*realm = '\0';
	*err_msg = NULL;

	if ((rc = MSK5_get_userid_and_realm( user, realm )) != TRUE)
	{
		*err_msg = "MSK5_get_userid_and_realm failed";
	}

	return !rc;
#endif /* USE_MSK5 */
}

#else	/* USE_KRB5 */

/*
 *=========================================================================*
 *
 * get_krb4_realm()
 *
 *=========================================================================*
 */
int get_krb4_realm(char *realm, char **err_msg)
{
	char	dummy[MAX_K_NAME_SZ+1];

	/* DETERMINE REALM OF USER'S TICKET FILE */
	*realm = 0;

#if defined(macintosh)
	if ((err = KClientGetLocalRealm(realm)) != noErr)
#else
	if (krb_get_tf_fullname(tkt_string(), dummy, dummy, realm))
#endif
	{
		*err_msg = "Have you authenticated to Kerberos?  Your ticket file is invalid.";
		return KX509_STATUS_CLNT_FIX;
	}

	return 0;
}

#endif	/* USE_KRB5 */
