/*                          
 * Copyright  ©  2005
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works 
 * and redistribute this software and such derivative works 
 * for any purpose, so long as the name of the university of 
 * michigan is not used in any advertising or publicity 
 * pertaining to the use or distribution of this software 
 * without specific, written prior authorization.  if the 
 * above copyright notice or any other identification of the 
 * university of michigan is included in any copy of any 
 * portion of this software, then the disclaimer below must 
 * also be included.
 *
 * this software is provided as is, without representation 
 * from the university of michigan as to its fitness for any 
 * purpose, and without warranty by the university of 
 * michigan of any kind, either express or implied, including 
 * without limitation the implied warranties of 
 * merchantability and fitness for a particular purpose. the 
 * regents of the university of michigan shall not be liable 
 * for any damages, including special, indirect, incidental, or 
 * consequential damages, with respect to any claim arising 
 * out of or in connection with the use of the software, even 
 * if it has been or is hereafter advised of the possibility of 
 * such damages.
 */

#include "kct_util.h"
#include <errno.h>

FILE   *logfile;
#ifdef DEBUG
int     kct_debug_level = KCTLOG_DEBUG;
#else
int     kct_debug_level = KCTLOG_WARNING;
#endif

void
free_memory(void *memory)
{

    if (memory)
	free(memory);
    else
	kct_log(KCTLOG_ALWAYS, "free_memory: called with a null pointer!\n");
    memory = NULL;
}

void   *
get_memory(size_t nelem, size_t elsize)
{

    void   *m;

    if ((m = (void *) calloc(nelem, elsize)) == NULL) {
	kct_log(KCTLOG_ALWAYS, "get_memory: allocating: %d * %d: %s\n",
		nelem, elsize, strerror(errno));
	exit(1);
    }

    return m;

}

int
kct_open_log(char *name)
{

    long    now;
    struct tm *tp;
    char    filename[512];

    now = time(0);
    tp = localtime(&now);
    sprintf(filename, "%s.%04d%02d%02d", name, tp->tm_year + 1900,
	    tp->tm_mon + 1, tp->tm_mday);

    logfile = freopen(filename, "a", stderr);
    if (!logfile) {
	/* Can't print to stderr, cuz it ain't there anymore! */
	fprintf(stdout, "kct_open_log: opening log file '%s': %s\n",
		filename, strerror(errno));
	exit(1);
    }
    kct_log(KCTLOG_ALWAYS, "Logging begins\n");

    return 1;
}

int
kct_reopen_log(char *name)
{
    kct_log(KCTLOG_ALWAYS, "Reopening the log file\n");

    fclose(logfile);
    kct_open_log(name);
}

void
kct_close_log()
{
    fclose(logfile);
}

char   *
get_time()
{
    char   *time_string;

    time_t  t;
    time(&t);
    time_string = ctime(&t);
    time_string[strlen(time_string) - 1] = '\0';
    return (time_string + 4);	/* Skip the Day of Week */
}

int
kct_log(int level, char *fmt, ...)
{

    va_list args;
    char   *p, *sval;
    int     ival;
    double  dval;
    int     i;

    if (!logfile)
	return (-1);

    if (level > kct_debug_level)
	return 0;

#ifdef NO_TIME
    fprintf(logfile, "[%d]  ", getpid());
#else
    fprintf(logfile, "%s [%d]: ", get_time(), getpid());
#endif

    va_start(args, fmt);

    for (p = fmt; *p; p++) {
	if (*p != '%') {
	    fprintf(logfile, "%c", (*p));
	    continue;
	}

	switch (*++p) {
	case 'c':
	    fprintf(logfile, "%c", (*p));
	    break;
	case 'd':
	    ival = va_arg(args, int);
	    fprintf(logfile, "%d", ival);
	    break;
	case 'f':
	    dval = va_arg(args, double);
	    fprintf(logfile, "%f", dval);
	    break;
	case 'x':
	    ival = va_arg(args, int);
	    fprintf(logfile, "0x%02x", ival);
	    break;
	case 's':
	    for (sval = va_arg(args, char *); *sval; sval++)
		fprintf(logfile, "%c", (*sval));
	    break;
	case 'p':
	    ival = va_arg(args, int);
	    fprintf(logfile, "%p", ival);
	    break;
	default:
	    fprintf(logfile, "%c", (*p));
	    break;
	}
    }

    va_end(args);

    fflush(logfile);

    return 1;
}

size_t
recvblock(int sd, unsigned char *buf, size_t total)
{

    int     len;
    unsigned int recvd = 0;

    while (recvd < total) {
	len = recv(sd, (char *) buf + recvd, total - recvd, 0);
	if (len <= 0)
	    return (-1);
	recvd += len;
    }
    return (recvd);
}

size_t
sendblock(int sd, unsigned char *buf, size_t total)
{

    int     len;
    unsigned int sent = 0;

    while (sent < total) {
	len = send(sd, (char *) buf + sent, total - sent, 0);
	if (len <= 0)
	    return (-1);
	sent += len;
    }
    return (sent);
}

void
print_buffer(unsigned char *buffer, size_t len, unsigned char *comment)
{

    int     i;

    fprintf(logfile, "%s ", comment);
    for (i = 0; i < len; i++) {
	fprintf(logfile, "%02X ", buffer[i]);
	if ((i + 1) % 22 == 0 && i != 0)
	    fprintf(logfile, "\n");
    }
    fprintf(logfile, "\n");
}
