/*
 * Copyright    2000,2005
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of the university of
 * michigan is not used in any advertising or publicity
 * pertaining to the use or distribution of this software
 * without specific, written prior authorization.  if the
 * above copyright notice or any other identification of the
 * university of michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * this software is provided as is, without representation
 * from the university of michigan as to its fitness for any
 * purpose, and without warranty by the university of
 * michigan of any kind, either express or implied, including
 * without limitation the implied warranties of
 * merchantability and fitness for a particular purpose. the
 * regents of the university of michigan shall not be liable
 * for any damages, including special, indirect, incidental, or
 * consequential damages, with respect to any claim arising
 * out of or in connection with the use of the software, even
 * if it has been or is hereafter advised of the possibility of
 * such damages.
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "kct.h"
#include "kct_authz.h"
#include "kct_err.h"
#include "kct_util.h"

/* Global list of valid requestors */
REQUESTOR *ok_reqrs = NULL;

/*
 * Build list of allowable requests
 */
int
kct_authz_init(char *filename)
{
    char    line[KCT_MAXLINELEN];
    char   *next_scan;
    char   *tok;
    REQUESTOR *reqr, *last_reqr;
    OK_SVC *okd, *last_okd;
    FILE   *fp;
    int     lineno;

    /*
     * Open and read the file containing the list of valid requesting
     * principals and that requestor's list of authorized services.  We
     * will not accept a request from a principal if its name does not
     * appear in this file, nor if the service requested is not authorized.
     */
    if (NULL == (fp = fopen(filename, "r"))) {
	kct_log(KCTLOG_ALWAYS,
		"kct_authz_init: opening authorization file '%s': %s\n",
		filename, strerror(errno));
	exit(1);
    }

    lineno = 0;
    while (NULL != fgets(line, sizeof(line), fp)) {
	lineno++;
	if (line[strlen(line) - 1] != '\n') {
	    kct_log(KCTLOG_ALWAYS,
		    "kct_authz_init: %s(%d): "
		    "line too long ignoring \"%s...\"\n",
		    filename, lineno, line);
	    continue;
	}
	line[strlen(line) - 1] = ' ';	/* replace new-line with blank */
	next_scan = &line[0];	/* start at beginning of new line */

	/* parse line from filename:
	 *
	 *      <pname/pinst> <svc/sinst>[ <svc/sinst>*]
	 *
	 * where:
	 *      pname/pinst  is the Kerberos principal and instance of
	 *                   the authenticated requestor
	 *      svc/sinst    is a Kerberos service principal and instance
	 *                   this requestor is authorized to request
	 *
	 *      NOTE: for svc/sinst:
	 *
	 *           "imap"          is principal only, null instance
	 *           "imap/critters" is principal and specific instance
	 *           "imap/*"        is principal and all instances (null too)
	 */

	/* Get <pname/pinst> from line.  Tokens are blank or tab-delimited */
	if (line[0] == '#')
	    continue;		/* comment, skip it */

	tok = (char *) strtok_r(line, " \t", &next_scan);
	if (!tok)
	    continue;		/* "empty" line, skip it */

	/* Initialize REQUESTOR record for this pname */
	reqr = (REQUESTOR *) malloc(sizeof(REQUESTOR));
	if (NULL == reqr) {
	    kct_log(KCTLOG_ALWAYS,
		    "kct_authz_init: %s(%d): "
		    "cannot allocate %d bytes for REQUESTOR\n",
		    filename, lineno, sizeof(REQUESTOR));
	    exit(1);
	}
	memset(reqr, 0, sizeof(REQUESTOR));

	reqr->pname = (char *) malloc(strlen(tok) + 1);
	if (NULL == reqr->pname) {
	    kct_log(KCTLOG_ALWAYS,
		    "kct_authz_init: %s(%d): "
		    "cannot allocate %d bytes for pname/pinst\n",
		    filename, lineno, strlen(tok) + 1);
	    exit(1);
	}
	strcpy(reqr->pname, tok);

	/* Append to linked-list of OK reqeustors */
	if (!ok_reqrs)
	    ok_reqrs = reqr;
	else
	    last_reqr->next = reqr;
	last_reqr = reqr;

	/* Build list of services that are OK for this requestor */
	while (tok = (char *) strtok_r(NULL, " \t", &next_scan)) {
	    char   *sep, *sinst_tok;
	    char   *snext_scan;

	    sep = strchr(tok, '/');
	    if (NULL == sep) {
		kct_log(KCTLOG_WARNING,
			"kct_authz_init: %s(%d): "
			"warning: null instance in service '%s'\n",
			filename, lineno, tok);
		sinst_tok = NULL;
	    } else {
		*sep = '\0';
		sinst_tok = sep + 1;
	    }

	    /* Initialize OK_SVC record for this pname */
	    okd = (OK_SVC *) malloc(sizeof(OK_SVC));
	    if (NULL == okd) {
		kct_log(KCTLOG_ALWAYS,
			"kct_authz_init: %s(%d): "
			"cannot allocate %d bytes for OK_SVC\n",
			filename, lineno, sizeof(OK_SVC));
		exit(1);
	    }
	    memset(okd, 0, sizeof(OK_SVC));

	    okd->sname = (char *) malloc(strlen(tok) + 1);
	    if (NULL == okd->sname) {
		kct_log(KCTLOG_ALWAYS,
			"kct_authz_init: %s(%d): "
			"cannot allocate %d bytes for sname\n",
			filename, lineno, strlen(tok));
		exit(1);
	    }
	    strcpy(okd->sname, tok);

	    if (sinst_tok) {
		okd->sinst = (char *) malloc(strlen(sinst_tok) + 1);
		if (NULL == okd->sinst) {
		    kct_log(KCTLOG_ALWAYS,
			    "kct_authz_init: %s(%d): "
			    "cannot allocate %d bytes for sinst\n",
			    filename, lineno, strlen(sinst_tok));
		    exit(1);
		}
		strcpy(okd->sinst, sinst_tok);
	    } else {
		okd->sinst = NULL;
	    }

	    /* Append to linked-list of Okay'd services */
	    if (!reqr->svcs)
		reqr->svcs = okd;
	    else
		last_okd->next = okd;
	    last_okd = okd;
	}
    }
    fclose(fp);

    return 0;
}

int
kct_free_svcs(OK_SVC * s)
{
    OK_SVC *s2;

    if (NULL == s)
	return;
    while (NULL != s) {
	s2 = s->next;
	if (s->sname)
	    free(s->sname);
	if (s->sinst)
	    free(s->sinst);
	free(s);
	s = s2;
    }
    return 0;
}

int
kct_free_reqr(REQUESTOR * r)
{
    if (NULL == r)
	return;
    if (r->pname)
	free(r->pname);
    if (r->svcs)
	kct_free_svcs(r->svcs);
    free(r);
    return 0;
}

int
kct_authz_destroy()
{
    REQUESTOR *reqr, *save_reqr;
    OK_SVC *okd;

    if (NULL == ok_reqrs)
	return 0;

    reqr = ok_reqrs;
    while (NULL != reqr) {
	save_reqr = reqr->next;
	kct_free_reqr(reqr);
	reqr = save_reqr;
    }
    ok_reqrs = NULL;
    return 0;
}

int
kct_authz_reinit(char *filename)
{
    kct_authz_destroy();
    kct_authz_init(filename);
    return 0;
}

int
kct_authz_dump_table()
{
    REQUESTOR *r;
    OK_SVC *s;

    if (NULL == ok_reqrs) {
	kct_log(KCTLOG_ALWAYS, "kct_authz_dump_table: table is empty\n");
	return 0;
    }

    for (r = ok_reqrs; r; r = r->next) {
	kct_log(KCTLOG_ALWAYS, "REQUESTOR %s\n", r->pname);

	kct_log(KCTLOG_ALWAYS, "SERVICES  ");
	for (s = r->svcs; s; s = s->next) {
	    kct_log(KCTLOG_ALWAYS, "%s/%s\n", s->sname, s->sinst);
	}
    }
    return 0;
}

/*
 * Determine whether the requestor is authorized to obtain
 * the target service ticket for the target user.
 */

int
kct_authz_verify(char *requestor, const char *service_string)
{

    REQUESTOR *reqr;
    OK_SVC *svc;
    char   *svc_inst;
    int     svc_len;
    int     svc_ok;
    char    service[KCT_MAX_BUF_SIZE + 1];
    char    service_instance[KCT_MAX_BUF_SIZE + 1];
    char   *slash;
    int     instance_specified = 0;

    kct_log(KCTLOG_INFO,
	    "kct_authz_verify: requestor='%s' service='%s'\n",
	    requestor, service_string);

    service_instance[0] = '\0';
    slash = strchr(service_string, '/');
    if (NULL != slash) {
	instance_specified = 1;
	strncpy(service_instance, slash + 1, sizeof(service_instance));
	service_instance[sizeof(service_instance) - 1] = '\0';
	strncpy(service, service_string, slash - service_string);
	service[slash - service_string] = '\0';
    } else {
	strncpy(service, service_string, sizeof(service));
	service[sizeof(service) - 1] = '\0';
    }

    svc_ok = 0;
    for (reqr = ok_reqrs; reqr && !svc_ok; reqr = reqr->next) {
	if (0 == strcmp(reqr->pname, requestor)) {
	    /*
	     * The requestor has an entry.  Now see what services
	     * it is allowed to issue tickets for.
	     */
	    for (svc = reqr->svcs; svc && !svc_ok; svc = svc->next) {
		/*
		 *  Matching rules:
		 *
		 *    The service names must match.
		 *
		 *    If the request specified an instance,
		 *      then there must be an instance in
		 *      the acl which is either a wildcard
		 *      or an exact match.
		 *
		 *    If the request did not specify an instance,
		 *      then there must be a wildcard instance
		 *      in the acl, or no instance specified.
		 */
		if (0 == strcmp(svc->sname, service)) {
		    int     inst_wild = 0;
		    int     inst_match = 0;

		    if (svc->sinst) {
			if (!strcmp(svc->sinst, "*"))
			    inst_wild = 1;
			else if (!strcmp(svc->sinst, service_instance))
			    inst_match = 1;
		    }
		    if (instance_specified) {
			if (svc->sinst && (inst_wild || inst_match))
			    svc_ok = 1;
		    } else if (!svc->sinst || inst_wild) {
			svc_ok = 1;
		    }
		}
	    }
	}
    }
    kct_log(KCTLOG_ALWAYS,
	    "KCT_REQ: Request from '%s' for service '%s/%s' %s\n",
	    requestor, service, service_instance,
	    svc_ok ? "granted" : "DENIED");
    if (0 == svc_ok) {
	return KCT_ERR_UNAUTHORIZED;
    }
    return KCT_ERR_NO_ERROR;
}
