/*
 * Copyright  ©  2005
 * The Regents of the University of Michigan
 * ALL RIGHTS RESERVED
 *
 * permission is granted to use, copy, create derivative works 
 * and redistribute this software and such derivative works 
 * for any purpose, so long as the name of the university of 
 * michigan is not used in any advertising or publicity 
 * pertaining to the use or distribution of this software 
 * without specific, written prior authorization.  if the 
 * above copyright notice or any other identification of the 
 * university of michigan is included in any copy of any 
 * portion of this software, then the disclaimer below must 
 * also be included.
 *
 * this software is provided as is, without representation 
 * from the university of michigan as to its fitness for any 
 * purpose, and without warranty by the university of 
 * michigan of any kind, either express or implied, including 
 * without limitation the implied warranties of 
 * merchantability and fitness for a particular purpose. the 
 * regents of the university of michigan shall not be liable 
 * for any damages, including special, indirect, incidental, or 
 * consequential damages, with respect to any claim arising 
 * out of or in connection with the use of the software, even 
 * if it has been or is hereafter advised of the possibility of 
 * such damages.
 */

#include "kct.h"
#include "kct_authz.h"

/*	#define DEBUG 	*/
/*	#define DEBUG_DIGEST	*/
/*	#define DEBUG_AFS_TKT	*/
/*	#define DEBUG_TIMESTAMP	*/

/*	#define CLIENT_NAME_LOOKUP */

/*
 * DEFAULTS for configurable options ...
 */

int     kct_port = 5039;
int     single_process = 0;
char    kct_keytab[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/kct.keytab";
char    tts_keytab[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/kct.keytab";
char    conf_file_name[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/kct.conf";
char    kct_acl_file[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/kct.acl";
char    kct_princ_name[KCT_MAX_BUF_SIZE + 1] = "kct";
char    tts_princ_name[KCT_MAX_BUF_SIZE + 1] = "tts";
char    kct_version[KCT_MAX_BUF_SIZE + 1] = "kct version 2";
char    ca_file[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/cert/ca-bundle.crt";
char    ca_path[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/cert";
char    log_file_name[MAX_KEYTAB_NAME_LEN + 1] = "/var/kct/logs/kct";
int     handshake_valid_time = KCT_DEFAULT_HANDSHAKE_VALID_TIME;
char    kct_implicit_service_name[KCT_MAX_BUF_SIZE + 1] = "web_server";
int     ignoreNonExistingPrincipals = 0;

typedef struct saved_handshake
{
    int     version;

    unsigned char *master_key;
    unsigned int master_key_length;

    unsigned char *c_h;		/* Client Hello */
    unsigned int c_h_l;		/* Client Hello length */

    unsigned char *s_h;		/* Server Hello */
    unsigned int s_h_l;		/* Server Hello length */

    unsigned char *s_c;		/* Server Certificate */
    unsigned int s_c_l;		/* Server Certificate length */

    unsigned char *s_c_r;	/* Server Certificate Request */
    unsigned int s_c_r_l;	/* Server Certificate Request length */

    unsigned char *s_k_e;	/* Server Key Exchange */
    unsigned int s_k_e_l;	/* Server Key Exchange length */

    unsigned char *c_c;		/* Client Certificate */
    unsigned int c_c_l;		/* Client Certificate length */

    unsigned char *c_k_e;	/* Client Key Exchange */
    unsigned int c_k_e_l;	/* Client Key Exchange length */

    unsigned char *c_v;		/* Client Verify */
    unsigned int c_v_l;		/* Client Verify length */

} SAVED_HANDSHAKE;

void   *handle;

static int handle_request(kct_context *);
static int handle_ap_req(kct_context *);
static int receive_ticket(kct_context *);
static int receive_handshake(kct_context *, SAVED_HANDSHAKE *,
			     unsigned char *);
static int receive_request(kct_context * context, int *key_length, char **key);

static int verify_handshake(kct_context *, SAVED_HANDSHAKE *);
static int generate_service_ticket(kct_context *, char *, char *,
				   unsigned char **, int *);
static int send_reply(int, long, unsigned char *, int);

static void cert_verify_mac(unsigned char *, int, EVP_MD_CTX *,
			    unsigned char *);
static void calculate_digest(SAVED_HANDSHAKE *, unsigned char *);

static int retrieve_cert(int, unsigned char *, unsigned int, EVP_PKEY **,
			 int *, X509 **);
static int verify_cert(X509 *);
static unsigned char *read_message(unsigned char **, unsigned int *);
static int verify_client_signature(unsigned char *, unsigned int,
				   unsigned char *, EVP_PKEY *);
static int verify_client_timestamp(unsigned char *);
static int retrieve_implicit_identity(kct_context *, char **, X509 *);
static int retrieve_explicit_identity(kct_context *, char **, char **, X509 *);
static int verify_server_identity(kct_context *);
static void process_exit(int, const struct sigaction *, struct sigaction *);
static int cb_local(int ok, X509_STORE_CTX * ctx);
static void usage();
static void read_config_file(char *, int);
static void initialize_handshake(SAVED_HANDSHAKE *);
static void initialize_kct_context(kct_context *);

void
usage()
{
    fprintf(stderr, "kct [list of parameters] or -f [configuration file]\n");
    fprintf(stderr, "List of parameters:\n");
    fprintf(stderr, "-s (enables single process mode of operation)\n");
    fprintf(stderr, "-port [port]\n");
    fprintf(stderr, "-k [kct keytab file name]\n");
    fprintf(stderr, "-pname [kct principal name]\n");
    fprintf(stderr, "-tts_keytab [tts keytab file name]\n");
    fprintf(stderr, "-tts_pname [tts principal name]\n");
    fprintf(stderr, "-CAfile [CA bundle file name]\n");
    fprintf(stderr, "-CApath [path to dir with certs]\n");
    fprintf(stderr, "-l [log file name]\n");
    fprintf(stderr, "-d [debug level]\n");
    fprintf(stderr, "-i (will allow KCT to issue tickets to "
	    "non-existing principals)\n");

    exit(1);
}

/*
 * Signal handling routine.  Sets the flag to indicate
 * that a HUP signal needs processing when appropriate
 * to do so.
 */
volatile int need_hup_processing = 0;

void
cleanup_on_ctlc(int signum)
{
    kct_log(KCTLOG_DEBUG, "SIGINT received\n");
    if (single_process)
	kadm5_destroy(handle);
    kct_authz_destroy();
    kct_close_log();
    exit(1);
}

void
process_hup(int signum)
{
    kct_log(KCTLOG_DEBUG, "SIGHUP, received\n");
    need_hup_processing = 1;
    return;
}

void
read_config_file(char *filename, int ignore_errors)
{
    FILE   *configfile = NULL;
    char    op[KCT_MAX_BUF_SIZE], value[KCT_MAX_BUF_SIZE];

    configfile = fopen(filename, "r");
    if (configfile == NULL) {
	if (ignore_errors == 0) {
	    fprintf(stderr, "couldn't open configuration file %s: %s\n",
		    filename, strerror(errno));
	    exit(1);
	} else
	    return;
    }

    while (fscanf(configfile, "%s %s", op, value) != EOF) {
	if (!strcmp(op, "KCT_PORT")) {
	    kct_port = atoi(value);
	} else if (!strcmp(op, "KCT_VERSION")) {
	    strncpy(kct_version, value, sizeof(kct_version));
	    kct_version[sizeof(kct_version) - 1] = '\0';
	} else if (!strcmp(op, "KCT_KEYTAB")) {
	    strncpy(kct_keytab, value, sizeof(kct_keytab));
	    kct_keytab[sizeof(kct_keytab) - 1] = '\0';
	} else if (!strcmp(op, "KCT_IMPLICIT_SERVICE_NAME")) {
	    strncpy(kct_implicit_service_name, value,
		    sizeof(kct_implicit_service_name));
	    kct_implicit_service_name[sizeof(kct_implicit_service_name) - 1] =
		'\0';
	} else if (!strcmp(op, "TTS_KEYTAB")) {
	    strncpy(tts_keytab, value, sizeof(tts_keytab));
	    tts_keytab[sizeof(tts_keytab) - 1] = '\0';
	} else if (!strcmp(op, "KCT_ACL_FILE")) {
	    strncpy(kct_acl_file, value, sizeof(kct_acl_file));
	    kct_acl_file[sizeof(kct_acl_file) - 1] = '\0';
	} else if (!strcmp(op, "KCT_PRINC_NAME")) {
	    strncpy(kct_princ_name, value, sizeof(kct_princ_name));
	    kct_princ_name[sizeof(kct_princ_name) - 1] = '\0';
	} else if (!strcmp(op, "TTS_PRINC_NAME")) {
	    strncpy(tts_princ_name, value, sizeof(tts_princ_name));
	    tts_princ_name[sizeof(tts_princ_name) - 1] = '\0';
	} else if (!strcmp(op, "CA_FILE")) {
	    strncpy(ca_file, value, sizeof(ca_file));
	    ca_file[sizeof(ca_file) - 1] = '\0';
	} else if (!strcmp(op, "CA_PATH")) {
	    strncpy(ca_path, value, sizeof(ca_path));
	    ca_path[sizeof(ca_path) - 1] = '\0';
	} else if (!strcmp(op, "DEBUG_LEVEL")) {
	    kct_debug_level = atoi(value);
	} else if (!strcmp(op, "LOG_FILE")) {
	    strncpy(log_file_name, value, sizeof(log_file_name));
	    log_file_name[sizeof(log_file_name) - 1] = '\0';
	} else if (!strcmp(op, "SINGLE_PROCESS")) {
	    single_process = atoi(value);
	} else if (!strcmp(op, "HANDSHAKE_VALID_TIME")) {
	    handshake_valid_time = atoi(value);
	} else {
	    fprintf(stderr, "unknown option %s\n", op);
	    goto end;
	}
    }

  end:
    fclose(configfile);
}

/*
 * Return a string describing a given error code
 */
static char *
error_to_string(int code)
{
    switch (code) {
    case KCT_ERR_NO_ERROR:
	return "No error";
	break;
    case KCT_ERR_INTERNAL_ERROR:
	return "Miscellaneous internal error";
	break;
    case KCT_ERR_RD_PRIV:
	return "Unable to decrypt private message";
	break;
    case KCT_ERR_INVALID_HANDSHAKE:
	return "Invalid handshake";
	break;
    case KCT_ERR_OLD_HANDSHAKE:
	return "Time skew in handshake";
	break;
    case KCT_ERR_INVALID_ID:
	return "Invalid ID found in handshake";
	break;
    case KCT_ERR_REALM_MISMATCH:
	return "Realm mismatch";
	break;
    case KCT_ERR_COMMUNICATION_ERROR:
	return "Communications error";
	break;
    case KCT_ERR_UNAUTHORIZED:
	return "Unauthorized web server request";
	break;
    case KCT_ERR_INTERNAL_KERBEROS_ERROR:
	return "Internal Kerberos-related error";
	break;
    case KCT_ERR_UNKNOWN_PRINCIPAL:
	return "Requested service principal does not exist";
	break;
    case KCT_ERR_BAD_PRINCIPAL:
	return "Bad principal";
    default:
	return "Unknown error";
	break;
    }
}

/*
 * KCT: 
 * 1. Process configuration parameters either from the console or 
 *    a configuration file
 * 2. Create a listening socket for incoming requests
 * 3. Upon receiving a message, fork off a new process to handle the
 *    request.  Requests are handled by the function 'handle_request'
 */

int
main(int argc, char *argv[])
{

    int     sock, csock;
    struct sockaddr_in srvaddr, clntaddr;
    struct hostent *phost = NULL;
    int     size = 1;
    kct_context *request = NULL;
    int     badop = 0;
    krb5_error_code retval = 0;
    int     rc = 0;

    argc--;
    argv++;
    /* Read default config file, ignore open errors */
    read_config_file(conf_file_name, 1);
    while (argc >= 1) {
	if (!strcmp(*argv, "-port")) {
	    if (--argc < 1)
		goto bad;
	    kct_port = atoi(*(++argv));
	} else if (!strcmp(*argv, "-k")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(kct_keytab, *(++argv), sizeof(kct_keytab));
	    kct_keytab[sizeof(kct_keytab) - 1] = '\0';
	} else if (!strcmp(*argv, "-version_string")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(kct_version, *(++argv), sizeof(kct_version));
	    kct_version[sizeof(kct_version) - 1] = '\0';
	} else if (!strcmp(*argv, "-tts_keytab")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(tts_keytab, *(++argv), sizeof(tts_keytab));
	    tts_keytab[sizeof(tts_keytab) - 1] = '\0';
	} else if (!strcmp(*argv, "-implicit_service_name")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(kct_implicit_service_name, *(++argv),
		    sizeof(kct_implicit_service_name));
	    kct_implicit_service_name[sizeof(kct_implicit_service_name) - 1] =
		'\0';
	} else if (!strcmp(*argv, "-pname")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(kct_princ_name, *(++argv), sizeof(kct_princ_name));
	    kct_princ_name[sizeof(kct_princ_name) - 1] = '\0';
	} else if (!strcmp(*argv, "-tts_pname")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(tts_princ_name, *(++argv), sizeof(tts_princ_name));
	    tts_princ_name[sizeof(tts_princ_name) - 1] = '\0';
	} else if (!strcmp(*argv, "-CAfile")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(ca_file, *(++argv), sizeof(ca_file));
	    ca_file[sizeof(ca_file) - 1] = '\0';
	} else if (!strcmp(*argv, "-CApath")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(ca_path, *(++argv), sizeof(ca_path));
	    ca_path[sizeof(ca_path) - 1] = '\0';
	} else if (!strcmp(*argv, "-l")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(log_file_name, *(++argv), sizeof(log_file_name));
	    log_file_name[sizeof(log_file_name) - 1] = '\0';
	} else if (!strcmp(*argv, "-d")) {
	    if (--argc < 1)
		goto bad;
	    kct_debug_level = atoi(*(++argv));
	} else if (!strcmp(*argv, "-f")) {
	    if (--argc < 1)
		goto bad;
	    strncpy(conf_file_name, *(++argv), sizeof(conf_file_name));
	    conf_file_name[sizeof(conf_file_name) - 1] = '\0';
	    read_config_file(conf_file_name, 0);
	} else if (!strcmp(*argv, "-s")) {
	    single_process = 1;
	} else if (!strcmp(*argv, "-i")) {
	    ignoreNonExistingPrincipals = 1;
	} else {
	    badop = 1;
	    fprintf(stderr, "unknown option %s\n", *argv);
	}

	argc--;
	argv++;
    }

    if (badop) {
      bad:
	usage();
    }

    kct_open_log(log_file_name);

    kct_log(KCTLOG_INFO, "LOG_FILE:              %s\n", log_file_name);
    kct_log(KCTLOG_INFO, "KCT_PORT:              %d\n", kct_port);
    kct_log(KCTLOG_INFO, "KCT_KEYTAB:            %s\n", kct_keytab);
    kct_log(KCTLOG_INFO, "TTS_KEYTAB:            %s\n", tts_keytab);
    kct_log(KCTLOG_INFO, "KCT_ACL_FILE:          %s\n", kct_acl_file);
    kct_log(KCTLOG_INFO, "KCT_PRINC_NAME:        %s\n", kct_princ_name);
    kct_log(KCTLOG_INFO, "TTS_PRINC_NAME:        %s\n", tts_princ_name);
    kct_log(KCTLOG_INFO, "CA_FILE:               %s\n", ca_file);
    kct_log(KCTLOG_INFO, "CA_PATH:               %s\n", ca_path);
    kct_log(KCTLOG_INFO, "DEBUG_LEVEL:           %d\n", kct_debug_level);
    kct_log(KCTLOG_INFO, "SINGLE_PROCESS:        %d\n", single_process);
    kct_log(KCTLOG_INFO, "HANDSHAKE_VALID_TIME:  %d\n", handshake_valid_time);

    if (kct_authz_init(kct_acl_file)) {
	kct_log(KCTLOG_ALWAYS,
		"Error initializing authorization table from file %s\n",
		kct_acl_file);
	exit(1);
    }

    if (single_process) {
	if ((retval =
	     kadm5_init(kct_princ_name, NULL, KADM5_ADMIN_SERVICE, NULL,
			KADM5_STRUCT_VERSION, KADM5_API_VERSION_2, &handle))) {
	    com_err("kct_server", retval, "initializing kadm5 library");
	    exit(1);
	}
    }

    /* Setup signal handling */
    if (SIG_ERR == signal(SIGPIPE, SIG_IGN)) {
	kct_log(KCTLOG_ALWAYS, "setting up SIGPIPE to ignore: %s\n",
		strerror(errno));
	return -1;
    }
    if (SIG_ERR == signal(SIGHUP, process_hup)) {
	kct_log(KCTLOG_ALWAYS, "setting up SIGHUP handler: %s\n",
		strerror(errno));
	return -1;
    }

    if (SIG_ERR == signal(SIGINT, cleanup_on_ctlc)) {
	kct_log(KCTLOG_ALWAYS, "setting up SIGINT handler: %s\n",
		strerror(errno));
	return -1;
    }

    bzero(&srvaddr, sizeof(srvaddr));
    bzero(&clntaddr, sizeof(clntaddr));

    if ((sock = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
	kct_log(KCTLOG_ALWAYS, "opening stream socket: %s\n", strerror(errno));
	return -1;
    }

    if (setsockopt
	(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &size, sizeof(size)) < 0) {
	kct_log(KCTLOG_ALWAYS, "setsockopt failed: %s\n", strerror(errno));
	return -1;
    }

    srvaddr.sin_family = AF_INET;
    srvaddr.sin_addr.s_addr = htonl(INADDR_ANY);
    srvaddr.sin_port = htons(kct_port);

    if (bind(sock, (struct sockaddr *) &srvaddr, sizeof(srvaddr)) == -1) {
	kct_log(KCTLOG_ALWAYS, "binding stream socket: %s\n", strerror(errno));
	close(sock);
	return -1;
    }

    listen(sock, KCT_QUEUE_LEN);

    size = sizeof(clntaddr);

    kct_log(KCTLOG_ALWAYS, "Beginning execution with debug level %d\n",
	    kct_debug_level);

    while (1) {

	if (need_hup_processing) {
	    kct_log(KCTLOG_ALWAYS, "handling sighup request\n");
	    kct_reopen_log(log_file_name);
	    kct_authz_reinit(kct_acl_file);
	    need_hup_processing = 0;
	    /* Re-install signal handler */
	    if (SIG_ERR == signal(SIGHUP, process_hup)) {
		kct_log(KCTLOG_ALWAYS, "setting up SIGHUP handler: %s\n",
			strerror(errno));
		exit(-1);
	    }
	}

	request = (kct_context *) get_memory(1, sizeof(kct_context));
	initialize_kct_context(request);

	if ((request->ws_sock = accept(sock, (struct sockaddr *) &clntaddr,
				       &size)) == -1) {
	    /* Don't print error message if we were interrupted */
	    if (errno != EINTR) {
		kct_log(KCTLOG_ALWAYS, "accept failed: %s\n", strerror(errno));
		exit(-1);
	    }
	    free_memory(request);
	    continue;
	}
#ifdef CLIENT_NAME_LOOKUP
	if ((phost = gethostbyaddr((char *) &clntaddr.sin_addr,
				   sizeof(struct in_addr),
				   clntaddr.sin_family)) == NULL) {
	    strncpy(request->ws_host, inet_ntoa(clntaddr.sin_addr),
		    sizeof(request->ws_host));
	    request->ws_host[sizeof(request->ws_host) - 1] = '\0';
	} else {
	    strncpy(request->ws_host, phost->h_name, sizeof(request->ws_host));
	    request->ws_host[sizeof(request->ws_host) - 1] = '\0';
	}
#else
	strncpy(request->ws_host, inet_ntoa(clntaddr.sin_addr),
		sizeof(request->ws_host));
	request->ws_host[sizeof(request->ws_host) - 1] = '\0';
#endif
	request->ws_port = htons(clntaddr.sin_port);

	kct_log(KCTLOG_ALWAYS, "KCT_REQ: from %s[%d]\n",
		request->ws_host, request->ws_port);

	if (single_process) {
	    handle_request(request);
	} else {
	    switch (fork()) {

	    case -1:		/* fork failed */
		kct_log(KCTLOG_ALWAYS, "fork failed: %s\n", strerror(errno));
		close(request->ws_sock);
		close(sock);
		free_memory(request);
		exit(-1);

	    case 0:		/* child process */
		handle_request(request);
		close(sock);
		kct_log(KCTLOG_DEBUG, "child process will exit...\n");
		exit(1);

	    default:		/* parent process */
		close(request->ws_sock);
		free_memory(request);

		/*
		 * Try to reap child processes that has already exited,
		 * but don't wait if there are none.
		 */

		rc = -1;
		while (rc != 0) {
		    rc = waitpid((pid_t) - 1, NULL, WNOHANG);
		    kct_log(KCTLOG_DEBUG,
			    "waitpid returned with rc %d errno %d\n",
			    rc, errno);
		}
		break;
	    }
	}
    }
    close(sock);

    kct_close_log();

  cleanup:
    if (single_process)
	kadm5_destroy(handle);

}

/*
 * This the main function that handles KCT requests.
 * Before processing a request, we set up a timer that will 
 * interrupt this process in KCT_PROCESS_TIMEOUT amount of time and
 * kill the process.  This was added to prevent dangling processes.
 *
 * 1. KCT handles kerberos AP_REQ. If AP_REQ fails, it returns
 * from this function and closes the connection to the client.
 *
 * 2. KCT reads a version: a request can submit either a handshake
 * of a Kerberos ticket.
 * 
 * 3a. If this is a handshake-based request, KCT receives SSL handshake bytes. 
 * SSL_session key is sent encrypted so it calls krb5_rd_priv.  It is 
 * encrypted with the session key. If decryption of this message fails, KCT 
 * returns an error message back to the client. 
 *
 * 4a. After a successful read of the handshake, KCT calls the
 * procedure to verify the handshake. Verification consists of
 * the following:
 * - matches identity in server's certificate against the identity
 *   used during the AP_REQ
 * - verifies signature of both client's and server's certificate.
 *   It checks that the certificate has been signed by a CA that
 *   the KCT trusts
 * - verifies client's signature inside of SSL handhshake (this
 *   code is exactly the same as the verification code in openssl)
 * - checks that the client's timestamp in the client hello message
 *   is not too old
 * If at any point verification fails, the KCT sends an error
 * message back to the client.
 *
 * 3b. If this is a ticket-based request, KCT receives/verifies an AP_REQ.
 * It also reads (a list of) service_name:lifetime pair, encrypted under
 * the Kerberos session key.
 *
 * 4b..... 
 * 
 * 5. After the handshake (or ticket) has been successfully verified, the KCT
 * calls the routine to create a service ticket.  At this point if
 * something goes wrong, then it's an internal error and the client
 * is notified. A ticket is created for the identity inside of the
 * client's certificate.
 *
 * 6. If the KCT was successful in creating a service ticket, it
 * sends it back to the client with KCT_ERR_NO_ERROR error message.
 */

static int
handle_request(kct_context * context)
{

    krb5_error_code retval = 0;
    SAVED_HANDSHAKE *sh = NULL;
    struct itimerval itimer;
    struct sigaction act;
    unsigned char *reply = NULL;
    int     len = 0, i = 0;

    unsigned xmitlen = 0, version = 0;
    unsigned char *p = NULL, *recv_buf = NULL;

    /*
     * Set up a timer. It will terminate this process after 5mins
     * This is done so that we don't have hanging processes.
     */
    sigemptyset(&act.sa_mask);
    act.sa_handler = (void (*)()) process_exit;
    act.sa_flags = SA_RESTART;
    sigaction(SIGVTALRM, &act, NULL);

    itimer.it_value.tv_sec = KCT_PROCESS_TIMEOUT;
    itimer.it_value.tv_usec = 0;
    itimer.it_interval.tv_sec = KCT_PROCESS_TIMEOUT;
    itimer.it_interval.tv_usec = 0;

    if (setitimer(ITIMER_VIRTUAL, &itimer, NULL) != 0) {
	kct_log(KCTLOG_ALWAYS,
		"handle_request: unable to set the interval timer: "
		"%s\n", strerror(errno));
	free_memory(context);
	return KCT_ERR_INTERNAL_ERROR;
    }

    /*
     * Initialize krb structures and receive/validate AP_REQ
     * Do a check on requester's identity
     */
    retval = handle_ap_req(context);
    if (KCT_ERR_NO_ERROR != retval) {
	goto send_error;
    }

    /*
     * Receive header & version
     */
    if (recvblock(context->ws_sock, (unsigned char *) &xmitlen,
		  sizeof(xmitlen)) < 0) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: socket closed: %s\n",
		strerror(errno));
	retval = KCT_ERR_COMMUNICATION_ERROR;
	goto send_error;
    }
    xmitlen = ntohl(xmitlen);

    /* check that header is within reasonable range */
    if (xmitlen <= 0 || xmitlen > KCT_MAX_MEMORY_ALLOC) {
	kct_log(KCTLOG_ALWAYS,
		"recieve_handshake: requested more "
		"memory (%d) than allowed (%d)\n",
		xmitlen, KCT_MAX_MEMORY_ALLOC);
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto send_error;
    }
#ifdef DEBUG
    kct_log(KCTLOG_DEBUG, "whole receive buffer: %d\n", xmitlen);
#endif

    p = recv_buf = (unsigned char *)
	get_memory(xmitlen, sizeof(unsigned char));
    if (recvblock(context->ws_sock, (unsigned char *) recv_buf, xmitlen) < 0) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: socket closed (2): %s\n",
		strerror(errno));
	retval = KCT_ERR_COMMUNICATION_ERROR;
	goto send_error;
    }

    memcpy(&version, p, sizeof(unsigned long));
    version = ntohl(version);

    switch (version) {
    case TLS1_VERSION:
    case SSL3_VERSION:
	kct_log(KCTLOG_DEBUG,
		"receive_handshake: handshake-based KCT request (%d)\n",
		version);

	/*
	 * Allocate memory for the ssl handshake that we are about to receive
	 * from the web server
	 */
	sh = (SAVED_HANDSHAKE *) get_memory(1, sizeof(SAVED_HANDSHAKE));
	initialize_handshake(sh);

	/*
	 * Receive and parse the handshake
	 */
	retval = receive_handshake(context, sh, recv_buf);
	if (KCT_ERR_NO_ERROR != retval) {
	    goto send_error;
	}

	/*
	 * Verify the handshake
	 */
	retval = verify_handshake(context, sh);
	if (KCT_ERR_NO_ERROR != retval) {
	    goto send_error;
	}

	break;
    case 777:
	kct_log(KCTLOG_DEBUG,
		"receive_handshake: ticket-based KCT request (%d)\n", version);
	if ((retval = receive_ticket(context)) != KCT_ERR_NO_ERROR)
	    goto send_error;

	if ((retval =
	     receive_request(context, NULL, NULL)) != KCT_ERR_NO_ERROR)
	    goto send_error;

	break;
    default:
	kct_log(KCTLOG_ALWAYS,
		"receive_handshake: invalid protocol version (%d)\n", version);
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto send_error;
    }

    for (i = 0; i < context->num_services; i++) {
	char   *service_id = context->services[i].service_name;
	char   *lifetime = context->services[i].lifetime;

	/*
	 * Verify that the requestor is authorized to issue tickets
	 * for each of the requested services
	 */
	retval = kct_authz_verify(context->ws_id, service_id);
	if (KCT_ERR_NO_ERROR != retval) {
	    goto send_error;
	}

	kct_log(KCTLOG_DEBUG, "KCT_REQ: Requesting '%s' for '%s' lifetime %s "
		"(from %s@%s)\n", context->client_id, service_id, lifetime,
		context->ws_id, context->ws_realm);

	/*
	 * Create and send service ticket
	 *
	 * The ticket will be issued for the identity inside of
	 * the client's certificate presented in the SSL handshake.
	 *
	 */
	retval = generate_service_ticket(context, service_id, lifetime,
					 &reply, &len);
	if (KCT_ERR_NO_ERROR != retval) {
	    kct_log(KCTLOG_ALWAYS, "KCT_REQ: Returning error %d, '%s'\n",
		    retval, error_to_string(retval));
	    goto send_error;
	}

	kct_log(KCTLOG_DEBUG, "!!!!!!!! about to send the ticket (%d)\n", len);

	send_reply(context->ws_sock, retval, reply, len);

	kct_log(KCTLOG_DEBUG, "!!!!!!!! done sending the ticket\n");

	if (i + 1 == context->num_services)
	    goto cleanup;
    }

  send_error:
    send_reply(context->ws_sock, retval, reply, len);

  cleanup:

    for (i = 0; i < context->num_services; i++) {
	if (context->services[i].service_name)
	    free_memory(context->services[i].service_name);
	if (context->services[i].lifetime)
	    free_memory(context->services[i].lifetime);
    }
    if (context->services)
	free_memory(context->services);

    if (recv_buf)
	free_memory(recv_buf);

    if (reply && len > 0)
	free_memory(reply);
    if (sh != NULL) {
	if (sh->c_h)
	    free_memory(sh->c_h);
	if (sh->s_h)
	    free_memory(sh->s_h);
	if (sh->s_c)
	    free_memory(sh->s_c);
	if (sh->s_c_r)
	    free_memory(sh->s_c_r);
	if (sh->s_k_e && sh->s_k_e_l > 0)
	    free_memory(sh->s_k_e);
	if (sh->c_c)
	    free_memory(sh->c_c);
	if (sh->c_k_e)
	    free_memory(sh->c_k_e);
	if (sh->c_v)
	    free_memory(sh->c_v);
	if (sh->master_key && sh->master_key_length > 0)
	    free_memory(sh->master_key);
	free_memory(sh);
    }

    if (context->ws_ticket)
	krb5_free_ticket(context->krb_context, context->ws_ticket);

    if (context->ws_auth_context)
	krb5_auth_con_free(context->krb_context, context->ws_auth_context);

    if (context->tkt_auth_context)
	krb5_auth_con_free(context->tkt_context, context->tkt_auth_context);

    if (context->krb_context)
	krb5_free_context(context->krb_context);

    if (context->tkt_context)
	krb5_free_context(context->tkt_context);

    if (context->client_id)
	free_memory(context->client_id);
    if (context->client_realm)
	free_memory(context->client_realm);
    if (context->ws_id)
	free_memory(context->ws_id);
    if (context->ws_realm)
	free_memory(context->ws_realm);
    if (context->ws_sock)
	close(context->ws_sock);

    if (context)
	free_memory(context);
    context = NULL;

    kct_log(KCTLOG_DEBUG, "returning from handle_request retval = %d\n",
	    retval);
    return retval;
}

static int
handle_ap_req(kct_context * context)
{

    char   *client_name = NULL;
    krb5_error_code retval = 0;
    krb5_enctype enctype;
    krb5_keytab_entry ktent;
    krb5_keytab keytab;
    krb5_principal server;

    kct_log(KCTLOG_DEBUG, "handle_ap_req: start\n");

    if ((retval = krb5_init_context(&context->krb_context)) != 0) {
	com_err("kct_server", retval,
		"handle_ap_req: krb5_init_context() failed");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_auth_con_init(context->krb_context,
				     &context->ws_auth_context)) != 0) {
	com_err("kct_server", retval,
		"handle_ap_req: krb5_auth_con_init() failed");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_kt_resolve(context->krb_context, kct_keytab,
				  &keytab)) != 0) {
	com_err("kct_server", retval,
		"handle_ap_req: krb5_kt_resolve() failed for file %s",
		kct_keytab);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_parse_name(context->krb_context, kct_princ_name,
				  &server)) != 0) {
	com_err("kct_server", retval,
		"handle_ap_req: krb5_parse_name() failed "
		"unable to generate Kerberos principal %s", kct_princ_name);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto server_err;
    }

    if ((retval =
	 krb5_recvauth(context->krb_context, &context->ws_auth_context,
		       (krb5_pointer) & context->ws_sock, kct_version, server,
		       0, keytab, &context->ws_ticket)) != 0) {
	com_err("kct_server", retval, "handle_ap_req: krb5_recvauth() failed");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto sendauth_err;
    }

    if (context->ws_ticket->enc_part2 == NULL) {
	enctype = context->ws_ticket->enc_part.enctype;
	if ((retval = krb5_kt_get_entry(context->krb_context, keytab,
					context->ws_ticket->server,
					context->ws_ticket->enc_part.kvno,
					enctype, &ktent)) != 0) {
	    com_err("kct_server", retval,
		    "handle_ap_req: krb5_kt_get_entry() failed: "
		    "unable to retrieve a keytab entry");
	    retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	    goto sendauth_err;
	}

	if ((retval = krb5_decrypt_tkt_part(context->krb_context, &ktent.key,
					    context->ws_ticket)) != 0) {
	    com_err("kct_server", retval,
		    "handle_ap_req: krb5_decrypt_tkt_part() failed");
	    retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	    goto sendauth_err;
	}

	krb5_free_keytab_entry_contents(context->tkt_context, &ktent);
    }

    if ((retval = krb5_unparse_name(context->krb_context,
				    context->ws_ticket->enc_part2->client,
				    &client_name)) != 0) {
	com_err("kct_server", retval,
		"handle_ap_req: krb5_unparse_name() failed");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto cleanup;
    }

    context->ws_id = (char *) get_memory(KCT_MAX_ID_LEN, sizeof(char));
    context->ws_realm = (char *) get_memory(KCT_MAX_REALM_LEN, sizeof(char));
    sscanf(client_name, "%[^@]@%s", context->ws_id, context->ws_realm);

    kct_log(KCTLOG_ALWAYS, "KCT_REQ: %s@%s at %s[%d]\n",
	    context->ws_id, context->ws_realm,
	    context->ws_host, context->ws_port);

#ifdef TGS_WS_REALM_MATCH
    /*
     * I'm not sure this check is needed.  I'm checking to make
     * sure that web server is in the same Kerberos realm as the
     * KCT. However, it might not be true in the case of cross-realm
     * authentication
     */
    {
	char   *default_realm = NULL;
	if ((retval = krb5_get_default_realm(context->krb_context,
					     &default_realm)) != 0) {
	    com_err("kct_server", retval,
		    "handle_ap_req: krb5_get_default_realm() failed");
	    retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	    if (default_realm)
		free_memory(default_realm);
	    goto cleanup;
	}

	if (strcasecmp(context->ws_realm, default_realm)) {
	    kct_log(KCTLOG_ALWAYS,
		    "handle_ap_req: received ticket for invalid realm %s\n",
		    context->ws_realm);
	    retval = KCT_ERR_REALM_MISMATCH;
	}

	/* KWC this should be a krb5 routine? XXX */
	free_memory(default_realm);
    }
#endif

  cleanup:
    if (client_name)
	free_memory(client_name);

  sendauth_err:
    if (server)
	krb5_free_principal(context->krb_context, server);

  server_err:
    if (keytab)
	krb5_kt_close(context->krb_context, keytab);

  context_err:
    kct_log(KCTLOG_DEBUG, "handle_ap_req: end retval = %d\n", retval);

    return retval;
}

static int
receive_handshake(kct_context * context, SAVED_HANDSHAKE * sh,
		  unsigned char *recv_buf)
{

    int     sock = context->ws_sock;

    unsigned long xmitlen;
    unsigned char *p = recv_buf;
    int     len = sizeof(unsigned long);

    krb5_data recv_data, recv_unenc_data;
    krb5_error_code retval = 0;

    krb5_address remote_kaddr;
    struct sockaddr_in remote_addr;
    size_t  addrlen;

    char   *services = NULL;
    char   *word = NULL;
    char   *service_name = NULL;
    char   *lifetime = NULL;
    int     i = 0;

    kct_log(KCTLOG_DEBUG, "receive_handshake: start\n");

    recv_data.data = NULL;
    recv_data.length = 0;
    recv_unenc_data.data = NULL;
    recv_unenc_data.length = 0;

    memcpy((char *) &sh->version, p, len);
    sh->version = ntohl(sh->version);
    p += len;

    /* check that version is valid */
    if (sh->version != TLS1_VERSION && sh->version != SSL3_VERSION) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid SSL version (%d)\n",
		sh->version);
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    /*
     * all but ServerKeyExchange messages must be present in the
     * handshake; presence of ServerKeyExchange is optional
     */
    sh->c_h = read_message(&p, &sh->c_h_l);
    if (sh->c_h_l <= 0 || sh->c_h == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid CLIENT_HELLO\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->s_h = read_message(&p, &sh->s_h_l);
    if (sh->s_h_l <= 0 || sh->s_h == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid SERVER_HELLO\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->s_c = read_message(&p, &sh->s_c_l);
    if (sh->s_c_l <= 0 || sh->s_c == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid SERVER_CERT\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->s_c_r = read_message(&p, &sh->s_c_r_l);
    if (sh->s_c_r_l <= 0 || sh->s_c_r == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid SERVER_CERT_REQ\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->s_k_e = read_message(&p, &sh->s_k_e_l);
    sh->c_c = read_message(&p, &sh->c_c_l);
    if (sh->c_c_l <= 0 || sh->c_c == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid CLIENT_CERT\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->c_k_e = read_message(&p, &sh->c_k_e_l);
    if (sh->c_k_e_l <= 0 || sh->c_k_e == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid CLIENT_KEY_EX\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }
    sh->c_v = read_message(&p, &sh->c_v_l);
    if (sh->c_v_l <= 0 || sh->c_v == NULL) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: invalid CLIENT_VERIFY\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    if (recvblock(sock, (unsigned char *) &recv_data.length,
		  sizeof(xmitlen)) < 0) {
	kct_log(KCTLOG_ALWAYS,
		"receive_handshake: Socket closed while reading length: %s\n",
		strerror(errno));
	return KCT_ERR_COMMUNICATION_ERROR;
    }

    recv_data.length = ntohl(recv_data.length);

    kct_log(KCTLOG_DEBUG, "non-handshake length %d\n", recv_data.length);

    /* check that header is within reasonable range */
    if (recv_data.length <= 0 || recv_data.length > KCT_MAX_MEMORY_ALLOC) {
	kct_log(KCTLOG_ALWAYS,
		"receive_handshake: requested more "
		"memory (%d) than allowed (%d)\n",
		recv_data.length, KCT_MAX_MEMORY_ALLOC);
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    recv_data.data = (char *)
	get_memory(recv_data.length, sizeof(unsigned char));

    if (recvblock(sock, (unsigned char *) recv_data.data, recv_data.length) <
	0) {
	kct_log(KCTLOG_ALWAYS,
		"recieve_handshake: Socket closed while reading data %s\n",
		strerror(errno));
	retval = KCT_ERR_COMMUNICATION_ERROR;
	goto recvdata_err;
    }

    if (retval = krb5_auth_con_setflags(context->krb_context,
					context->ws_auth_context,
					KRB5_AUTH_CONTEXT_DO_SEQUENCE)) {
	com_err("kct_server", retval, "unable to set authcontext flags");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    memset((char *) &remote_addr, 0, sizeof(remote_addr));
    addrlen = sizeof(remote_addr);
    if (getpeername(context->ws_sock, (struct sockaddr *) &remote_addr,
		    &addrlen) < 0) {
	kct_log(KCTLOG_ALWAYS, "receive_handshake: getpeername failed\n");
	retval = KCT_ERR_INTERNAL_ERROR;
	goto recvdata_err;
    }

    remote_kaddr.addrtype = ADDRTYPE_INET;
    remote_kaddr.length = sizeof(remote_addr.sin_addr);
    remote_kaddr.contents = (krb5_octet *) & remote_addr.sin_addr;

    if ((retval = krb5_auth_con_setaddrs(context->krb_context,
					 context->ws_auth_context, NULL,
					 &remote_kaddr)) != 0) {
	com_err("kct_server", retval, "unable to set krb addr");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    remote_kaddr.addrtype = ADDRTYPE_IPPORT;
    remote_kaddr.length = sizeof(remote_addr.sin_port);
    remote_kaddr.contents = (krb5_octet *) & remote_addr.sin_port;

    if ((retval = krb5_auth_con_setports(context->krb_context,
					 context->ws_auth_context, NULL,
					 &remote_kaddr)) != 0) {
	com_err("kct_server", retval, "unable to set krb port");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    if (retval = krb5_rd_priv(context->krb_context,
			      context->ws_auth_context,
			      &recv_data, &recv_unenc_data, NULL)) {
	com_err("kct_server", retval, "while verifying PRIV message");
	retval = KCT_ERR_RD_PRIV;
	goto cleanup;
    }

    p = (unsigned char *) recv_unenc_data.data;
    memcpy(&xmitlen, p, sizeof(xmitlen));
    p += sizeof(xmitlen);
    sh->master_key_length = ntohl(xmitlen);
    if (sh->master_key_length) {
	sh->master_key = (unsigned char *) get_memory(sh->master_key_length,
						      sizeof(unsigned char));
	memcpy(sh->master_key, p, sh->master_key_length);
	p += sh->master_key_length;
#ifdef DEBUG
	print_buffer(sh->master_key, sh->master_key_length, "master_key: ");
#endif
    } else {
	kct_log(KCTLOG_ALWAYS,
		"recieve_handshake: invalid SSL session key length\n");
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto cleanup;
    }

    memcpy(&xmitlen, p, sizeof(xmitlen));
    p += sizeof(xmitlen);
    context->num_services = ntohl(xmitlen);
    if (context->num_services) {
	memcpy(&xmitlen, p, sizeof(xmitlen));
	p += sizeof(xmitlen);
	len = ntohl(xmitlen);
    } else {
	kct_log(KCTLOG_ALWAYS,
		"recieve_handshake: invalid number of services\n");
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto cleanup;
    }

    kct_log(KCTLOG_DEBUG, " %d services len %d\n", context->num_services, len);

    services = (char *) get_memory(len, sizeof(char));
    memcpy(services, p, len);
    kct_log(KCTLOG_DEBUG, "request for services %s\n", services);

    context->services = (kct_services *)
	get_memory(context->num_services, sizeof(kct_services));

    p = services;
    word = (char *) get_memory(len, sizeof(char));
    service_name = (char *) get_memory(len, sizeof(char));
    lifetime = (char *) get_memory(len, sizeof(char));
    for (i = 0; i < context->num_services; i++) {
	memset(word, '\0', sizeof(word));
	memset(service_name, '\0', sizeof(service_name));
	memset(lifetime, '\0', sizeof(lifetime));

	if (!sscanf(p, "%s", word))
	    break;
	sscanf(word, "%[^:]:%s", service_name, lifetime);

	context->services[i].service_name = (char *)
	    get_memory(strlen(service_name) + 1, sizeof(char));
	strcpy(context->services[i].service_name, service_name);

	context->services[i].lifetime = (char *)
	    get_memory(strlen(lifetime) + 1, sizeof(char));
	strcpy(context->services[i].lifetime, lifetime);

	kct_log(KCTLOG_DEBUG, "service name %s lifetime %s\n",
		service_name, lifetime);
	p += strlen(word);
    }

    if (word)
	free_memory(word);
    if (service_name)
	free_memory(service_name);
    if (lifetime)
	free_memory(lifetime);
    if (services)
	free_memory(services);

  cleanup:
    if (recv_unenc_data.data)
	krb5_free_data_contents(context->krb_context, &recv_unenc_data);

  recvdata_err:
    if (recv_data.data)
	krb5_free_data_contents(context->krb_context, &recv_data);

    kct_log(KCTLOG_DEBUG, "receive_handshake: end retval = %d\n", retval);

    return retval;
}

static int
receive_ticket(kct_context * context)
{
    char   *client_name = NULL;
    krb5_error_code retval = 0;
    krb5_enctype enctype;
    krb5_keytab_entry ktent;
    krb5_keytab keytab;
    krb5_principal server;
    krb5_ticket *ticket = NULL;

    kct_log(KCTLOG_DEBUG, "receive_ticket: start\n");

    if ((retval = krb5_init_context(&context->tkt_context)) != 0) {
	com_err("kct_server", retval, "unable to initialize krb5");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_auth_con_init(context->tkt_context,
				     &context->tkt_auth_context)) != 0) {
	com_err("kct_server", retval, "unable to initialize auth_context\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_kt_resolve(context->tkt_context, tts_keytab,
				  &keytab)) != 0) {
	com_err("kct_server", retval, "unable to resolve keytable file %s",
		tts_keytab);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto context_err;
    }

    if ((retval = krb5_parse_name(context->tkt_context, tts_princ_name,
				  &server)) != 0) {
	com_err("kct_server", retval,
		"unable to generate service principal %s", tts_princ_name);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto server_err;
    }

    if ((retval =
	 krb5_recvauth(context->tkt_context, &context->tkt_auth_context,
		       (krb5_pointer) & context->ws_sock, kct_version, server,
		       0, keytab, &ticket)) != 0) {
	com_err("kct_server", retval, "recvauth failed: %d", retval);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvauth_err;
    }

    enctype = ticket->enc_part.enctype;
    if ((retval = krb5_kt_get_entry(context->tkt_context, keytab,
				    ticket->server,
				    ticket->enc_part.kvno, enctype,
				    &ktent)) != 0) {
	com_err("kct_server", retval, "unable to retrieve a keytab entry");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto getentry_err;
    }

    if ((retval = krb5_decrypt_tkt_part(context->tkt_context, &ktent.key,
					ticket)) != 0) {
	com_err("kct_server", retval, "unable to decrypt the ticket");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto getentry_err;
    }

    if ((retval = krb5_unparse_name(context->tkt_context,
				    ticket->enc_part2->client,
				    &client_name)) != 0) {
	com_err("kct_server", retval, "unable to parse the name in the tkt");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto cleanup;
    }

    context->client_id = (char *) get_memory(KCT_MAX_ID_LEN, sizeof(char));
    context->client_realm =
	(char *) get_memory(KCT_MAX_REALM_LEN, sizeof(char));
    sscanf(client_name, "%[^@]@%s", context->client_id, context->client_realm);

    kct_log(KCTLOG_DEBUG, "retrive_identity: name='%s' id='%s' realm='%s'\n",
	    client_name, context->client_id, context->client_realm);

  cleanup:
    if (client_name)
	free_memory(client_name);
    krb5_free_keytab_entry_contents(context->tkt_context, &ktent);

  getentry_err:
    if (ticket) {
	krb5_free_ticket(context->tkt_context, ticket);
    }

  recvauth_err:
    if (server)
	krb5_free_principal(context->tkt_context, server);

  server_err:
    if (keytab)
	krb5_kt_close(context->tkt_context, keytab);

  context_err:
    kct_log(KCTLOG_DEBUG, "handle_ap_req: end retval = %d\n", retval);

    return retval;
}

static int
receive_request(kct_context * context, int *key_length, char **key)
{

    int     sock = context->ws_sock;

    unsigned long xmitlen;
    unsigned char *p = NULL;
    int     len = sizeof(unsigned long);

    krb5_data recv_data, recv_unenc_data;
    krb5_error_code retval = 0;

    krb5_address remote_kaddr;
    struct sockaddr_in remote_addr;
    size_t  addrlen;

    char   *services = NULL;
    char   *word = NULL;
    char   *service_name = NULL;
    char   *lifetime = NULL;
    int     i = 0;

    kct_log(KCTLOG_DEBUG, "receive_request: start\n");

    recv_data.data = NULL;
    recv_data.length = 0;
    recv_unenc_data.data = NULL;
    recv_unenc_data.length = 0;

    if (recvblock(sock, (unsigned char *) &recv_data.length,
		  sizeof(xmitlen)) < 0) {
	kct_log(KCTLOG_ALWAYS,
		"receive_request: Socket closed while reading length: %s\n",
		strerror(errno));
	return KCT_ERR_COMMUNICATION_ERROR;
    }

    recv_data.length = ntohl(recv_data.length);

    kct_log(KCTLOG_DEBUG, "non-handshake length %d\n", recv_data.length);

    /* check that header is within reasonable range */
    if (recv_data.length <= 0 || recv_data.length > KCT_MAX_MEMORY_ALLOC) {
	kct_log(KCTLOG_ALWAYS,
		"receive_request: requested more "
		"memory (%d) than allowed (%d)\n",
		recv_data.length, KCT_MAX_MEMORY_ALLOC);
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    recv_data.data = (char *)
	get_memory(recv_data.length, sizeof(unsigned char));
    if (recvblock(sock, (unsigned char *) recv_data.data, recv_data.length) <
	0) {
	kct_log(KCTLOG_ALWAYS,
		"recieve_handshake: Socket closed while reading data\n");
	retval = KCT_ERR_COMMUNICATION_ERROR;
	goto recvdata_err;
    }

    if (retval = krb5_auth_con_setflags(context->tkt_context,
					context->tkt_auth_context,
					KRB5_AUTH_CONTEXT_DO_SEQUENCE)) {
	com_err("kct_server", retval, "unable to set authcontext flags");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    memset((char *) &remote_addr, 0, sizeof(remote_addr));
    addrlen = sizeof(remote_addr);
    if (getpeername(context->ws_sock, (struct sockaddr *) &remote_addr,
		    &addrlen) < 0) {
	kct_log(KCTLOG_ALWAYS, "receive_request: getpeername failed %s\n",
		strerror(errno));
	retval = KCT_ERR_INTERNAL_ERROR;
	goto recvdata_err;
    }

    remote_kaddr.addrtype = ADDRTYPE_INET;
    remote_kaddr.length = sizeof(remote_addr.sin_addr);
    remote_kaddr.contents = (krb5_octet *) & remote_addr.sin_addr;

    if ((retval = krb5_auth_con_setaddrs(context->tkt_context,
					 context->tkt_auth_context, NULL,
					 &remote_kaddr)) != 0) {
	com_err("kct_server", retval, "unable to set krb addr");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    remote_kaddr.addrtype = ADDRTYPE_IPPORT;
    remote_kaddr.length = sizeof(remote_addr.sin_port);
    remote_kaddr.contents = (krb5_octet *) & remote_addr.sin_port;

    if ((retval = krb5_auth_con_setports(context->tkt_context,
					 context->tkt_auth_context, NULL,
					 &remote_kaddr)) != 0) {
	com_err("kct_server", retval, "unable to set krb port");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto recvdata_err;
    }

    if (retval = krb5_rd_priv(context->tkt_context, context->tkt_auth_context,
			      &recv_data, &recv_unenc_data, NULL)) {
	com_err("kct_server", retval, "while verifying PRIV message");
	retval = KCT_ERR_RD_PRIV;
	goto cleanup;
    }

    p = (unsigned char *) recv_unenc_data.data;
    memcpy(&xmitlen, p, sizeof(xmitlen));
    p += sizeof(xmitlen);
    if (key_length != NULL && key != NULL) {
	if ((*key_length = ntohl(xmitlen)) > 0) {
	    *key =
		(unsigned char *) get_memory(*key_length,
					     sizeof(unsigned char));
	    memcpy(key, p, *key_length);
	    p += *key_length;
#ifdef DEBUG
	    print_buffer(*key, *key_length, "master_key: ");
#endif
	}
    }

    memcpy(&xmitlen, p, sizeof(xmitlen));
    p += sizeof(xmitlen);
    context->num_services = ntohl(xmitlen);
    if (context->num_services) {
	memcpy(&xmitlen, p, sizeof(xmitlen));
	p += sizeof(xmitlen);
	len = ntohl(xmitlen);
    } else {
	kct_log(KCTLOG_ALWAYS,
		"recieve_request: invalid number of services\n");
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto cleanup;
    }

    kct_log(KCTLOG_DEBUG, " %d services len %d\n", context->num_services, len);

    services = (char *) get_memory(len, sizeof(char));
    memcpy(services, p, len);

    kct_log(KCTLOG_DEBUG, "request for services %s\n", services);

    context->services = (kct_services *)
	get_memory(context->num_services, sizeof(kct_services));

    p = services;
    word = (char *) get_memory(len, sizeof(char));
    service_name = (char *) get_memory(len, sizeof(char));
    lifetime = (char *) get_memory(len, sizeof(char));
    for (i = 0; i < context->num_services; i++) {
	memset(word, '\0', sizeof(word));
	memset(service_name, '\0', sizeof(service_name));
	memset(lifetime, '\0', sizeof(lifetime));

	if (!sscanf(p, "%s", word))
	    break;
	sscanf(word, "%[^:]:%s", service_name, lifetime);

	context->services[i].service_name = (char *)
	    get_memory(strlen(service_name) + 1, sizeof(char));
	strcpy(context->services[i].service_name, service_name);

	context->services[i].lifetime = (char *)
	    get_memory(strlen(lifetime) + 1, sizeof(char));
	strcpy(context->services[i].lifetime, lifetime);

	kct_log(KCTLOG_DEBUG, "service name %s lifetime %s\n",
		service_name, lifetime);
	p += strlen(word);
    }
    if (word)
	free_memory(word);
    if (service_name)
	free_memory(service_name);
    if (lifetime)
	free_memory(lifetime);
    if (services)
	free_memory(services);

  cleanup:
    if (recv_unenc_data.data)
	krb5_free_data_contents(context->krb_context, &recv_unenc_data);

  recvdata_err:
    if (recv_data.data)
	krb5_free_data_contents(context->krb_context, &recv_data);

    kct_log(KCTLOG_DEBUG, "receive_request: end retval = %d\n", retval);

    return retval;
}

static int
verify_handshake(kct_context * context, SAVED_HANDSHAKE * s)
{

    unsigned char *verify_md = NULL;
    X509   *client_cert = NULL, *server_cert = NULL;
    EVP_PKEY *client_pkey = NULL, *server_pkey = NULL;
    int     client_type = 0, server_type = 0, retval = 0;
    char    name_buf[KCT_MAX_REALM_LEN];

    verify_md = (unsigned char *)
	get_memory(EVP_MAX_MD_SIZE * 5, sizeof(unsigned char));

    calculate_digest(s, verify_md);

    retval = retrieve_cert(s->version, s->s_c + 4, s->s_c_l - 4,
			   &server_pkey, &server_type, &server_cert);
    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }

    retval = retrieve_cert(s->version, s->c_c + 4, s->c_c_l - 4,
			   &client_pkey, &client_type, &client_cert);
    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }

    retval =
	verify_client_signature(s->c_v + 4, s->c_v_l - 4, verify_md,
				client_pkey);
    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }
#ifdef USE_CLIENT_TIMESTAMP
    /*
     * MSIE doesn't (always) put timestamp in CLIENT_HELLO
     */
    retval = verify_client_timestamp(s->c_h + 6);
#else
    /*
     * Use the server's timestamp instead...
     */
    retval = verify_client_timestamp(s->s_h + 6);
#endif
    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }

    retval = retrieve_explicit_identity(context, &context->client_id,
					&context->client_realm, client_cert);
    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }
#ifdef UOFM_ID_CHECK
    {
	int     idlen;
	idlen = strlen(context->client->id);
	/* check if id is with in the specified range */
	if (idlen < UOFM_MIN_ID_LEN || idlen > UOFM_MAX_ID_LEN) {
	    retval = KCT_ERR_INVALID_ID;
	    goto cleanup;
	}
    }
#endif

    if (context->ws_id)
	free_memory(context->ws_id);
    if (context->ws_realm)
	free_memory(context->ws_realm);

    /*
     * Attempt to retrieve explicit server id from emailAddress of cert
     * If that fails,
     *     If the first part of authenticator's principal matches
     *                    KCT_IMPLICIT_SERVICE_NAME
     *     then
     *         retrieve implicit server id from CN of certificate
     *         and ensure that it matches the instance of the authenticator
     */
    snprintf(name_buf, sizeof(name_buf), "%.*s",
	     context->ws_ticket->enc_part2->client->data[0].length,
	     context->ws_ticket->enc_part2->client->data[0].data);
    kct_log(KCTLOG_DEBUG,
	    "verify_handshake: principal of incoming service ticket: '%s'\n",
	    name_buf);

    snprintf(name_buf, sizeof(name_buf), "%.*s",
	     context->ws_ticket->enc_part2->client->data[1].length,
	     context->ws_ticket->enc_part2->client->data[1].data);
    kct_log(KCTLOG_DEBUG,
	    "verify_handshake: instance of incoming service ticket: '%s'\n",
	    name_buf);

    retval = retrieve_explicit_identity(context, &context->ws_id,
					&context->ws_realm, server_cert);
    if (KCT_ERR_NO_ERROR != retval) {	/* server id not in emailAddr */
	if ((strlen(kct_implicit_service_name) ==
	     context->ws_ticket->enc_part2->client->data[0].length)
	    && (strncmp(kct_implicit_service_name,
			context->ws_ticket->enc_part2->client->data[0].data,
			strlen(kct_implicit_service_name)) == 0)) {
	    /* authent principal is KCT_IMPLICIT_SERVICE_NAME */
	    char    s[KCT_MAX_BUF_SIZE];

	    strncpy(context->ws_realm,
		    context->ws_ticket->enc_part2->client->realm.data,
		    context->ws_ticket->enc_part2->client->realm.length);
	    context->ws_realm[context->ws_ticket->enc_part2->client->realm.
			      length] = '\0';
	    retval =
		retrieve_implicit_identity(context, &context->ws_id,
					   server_cert);
	    /* got implicit identity (cert's CN) --  ensure it matches krb authent */
	    kct_log(KCTLOG_DEBUG,
		    "verify_handshake: comparing length %d with %d\n",
		    (strlen(context->ws_id) -
		     strlen(kct_implicit_service_name) - 1),
		    context->ws_ticket->enc_part2->client->data[1].length);
	    snprintf(s, sizeof(s), "%.*s",
		     context->ws_ticket->enc_part2->client->data[1].length,
		     context->ws_ticket->enc_part2->client->data[1].data);
	    kct_log(KCTLOG_DEBUG,
		    "verify_handshake: comparing data '%s' with " "'%s'\n",
		    context->ws_id + strlen(kct_implicit_service_name) + 1, s);

	    if (KCT_ERR_NO_ERROR == retval) {
		if ((strlen(context->ws_id) -
		     strlen(kct_implicit_service_name) - 1 !=
		     context->ws_ticket->enc_part2->client->data[1].length)
		    ||
		    (strncmp
		     (context->ws_id + strlen(kct_implicit_service_name) + 1,
		      context->ws_ticket->enc_part2->client->data[1].data,
		      context->ws_ticket->enc_part2->client->data[1].length)))
		    retval = KCT_ERR_INVALID_ID;
	    }
	}
    }

    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }

    retval = verify_server_identity(context);

    if (KCT_ERR_NO_ERROR != retval) {
	goto cleanup;
    }

  cleanup:

    if (verify_md)
	free_memory(verify_md);
    if (server_cert)
	X509_free(server_cert);
    if (client_cert)
	X509_free(client_cert);
    if (client_pkey)
	EVP_PKEY_free(client_pkey);
    if (server_pkey)
	EVP_PKEY_free(server_pkey);

    return retval;

}

/*
 * this function checks that the identity inside the server
 * certificate used during the handshake, is the same as the
 * identity inside of the ticket used in AP_REQ
 */
static int
verify_server_identity(kct_context * context)
{

    char    s[KCT_MAX_BUF_SIZE];
    snprintf(s, sizeof(s),
	     context->ws_ticket->enc_part2->client->length >
	     1 ? "%.*s/%.*s" : "%.*s",
	     context->ws_ticket->enc_part2->client->data[0].length,
	     context->ws_ticket->enc_part2->client->data[0].data,
	     context->ws_ticket->enc_part2->client->data[1].length,
	     context->ws_ticket->enc_part2->client->data[1].data);

    if (strncmp(context->ws_id, s, strlen(s)) == 0)
	return KCT_ERR_NO_ERROR;
    else {
	kct_log(KCTLOG_ALWAYS,
		"verify_server_identity: len %d, Certificate identity '%s' != "
		"Kerberos identity '%s'\n", strlen(s), context->ws_id, s);
	return KCT_ERR_INVALID_HANDSHAKE;
    }
}

static int
generate_service_ticket(kct_context * kct_context,
			char *service_id,
			char *lifetime, unsigned char **reply_buf, int *len)
{

    krb5_context context = kct_context->krb_context;
    krb5_principal server;
    krb5_principal client;
    krb5_principal ws;
    kadm5_principal_ent_rec kadm5_server;
    kadm5_principal_ent_rec kadm5_client;
    kadm5_principal_ent_rec kadm5_ws;

    krb5_keyblock encrypting_key;
    krb5_keyblock session_key;
    krb5_keysalt encrypting_keysalt;
    int     encrypting_kvno;

    krb5_timestamp kct_time;

    krb5_data *response = NULL;
    krb5_kdc_rep reply;
    krb5_enc_kdc_rep_part reply_encpart;
    krb5_ticket ticket_reply;
    krb5_enc_tkt_part enc_tkt_reply;
    krb5_error_code retval = 0;

    krb5_enctype useenctype = ENCTYPE_DES_CBC_CRC;
    krb5_last_req_entry *nolrarray[2], nolrentry;
    int     errcode;
    const char *status = 0;
    krb5_data empty_str = { 0, 0, "" };

    krb5_deltat krb_lifetime;
    kadm5_config_params realm_params;

    kct_log(KCTLOG_DEBUG, "generate_service_ticket: start\n");

    session_key.contents = 0;

    if ((retval = krb5_parse_name(context, kct_context->client_id,
				  &client)) != 0) {
	com_err("kct_server", retval, "unable to create %s service principal",
		kct_context->client_id);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto client_err;
    }

    if ((retval = krb5_parse_name(context, service_id, &server)) != 0) {
	com_err("kct_server", retval, "unable to create %s service principal",
		service_id);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto service_err;
    }

    if ((retval = krb5_parse_name(context, kct_context->ws_id, &ws)) != 0) {
	com_err("kct_server", retval, "unable to create %s service principal",
		kct_context->ws_id);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto ws_err;
    }

    if (!single_process) {
	if ((retval =
	     kadm5_init(kct_princ_name, NULL, KADM5_ADMIN_SERVICE, NULL,
			KADM5_STRUCT_VERSION, KADM5_API_VERSION_2, &handle))) {
	    com_err("kct_server", retval, "initializing kadm5 library");
	    retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	    goto kadminit_err;
	}
    }

    if ((retval = kadm5_get_principal(handle, server, &kadm5_server,
				      KADM5_KEY_DATA | KADM5_ATTRIBUTES |
				      KADM5_PRINCIPAL_NORMAL_MASK))) {
	com_err("kct_server", retval, "while getting a principal %s",
		service_id);
	retval = KCT_ERR_UNKNOWN_PRINCIPAL;
	goto kadmprinc_err;
    }

    if (!ignoreNonExistingPrincipals) {
	if ((retval = kadm5_get_principal(handle, client, &kadm5_client,
					  KADM5_KEY_DATA | KADM5_ATTRIBUTES |
					  KADM5_PRINCIPAL_NORMAL_MASK))) {
	    com_err("kct_server", retval, "while getting a principal %s",
		    kct_context->client_id);
	    retval = KCT_ERR_UNKNOWN_PRINCIPAL;
	    goto kadm5server_err;
	}

	if (kadm5_client.attributes & KRB5_KDB_DISALLOW_ALL_TIX) {
	    kct_log(KCTLOG_ERR, "Disallow all tickets flag is set for %s\n",
		    kct_context->client_id);
	    retval = KCT_ERR_BAD_PRINCIPAL;
	    goto kadm5server_err;
	}

	/* 06/29/2005: Note: billdo asked that we also check for when the
	 * user's password has been changed and if so, we don't reissue the
	 * the new ticket. However, since KDC is stateless we have no way of
	 * knowing if the passwords has been changed since the last time we 
	 * issued a service ticket. We know between TGTs because we can check
	 * the difference in the key version....
	 */
    }

    if ((retval = kadm5_get_principal(handle, ws, &kadm5_ws,
				      KADM5_KEY_DATA | KADM5_ATTRIBUTES |
				      KADM5_PRINCIPAL_NORMAL_MASK))) {
	com_err("kct_server", retval, "while getting a principal %s",
		kct_context->client_id);
	retval = KCT_ERR_UNKNOWN_PRINCIPAL;
	goto kadm5client_err;
    }

    /* need to perform certain checks on principal(s) 
     * 1. no ticket flag is not set
     * 2. if client has changed their password (?)
     */
    if (kadm5_server.attributes & KRB5_KDB_DISALLOW_ALL_TIX) {
	kct_log(KCTLOG_ERR, "Disallow all tickets flag is set for %s\n",
		service_id);
	retval = KCT_ERR_BAD_PRINCIPAL;
	goto kadmdecrypt_err;
    }

    if (kadm5_ws.attributes & KRB5_KDB_DISALLOW_ALL_TIX) {
	kct_log(KCTLOG_ERR, "Disallow all tickets flag is set for %s\n",
		kct_context->ws_id);
	retval = KCT_ERR_BAD_PRINCIPAL;
	goto kadmdecrypt_err;
    }

    if ((retval = kadm5_get_config_params(context, NULL, NULL, NULL,
					  &realm_params))) {
	com_err("kct_server", retval, "while getting realm parameters\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto kadmdecrypt_err;
    }

    if ((retval = kadm5_decrypt_key(handle, &kadm5_server, -1,	/*ENCTYPE_DES_CBC_CRC, */
				    -1,
				    0,
				    &encrypting_key, &encrypting_keysalt,
				    &encrypting_kvno))) {
	if ((retval =
	     kadm5_decrypt_key(handle, &kadm5_server, -1, -1, 0,
			       &encrypting_key, NULL, NULL))) {
	    com_err("kct_server", retval, "Couldn't find any matching key");
	    retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	    goto kadmdecrypt_err;
	}
    }

    if ((retval = krb5_timeofday(context, &kct_time))) {
	com_err("kct_server", retval, "time of day failed\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto kadmdecrypt_err;
    }

    if (retval = krb5_c_make_random_key(context, useenctype, &session_key)) {
	com_err("kct_server", retval, "creating a random key failed\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto kadmdecrypt_err;
    }

    ticket_reply.server = server;	/* XXX careful for realm... */

#ifdef DEBUG_AFS_TKT
    {
	char    n[KCT_MAX_BUF_SIZE];
	char    r[KCT_MAX_BUF_SIZE];

	snprintf(n, sizeof(n), "%.*s", client->data->length,
		 client->data->data);
	snprintf(r, sizeof(r), "%.*s", client->realm->length,
		 client->realm->data);
	kct_log(KCTLOG_DEBUG, "client name '%s' realm '%s'\n", n, r);

	snprintf(n, sizeof(n), "%.*s", ticket_reply.server->data->length,
		 ticket_reply.server->data->data);
	snprintf(r, sizeof(r), "%.*s", ticket_reply.server->realm->length,
		 ticket_reply.server->realm->data);
	kct_log(KCTLOG_DEBUG, "server name '%s' realm '%s'\n", n, r);

	print_buffer(session_key.contents, session_key.length, "afs key: ");
    }
#endif

    enc_tkt_reply.flags = 0;
    enc_tkt_reply.caddrs = kct_context->ws_ticket->enc_part2->caddrs;
    reply_encpart.caddrs = 0;
    enc_tkt_reply.times.authtime = kct_time;
    enc_tkt_reply.times.starttime = kct_time;

    if ((retval = krb5_string_to_deltat(lifetime, &krb_lifetime))) {
	kct_log(KCTLOG_ALWAYS, "Bad lifetime %s\n", lifetime);
	retval = KCT_ERR_MALFORMED_LIFETIME;
	goto sessionkey_err;
    }

    if (!(realm_params.mask & KADM5_CONFIG_MAX_LIFE)) {
	kct_log(KCTLOG_ERR, "Cannot determine maximum ticket lifetime\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto kadm5params_err;
    }

    if (!strncmp(service_id, tts_princ_name, strlen(service_id))) {
	kct_log(KCTLOG_DEBUG,
		"Requested lifetime = %d Max server lifetime = %d\n",
		krb_lifetime, kadm5_server.max_life);
	enc_tkt_reply.times.endtime =
	    kct_time + min(krb_lifetime, kadm5_server.max_life);
    } else {
	kct_log(KCTLOG_DEBUG, "%s %s Requested lifetime = %d "
		"Max server lifetime = %d Max realm lifetime = %d\n",
		service_id, tts_princ_name, krb_lifetime,
		kadm5_server.max_life, realm_params.max_life);

	enc_tkt_reply.times.endtime = kct_time +
	    min(krb_lifetime,
		min(kadm5_server.max_life, realm_params.max_life));
    }

    enc_tkt_reply.times.renew_till = 0;
    enc_tkt_reply.authorization_data = 0;

    enc_tkt_reply.session = &session_key;
    enc_tkt_reply.client = client;
    enc_tkt_reply.transited.tr_type = KRB5_DOMAIN_X500_COMPRESS;
    enc_tkt_reply.transited.tr_contents = empty_str;	/* equivalent of "" */

    ticket_reply.enc_part2 = &enc_tkt_reply;

    if ((retval = krb5_encrypt_tkt_part(context, &encrypting_key,
					&ticket_reply)) != 0) {
	com_err("kct_server", retval, "encrypting of tkt part failed %d\n",
		retval);
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto kadm5params_err;
    }

    ticket_reply.enc_part.kvno = kadm5_server.key_data->key_data_kvno;

    /* Start assembling the response */
    reply.msg_type = KRB5_TGS_REP;
    reply.padata = 0;		/* always */
    reply.client = client;
    reply.enc_part.kvno = 0;	/* We are using the session key */
    reply.ticket = &ticket_reply;

    reply_encpart.msg_type = KRB5_TGS_REP;
    reply_encpart.session = &session_key;
    reply_encpart.nonce = 0;
    reply_encpart.times = enc_tkt_reply.times;
    reply_encpart.times.authtime = enc_tkt_reply.times.authtime;

    nolrentry.lr_type = KRB5_LRQ_NONE;
    nolrentry.value = 0;
    nolrarray[0] = &nolrentry;
    nolrarray[1] = 0;
    reply_encpart.last_req = nolrarray;	/* not available for TGS reqs */
    reply_encpart.key_exp = 0;	/* ditto */
    reply_encpart.flags = enc_tkt_reply.flags;
    reply_encpart.server = ticket_reply.server;
    reply.enc_part.enctype = kct_context->ws_ticket->enc_part.enctype;

#ifdef DEBUG_AFS_TKT
    print_buffer(kct_context->ws_ticket->enc_part2->session->contents,
		 kct_context->ws_ticket->enc_part2->session->length,
		 "kct session key: ");
#endif

    if ((retval =
	 krb5_encode_kdc_rep(context, KRB5_TGS_REP, &reply_encpart,
			     0, kct_context->ws_ticket->enc_part2->session,
			     &reply, &response)) != 0) {
	com_err("kct_server", retval, "encoding kdc reply failed\n");
	retval = KCT_ERR_INTERNAL_KERBEROS_ERROR;
	goto encrypting_err;
    }

    (*len) = response->length;
    (*reply_buf) = (unsigned char *)
	get_memory(response->length, sizeof(unsigned char));
    memcpy((*reply_buf), response->data, response->length);

    kct_log(KCTLOG_ALWAYS, "KCT_REQ: ISSUE: authtime %d, %s for %s "
	    "lifetime %d (to %s@%s)\n", kct_time,
	    kct_context->client_id, service_id,
	    enc_tkt_reply.times.endtime - kct_time,
	    kct_context->ws_id, kct_context->ws_realm);

  cleanup:
    if (response)
	krb5_free_data(context, response);
    if (ticket_reply.enc_part.ciphertext.data) {
	memset(ticket_reply.enc_part.ciphertext.data, 0,
	       ticket_reply.enc_part.ciphertext.length);
	free_memory(ticket_reply.enc_part.ciphertext.data);
    }
    if (reply.enc_part.ciphertext.data) {
	memset(reply.enc_part.ciphertext.data, 0,
	       reply.enc_part.ciphertext.length);
	free_memory(reply.enc_part.ciphertext.data);
    }

  encrypting_err:
  kadm5params_err:
    if (encrypting_key.contents)
	krb5_free_keyblock_contents(context, &encrypting_key);

    kadm5_free_config_params(handle, &realm_params);

  sessionkey_err:
    if (session_key.contents)
	krb5_free_keyblock_contents(context, &session_key);

  kadmdecrypt_err:
    kadm5_free_principal_ent(handle, &kadm5_ws);

  kadm5client_err:
    if (!ignoreNonExistingPrincipals)
	kadm5_free_principal_ent(handle, &kadm5_client);

  kadm5server_err:
    kadm5_free_principal_ent(handle, &kadm5_server);

  kadmprinc_err:
    if (!single_process)
	kadm5_destroy(handle);

  kadminit_err:
    if (ws)
	krb5_free_principal(context, ws);

  ws_err:
    if (server)
	krb5_free_principal(context, server);

  service_err:
    if (client)
	krb5_free_principal(context, client);

  client_err:
    kct_log(KCTLOG_DEBUG, "generate_service_ticket: end retval = %d\n",
	    retval);

    return retval;

}

/*
 * Send the reply to the client
 *
 * First send a header telling the client how long the reply
 * will be.  Then attach the error number in the reply (putting
 * it before the the service ticket, if there is one)
 */
static int
send_reply(int sock, long retval, unsigned char *reply, int len)
{

    int     rc = 0;
    long    xmitlen = 0;
    unsigned char *send_buf = NULL;

    send_buf = (unsigned char *)
	get_memory(len + sizeof(long), sizeof(unsigned char));
    xmitlen = htonl(retval);
    memcpy(send_buf, (char *) &xmitlen, sizeof(long));
    if (len)
	memcpy(send_buf + sizeof(long), reply, len);

    xmitlen = htonl(len + sizeof(long));

#ifdef DEBUG
    print_buffer(send_buf, len + sizeof(long), "send buffer:\n");
#endif

    kct_log(KCTLOG_DEBUG, "sending: %d\n", len + sizeof(long));

    if (sendblock(sock, (unsigned char *) &xmitlen, sizeof(xmitlen)) < 0) {
	kct_log(KCTLOG_ALWAYS,
		"send_reply: socket closed sending length: %s\n",
		strerror(errno));
	close(sock);
	sock = 0;
	rc = KCT_ERR_COMMUNICATION_ERROR;
	goto send_err;
    }

    if (sendblock(sock, (unsigned char *) send_buf, len + sizeof(long)) < 0) {
	kct_log(KCTLOG_ALWAYS, "send_reply: socket closed sending data: %s\n",
		strerror(errno));
	close(sock);
	sock = 0;
	rc = KCT_ERR_COMMUNICATION_ERROR;
    }

  send_err:
    if (send_buf)
	free_memory(send_buf);

    return rc;
}

/*
 * To verify the client's signature inside of the certificate,
 * we need to calculate the digest of all handshake messages and
 * verify that the digest is the same as the signed digest with
 * in client verify message
 */
static void
calculate_digest(SAVED_HANDSHAKE * s, unsigned char *p)
{

    EVP_MD_CTX d1;
    EVP_MD_CTX d2;
    EVP_MD_CTX ctx1;
    const EVP_MD *md5;
    const EVP_MD *sha1;
    unsigned int ret;

#ifdef DEBUG
    kct_log(KCTLOG_DEBUG, "calculate_digest: calculating the digest\n");
#endif

    EVP_add_digest(EVP_md5());
    EVP_add_digest(EVP_sha1());
    EVP_add_digest_alias(SN_md5, "md5");
    EVP_add_digest_alias(SN_sha1, "sha1");
    md5 = EVP_get_digestbyname("md5");
    sha1 = EVP_get_digestbyname("sha1");

    EVP_DigestInit(&d1, md5);
    EVP_DigestInit(&d2, sha1);

    if (s->c_h != NULL) {
	EVP_DigestUpdate(&d1, s->c_h, s->c_h_l);
	EVP_DigestUpdate(&d2, s->c_h, s->c_h_l);
    }
    if (s->s_h != NULL) {
	EVP_DigestUpdate(&d1, s->s_h, s->s_h_l);
	EVP_DigestUpdate(&d2, s->s_h, s->s_h_l);
    }
    if (s->s_c != NULL) {
	EVP_DigestUpdate(&d1, s->s_c, s->s_c_l);
	EVP_DigestUpdate(&d2, s->s_c, s->s_c_l);
    }
    if (s->s_k_e != NULL && s->s_k_e_l > 0) {
	EVP_DigestUpdate(&d1, s->s_k_e, s->s_k_e_l);
	EVP_DigestUpdate(&d2, s->s_k_e, s->s_k_e_l);
    }
    if (s->s_c_r != NULL) {
	EVP_DigestUpdate(&d1, s->s_c_r, s->s_c_r_l);
	EVP_DigestUpdate(&d2, s->s_c_r, s->s_c_r_l);
    }
    if (s->c_c != NULL) {
	EVP_DigestUpdate(&d1, s->c_c, s->c_c_l);
	EVP_DigestUpdate(&d2, s->c_c, s->c_c_l);
    }
    if (s->c_k_e != NULL) {
	EVP_DigestUpdate(&d1, s->c_k_e, s->c_k_e_l);
	EVP_DigestUpdate(&d2, s->c_k_e, s->c_k_e_l);
    }

    if (s->version == TLS1_VERSION) {
	EVP_MD_CTX_copy(&ctx1, &d1);
	EVP_DigestFinal(&ctx1, &p[0], &ret);

	EVP_MD_CTX_copy(&ctx1, &d2);
	EVP_DigestFinal(&ctx1, &p[MD5_DIGEST_LENGTH], &ret);

	EVP_MD_CTX_cleanup(&ctx1);
    } else {
	cert_verify_mac(s->master_key, s->master_key_length, &d1, &p[0]);
	cert_verify_mac(s->master_key, s->master_key_length, &d2,
			&p[MD5_DIGEST_LENGTH]);
    }

#ifdef DEBUG_DIGEST
    print_buffer(p, EVP_MAX_MD_SIZE * 2, "my digest: ");
    print_buffer(s->c_v, s->c_v_l, "signature: ");
#endif

    EVP_MD_CTX_cleanup(&d1);
    EVP_MD_CTX_cleanup(&d2);

}

/*
 * this function is used to calculate the final mac
 * for SSLv3 handshake
 */
static void
cert_verify_mac(unsigned char *master_key, int master_key_length,
		EVP_MD_CTX * in_ctx, unsigned char *p)
{

    static unsigned char ssl3_pad_1[48] = {
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36
    };

    static unsigned char ssl3_pad_2[48] = {
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c,
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c,
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c,
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c,
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c,
	0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c, 0x5c
    };
    unsigned int ret = 0;
    int     npad = 0, n = 0;
    unsigned int i = 0;
    unsigned char md_buf[EVP_MAX_MD_SIZE];
    EVP_MD_CTX ctx;

    EVP_MD_CTX_init(&ctx);
    EVP_MD_CTX_copy_ex(&ctx, in_ctx);

    n = EVP_MD_CTX_size(&ctx);
    npad = (48 / n) * n;

    EVP_DigestUpdate(&ctx, master_key, master_key_length);
    EVP_DigestUpdate(&ctx, ssl3_pad_1, npad);
    EVP_DigestFinal_ex(&ctx, md_buf, &i);

    EVP_DigestInit_ex(&ctx, EVP_MD_CTX_md(&ctx), NULL);

    EVP_DigestUpdate(&ctx, master_key, master_key_length);
    EVP_DigestUpdate(&ctx, ssl3_pad_2, npad);
    EVP_DigestUpdate(&ctx, md_buf, i);
    EVP_DigestFinal_ex(&ctx, p, &ret);

    EVP_MD_CTX_cleanup(&ctx);
}

/*
 * This function retrieves a certificate out of the buffer
 * (handshake bytes).  The code for extracting the certificate
 * is taken from openssl library.  After the certificate is
 * extracted, verify the signature of the CA.
 */
#define n2l3(c,l)       ((l =(((unsigned long)(c[0]))<<16)| \
			     (((unsigned long)(c[1]))<< 8)| \
			     (((unsigned long)(c[2]))    )),c+=3)

static int
retrieve_cert(int v, unsigned char *p, unsigned int plen,
	      EVP_PKEY ** pkey, int *type, X509 ** cert)
{

    STACK_OF(X509) * sk = NULL;
    X509   *x = NULL;
    unsigned long l, nc, slen;
    unsigned char *q;

    if ((sk = sk_X509_new_null()) == NULL) {
	kct_log(KCTLOG_ALWAYS,
		"retrieve_cert: error while getting client's certificate\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    n2l3(p, slen);

    if (slen + 3 != plen) {
	kct_log(KCTLOG_ALWAYS,
		"retrieve_cert: decoding error while getting certificate\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    for (nc = 0; nc < slen;) {
	n2l3(p, l);
	if ((l + nc + 3) > slen) {
	    kct_log(KCTLOG_ALWAYS,
		    "retrieve_cert: bad length l==%d, nc==%d, slen==%d\n",
		    l, nc, slen);
	    return KCT_ERR_INVALID_HANDSHAKE;
	}

	q = p;
	x = d2i_X509(NULL, &p, l);
	if (x == NULL) {
	    kct_log(KCTLOG_ALWAYS,
		    "retrieve_cert: failed to retrieve cert from stream\n");
	    return KCT_ERR_INVALID_HANDSHAKE;
	}
	if (p != (q + l)) {
	    kct_log(KCTLOG_ALWAYS,
		    "retrieve_cert: wierd length error retrieving cert from "
		    "stream p==0x08x, l==%d, q==0x%08x\n", p, l, q);
	    return KCT_ERR_INVALID_HANDSHAKE;
	}
	if (!sk_X509_push(sk, x)) {
	    kct_log(KCTLOG_ALWAYS,
		    "retrieve_cert: error pushing cert onto stack\n");
	    return KCT_ERR_INVALID_HANDSHAKE;
	}

	x = NULL;
	nc += l + 3;
    }

    if (sk_X509_num(sk) <= 0 && v == SSL3_VERSION) {
	/* TLS does not mind 0 certs returned */
	kct_log(KCTLOG_ALWAYS, "retrieve_cert: certificate list is 0\n");
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    (*cert) = sk_X509_shift(sk);
    (*pkey) = X509_get_pubkey((*cert));
    (*type) = X509_certificate_type((*cert), (*pkey));

    /* Now need to verify the certificate chain */

    {
	X509_STORE *certstore = NULL;
	X509_STORE_CTX certstorectx;
	X509_LOOKUP *lookup = NULL;

	char   *CAFile = ca_file;
	char   *CAPath = ca_path;
	int     i = 0;
	int     retval = KCT_ERR_NO_ERROR;

	ERR_load_crypto_strings();
	signal(SIGPIPE, SIG_IGN);
	CRYPTO_malloc_init();
	OpenSSL_add_all_algorithms();

	if ((certstore = X509_STORE_new()) == NULL) {
	    kct_log(KCTLOG_ALWAYS,
		    "retrieve_cert: couldn't create X509_STORE\n");
	    retval = KCT_ERR_INTERNAL_ERROR;
	    goto cleanup;
	}

	X509_STORE_set_verify_cb_func(certstore, cb_local);

	lookup = X509_STORE_add_lookup(certstore, X509_LOOKUP_file());
	if (lookup == NULL) {
	    retval = KCT_ERR_INTERNAL_ERROR;
	    goto cleanup;
	}
	if (CAFile) {
	    i = X509_LOOKUP_load_file(lookup, CAFile, X509_FILETYPE_PEM);
	    if (!i) {
		kct_log(KCTLOG_ALWAYS,
			"retrieve_cert: Error loading file %s\n", CAFile);
		retval = KCT_ERR_INTERNAL_ERROR;
		goto cleanup;
	    }
	} else
	    X509_LOOKUP_load_file(lookup, NULL, X509_FILETYPE_DEFAULT);

	lookup = X509_STORE_add_lookup(certstore, X509_LOOKUP_hash_dir());
	if (lookup == NULL) {
	    retval = KCT_ERR_INTERNAL_ERROR;
	    goto cleanup;
	}
	if (CAPath) {
	    i = X509_LOOKUP_add_dir(lookup, CAPath, X509_FILETYPE_PEM);
	    if (!i) {
		kct_log(KCTLOG_ALWAYS,
			"retrieve_cert: Error loading directory %s\n", CAPath);
		retval = KCT_ERR_INTERNAL_ERROR;
		goto cleanup;
	    }
	} else
	    X509_LOOKUP_add_dir(lookup, NULL, X509_FILETYPE_DEFAULT);

	X509_STORE_CTX_init(&certstorectx, certstore, (*cert), NULL);
	i = X509_verify_cert(&certstorectx);
	X509_STORE_CTX_cleanup(&certstorectx);
	if (!i) {
	    kct_log(KCTLOG_ALWAYS, "retrieve_cert: cert verify failed\n");
	    retval = KCT_ERR_INVALID_HANDSHAKE;
	    goto cleanup;
	}
	X509_STORE_free(certstore);
    }

  cleanup:
    EVP_cleanup();
    CRYPTO_cleanup_all_ex_data();
    ERR_remove_state(0);
    ERR_free_strings();

    if (x)
	X509_free(x);
    if (sk)
	sk_X509_pop_free(sk, X509_free);

    return KCT_ERR_NO_ERROR;
}

/*
 * This function is used during debugging.  It is called if
 * verification of the certificate fails.
 */
static int
cb_local(int ok, X509_STORE_CTX * ctx)
{
    char    buf[256];
    static int v_verbose = 0;

    if (!ok) {
	kct_log(KCTLOG_ALWAYS,
		"cb_local: Certificate verification failed...\n");
	X509_NAME_oneline(X509_get_subject_name(ctx->current_cert), buf, 256);
	kct_log(KCTLOG_ALWAYS,
		"cb_local: %s: error %d at %d depth lookup: %s\n", buf,
		ctx->error, ctx->error_depth,
		X509_verify_cert_error_string(ctx->error));

	/*
	 * If a cert within the verification path is expired, then
	 * fail the request.  Log the problem, and return the
	 * current value of "ok".
	 */
	if (ctx->error == X509_V_ERR_CERT_HAS_EXPIRED) {
	    kct_log(KCTLOG_ALWAYS, "cb_local: failing: expired cert\n");
	    return ok;
	}

	/*
	 * Since we are just checking the certificates, it is
	 * ok if they are self signed. But we should still warn
	 * the user.
	 */
	if (ctx->error == X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) {
	    kct_log(KCTLOG_ALWAYS,
		    "cb_local: ignoring: depth 0 self signed cert\n");
	    ok = 1;
	}

	/*
	 * Continue after extension errors too
	 */
	if (ctx->error == X509_V_ERR_INVALID_CA) {
	    kct_log(KCTLOG_ALWAYS, "cb_local: ignoring: invalid ca\n");
	    ok = 1;
	}
	if (ctx->error == X509_V_ERR_PATH_LENGTH_EXCEEDED) {
	    kct_log(KCTLOG_ALWAYS,
		    "cb_local: ignoring: path length exceeded\n");
	    ok = 1;
	}
	if (ctx->error == X509_V_ERR_INVALID_PURPOSE) {
	    kct_log(KCTLOG_ALWAYS, "cb_local: ignoring: invalid purpose\n");
	    ok = 1;
	}
    }
    if (!v_verbose)
	ERR_clear_error();
    return (ok);
}

/*
 * this code verifies client's signature within the handshake
 * code is taken from openssl library (unmodified)
 */
#define n2s(c,s)        ((s=(((unsigned int)(c[0]))<< 8)| \
			    (((unsigned int)(c[1])))),c+=2)

static int
verify_client_signature(unsigned char *p, unsigned int plen,
			unsigned char *verify_md, EVP_PKEY * pkey)
{
    int     i, j, ilen, pkey_size;

    n2s(p, ilen);
    plen -= 2;

#ifdef DEBUG
    kct_log(KCTLOG_DEBUG,
	    "verify_client_signature: verifying the "
	    "validity of client's signature\n");
#endif

    if (ilen > plen) {
	kct_log(KCTLOG_ALWAYS,
		"verify_client_signature: ilen==%d > plen==%d\n", ilen, plen);
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    pkey_size = EVP_PKEY_size(pkey);
    if ((ilen > pkey_size) || (plen > pkey_size) || (plen <= 0)) {
	kct_log(KCTLOG_ALWAYS,
		"verify_client_signature: length error ilen==%d, plen==%d, "
		"pkey_size==%d\n", ilen, plen, pkey_size);
	return KCT_ERR_INVALID_HANDSHAKE;
    }
#ifndef NO_RSA
    if (pkey->type == EVP_PKEY_RSA) {
	i = RSA_verify(NID_md5_sha1, verify_md,
		       MD5_DIGEST_LENGTH + SHA_DIGEST_LENGTH, p, ilen,
		       pkey->pkey.rsa);
	if (i < 0) {
	    kct_log(KCTLOG_ALWAYS,
		    "verify_client_signature: SSL_R_BAD_RSA_DECRYPT\n");
	    return KCT_ERR_INVALID_HANDSHAKE;
	}
	if (i == 0) {
	    kct_log(KCTLOG_ALWAYS,
		    "verify_client_signature: SSL_R_BAD_RSA_SIGNATURE\n");
	    return KCT_ERR_INVALID_HANDSHAKE;
	}
    } else
#endif
#ifndef NO_DSA
    if (pkey->type == EVP_PKEY_DSA) {
	j = DSA_verify(pkey->save_type, &(verify_md[MD5_DIGEST_LENGTH]),
		       SHA_DIGEST_LENGTH, p, i, pkey->pkey.dsa);
	if (j <= 0) {
	    kct_log(KCTLOG_ALWAYS,
		    "verify_client_signature: DSA_BAD_SIGNATURE\n");
	    return KCT_ERR_INVALID_HANDSHAKE;
	}
    } else
#endif
    {
	kct_log(KCTLOG_ALWAYS,
		"verify_client_signature: "
		"invalid encryption type (%d)\n", pkey->type);
	return KCT_ERR_INVALID_HANDSHAKE;
    }

    return KCT_ERR_NO_ERROR;
}

/*
 * Look up the timestamp in client hello message. 
 * Look up the current time (in secs)
 * Verify that the difference is within valid time period
 */
#define n2l(c,l)        (l =((unsigned long)(*((c)++)))<<24, \
			 l|=((unsigned long)(*((c)++)))<<16, \
			 l|=((unsigned long)(*((c)++)))<< 8, \
			 l|=((unsigned long)(*((c)++))))

static int
verify_client_timestamp(unsigned char *p)
{
    unsigned char cTime[SSL3_RANDOM_SIZE];
    unsigned char *pTime;
    unsigned long rTime;
    unsigned long currenttime;
    char    c_ctime[] = "Fri Sep 13 00:00:00 1986\n";
    char    k_ctime[] = "Fri Sep 13 00:00:00 1986\n";

    memcpy(cTime, p, SSL3_RANDOM_SIZE);
    pTime = cTime;
    n2l(pTime, rTime);

    currenttime = time(NULL);

    /* BILLDO -- copy static results of get_time to distinct storage */
    strcpy(c_ctime, get_time(rTime));
    strcpy(k_ctime, get_time(currenttime));

    kct_log(KCTLOG_DEBUG,
	    "verify_client_timestamp: client=%d (%s) currenttime=%d (%s)\n",
	    rTime, c_ctime, currenttime, k_ctime);

    if (abs(currenttime - rTime) > handshake_valid_time) {
	kct_log(KCTLOG_ALWAYS,
		"verify_client_timestamp: expired: client=%s current=%s\n",
		c_ctime, k_ctime);
	return KCT_ERR_OLD_HANDSHAKE;
    }

    return KCT_ERR_NO_ERROR;
}

static void
process_exit(int sig, const struct sigaction *act, struct sigaction *oact)
{

    kct_log(KCTLOG_ALWAYS,
	    "process_exit: process received a wake up call...\n");
    exit(1);
}

static unsigned char *
read_message(unsigned char **p, unsigned int *len)
{
    unsigned char *buf;
    int     hlen = sizeof(unsigned long);

    memcpy((char *) len, (*p), hlen);
    (*len) = ntohl((*len));

#ifdef DEBUG
    kct_log(KCTLOG_DEBUG, "read_message: %d\n", *len);
#endif

    /* check that length is within reasonable range */
    if ((*len) <= 0 || (*len) > KCT_MAX_MEMORY_ALLOC) {
	(*p) += hlen;
	return NULL;
    }

    (*p) += hlen;
    buf = (unsigned char *) get_memory((*len), sizeof(unsigned char));
    memcpy(buf, (*p), (*len));
    (*p) += (*len);

    return buf;
}

static void
initialize_handshake(SAVED_HANDSHAKE * sh)
{
    sh->master_key = NULL;
    sh->master_key_length = 0;
    sh->c_h = NULL;
    sh->c_h_l = 0;
    sh->s_h = NULL;
    sh->s_h_l = 0;
    sh->s_c = NULL;
    sh->s_c_l = 0;
    sh->s_c_r = NULL;
    sh->s_c_r_l = 0;
    sh->s_k_e = NULL;
    sh->s_k_e_l = 0;
    sh->c_c = NULL;
    sh->c_c_l = 0;
    sh->c_k_e = NULL;
    sh->c_k_e_l = 0;
    sh->c_v = NULL;
    sh->c_v_l = 0;
}

static void
initialize_kct_context(kct_context * k)
{
    k->client_id = NULL;
    k->client_realm = NULL;
    k->ws_id = NULL;
    k->ws_realm = NULL;
    k->services = NULL;
    k->num_services = 0;
    memset(k->ws_host, '\0', sizeof(k->ws_host));
    k->ws_port = 0;
    k->ws_ticket = NULL;

}

/*
 *  static const struct {
 *    char *name;
 *    int   nid;
 *  } cert_struct[] = {
 *    { "C",     NID_countryName            },
 *    { "ST",    NID_stateOrProvinceName    }, 
 *    { "SP",    NID_stateOrProvinceName    }, 
 *    { "L",     NID_localityName           },
 *    { "O",     NID_organizationName       },
 *    { "OU",    NID_organizationalUnitName },
 *    { "CN",    NID_commonName             },
 *    { "T",     NID_title                  },
 *    { "I",     NID_initials               },
 *    { "G",     NID_givenName              },
 *    { "S",     NID_surname                },
 *    { "D",     NID_description            },
 *    { "UID",   NID_uniqueIdentifier       },
 *    { "Email", NID_pkcs9_emailAddress     },
 *    { NULL,    0                          }
 *  };
 */

static int
retrieve_implicit_identity(kct_context * context, char **id, X509 * cert)
{

    X509_NAME *client;
    char   *tmp;
    int     result;
    int     retval = 0;

    kct_log(KCTLOG_DEBUG,
	    "retrieve_implicit_identity: retrieving identity "
	    "from the commonName of certificate\n");

    client = X509_get_subject_name(cert);
    tmp = (char *) get_memory(KCT_MAX_BUF_SIZE, sizeof(char));
    (*id) = (char *) get_memory(KCT_MAX_ID_LEN, sizeof(char));

    client = X509_get_subject_name(cert);
    result = X509_NAME_get_text_by_NID(client, NID_commonName,
				       tmp, KCT_MAX_BUF_SIZE);
    if (result < 0) {
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto err;
    }

    sprintf(*id, "%s/%s", kct_implicit_service_name, tmp);

    kct_log(KCTLOG_DEBUG, "retrieve_implicit_identity: id='%s'\n", *id);

  err:
    free_memory(tmp);

    return retval;
}

static int
retrieve_explicit_identity(kct_context * context, char **id,
			   char **realm, X509 * cert)
{

    X509_NAME *client;
    char   *tmp;
    int     result;
    int     retval = 0;

    kct_log(KCTLOG_DEBUG,
	    "retrieve_explicit_identity: retrieving identity "
	    "from the emailAddress of certificate\n");

    client = X509_get_subject_name(cert);
    tmp = (char *) get_memory(KCT_MAX_BUF_SIZE, sizeof(char));
    (*id) = (char *) get_memory(KCT_MAX_ID_LEN, sizeof(char));
    (*realm) = (char *) get_memory(KCT_MAX_REALM_LEN, sizeof(char));

    client = X509_get_subject_name(cert);
    result = X509_NAME_get_text_by_NID(client, NID_pkcs9_emailAddress, tmp,
				       KCT_MAX_BUF_SIZE);
    if (result < 0) {
	retval = KCT_ERR_INVALID_HANDSHAKE;
	goto err;
    }

    sscanf(tmp, "%[^@]@%s", (*id), (*realm));

    kct_log(KCTLOG_DEBUG, "retrieve_explicit_identity: full='%s' id='%s' "
	    "realm='%s'\n", tmp, *id, *realm);

  err:
    free_memory(tmp);

    return retval;
}
