/*
 *  linux/net/sunrpc/gss_spkm3_seal.c
 *
 *  Copyright (c) 2003 The Regents of the University of Michigan.
 *  All rights reserved.
 *
 *  Andy Adamson <andros@umich.edu>
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of the University nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <linux/types.h>
#include <linux/slab.h>
#include <linux/jiffies.h>
#include <linux/sunrpc/gss_spkm3.h>
#include <linux/random.h>
#include <linux/crypto.h>
#include <linux/pagemap.h>
#include <linux/scatterlist.h>

#ifdef RPC_DEBUG
# define RPCDBG_FACILITY        RPCDBG_AUTH
#endif

/*
 * spkm3_make_token()
 *
 * Only SPKM_MIC_TOK with md5 intg-alg is supported
 */

u32
spkm3_make_token(struct spkm3_ctx *ctx,
		   struct xdr_buf * text, struct xdr_netobj * token,
		   int toktype)
{
	s32			checksum_type;
	char			tokhdrbuf[25];
	char			cksumdata[16];
	struct xdr_netobj	md5cksum = {.len = 0, .data = cksumdata};
	struct xdr_netobj	mic_hdr = {.len = 0, .data = tokhdrbuf};
	int			tokenlen = 0;
	unsigned char		*ptr;
	s32			now;
	int			ctxelen = 0, ctxzbit = 0;
	int			md5elen = 0, md5zbit = 0;

	now = jiffies;

	if (ctx->ctx_id.len != 16) {
		dprintk("RPC: spkm3_make_token BAD ctx_id.len %d\n",
			ctx->ctx_id.len);
		goto out_err;
	}

	if(!g_OID_equal(&ctx->intg_alg, &hmac_md5_oid)) {
	        dprintk("RPC: gss_spkm3_seal: unsupported I-ALG algorithm."
		        "only support hmac-md5 I-ALG.\n");
		print_hexl((u32 *)ctx->intg_alg.data, ctx->intg_alg.len, 0);
	        goto out_err;
	} else {
		checksum_type = CKSUMTYPE_HMAC_MD5;
	}


	if (!g_OID_equal(&ctx->conf_alg, &cast5_cbc_oid)) {
	        dprintk("RPC: gss_spkm3_seal: unsupported C-ALG algorithm\n");
	        goto out_err;
	}

	if (toktype == SPKM_MIC_TOK) {
		/* Calculate checksum over the mic-header */
		asn1_bitstring_len(&ctx->ctx_id, &ctxelen, &ctxzbit);
		spkm3_mic_header(&mic_hdr.data, &mic_hdr.len, ctx->ctx_id.data,
		                         ctxelen, ctxzbit);
                if (make_spkm3_checksum(checksum_type, &ctx->derived_integ_key,
					(char *)mic_hdr.data, mic_hdr.len,
					text, 0, &md5cksum))
                        goto out_err;

		asn1_bitstring_len(&md5cksum, &md5elen, &md5zbit);
		tokenlen = 10 + ctxelen + 1 + md5elen + 1;

		/* Create token header using generic routines */
		token->len = g_token_size(&ctx->mech_used, tokenlen);

		ptr = token->data;
		g_make_token_header(&ctx->mech_used, tokenlen, &ptr);

		spkm3_make_mic_token(&ptr, tokenlen, &mic_hdr, &md5cksum, md5elen, md5zbit);
	} else if (toktype == SPKM_WRAP_TOK) { /* Not Supported */
		dprintk("RPC: gss_spkm3_seal: SPKM_WRAP_TOK not supported\n");
		goto out_err;
	}

	/* XXX need to implement sequence numbers, and ctx->expired */

	return  GSS_S_COMPLETE;
out_err:
        if(md5cksum.data)
		kfree(md5cksum.data);

	token->data = NULL;
	token->len = 0;
	return GSS_S_FAILURE;
}

static int
spkm3_process_xdr_buf(struct xdr_buf *buf, int offset, int len,
                int (*actor)(struct scatterlist *, void *), void *data)
{
        int i, page_len, thislen, page_offset, ret = 0;
        struct scatterlist      sg[1];

        if (offset >= buf->head[0].iov_len) {
                offset -= buf->head[0].iov_len;
        } else {
                thislen = buf->head[0].iov_len - offset;
                if (thislen > len)
                        thislen = len;
                sg_set_buf(sg, buf->head[0].iov_base + offset, thislen);
                ret = actor(sg, data);
                if (ret)
                        goto out;
                offset = 0;
                len -= thislen;
        }
        if (len == 0)
                goto out;

        if (offset >= buf->page_len) {
                offset -= buf->page_len;
        } else {
                page_len = buf->page_len - offset;
                if (page_len > len)
                        page_len = len;
                len -= page_len;
                page_offset = (offset + buf->page_base) & (PAGE_CACHE_SIZE - 1);                i = (offset + buf->page_base) >> PAGE_CACHE_SHIFT;
                thislen = PAGE_CACHE_SIZE - page_offset;
                do {
                        if (thislen > page_len)
                                thislen = page_len;
                        sg->page = buf->pages[i];
                        sg->offset = page_offset;
                        sg->length = thislen;
                        ret = actor(sg, data);
                        if (ret)
                                goto out;
                        page_len -= thislen;
                        i++;
                        page_offset = 0;
                        thislen = PAGE_CACHE_SIZE;
                } while (page_len != 0);
                offset = 0;
        }
        if (len == 0)
                goto out;

        if (offset < buf->tail[0].iov_len) {
                thislen = buf->tail[0].iov_len - offset;
                if (thislen > len)
                        thislen = len;
                sg_set_buf(sg, buf->tail[0].iov_base + offset, thislen);
                ret = actor(sg, data);
                len -= thislen;
        }
        if (len != 0)
                ret = -EINVAL;
out:
        return ret;
}


static int
spkm3_checksummer(struct scatterlist *sg, void *data)
{
        struct crypto_tfm *tfm = (struct crypto_tfm *)data;

        crypto_hmac_update(tfm, sg, 1);

        return 0;
}

/* checksum the plaintext data and hdrlen bytes of the token header */
s32
make_spkm3_checksum(s32 cksumtype, struct xdr_netobj *key, char *header,
		    int hdrlen, struct xdr_buf *body,
		    int body_offset, struct xdr_netobj *cksum)
{
        char                            *cksumname;
        struct crypto_tfm               *tfm = NULL; /* XXX add to ctx? */
        struct scatterlist              sg[1];

        switch (cksumtype) {
                case CKSUMTYPE_HMAC_MD5:
                        cksumname = "md5";
                        break;
                default:
                        dprintk("RPC:      spkm3_make_checksum:"
                                " unsupported checksum %d", cksumtype);
                        return GSS_S_FAILURE;
        }

	if(key->data == NULL || key->len <= 0) return GSS_S_FAILURE;

        if (!(tfm = crypto_alloc_tfm(cksumname, CRYPTO_TFM_REQ_MAY_SLEEP)))
                return GSS_S_FAILURE;
        cksum->len = crypto_tfm_alg_digestsize(tfm);

	crypto_hmac_init(tfm, key->data, &key->len);
        sg_set_buf(sg, header, hdrlen);
        crypto_hmac_update(tfm, sg, 1);

        spkm3_process_xdr_buf(body, body_offset, body->len - body_offset,
			      spkm3_checksummer, tfm);
        crypto_hmac_final(tfm, key->data, &key->len, cksum->data);
        crypto_free_tfm(tfm);

        return 0;
}

EXPORT_SYMBOL(make_spkm3_checksum);
