/*
 * Dean Hildebrand
 *
 * This state machine injects the required information into the PVFS2
 * client cache to perform I/O for pNFS.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#define __PINT_REQPROTO_ENCODE_FUNCS_C
#include "gossip.h"
#include "pvfs2-debug.h"
#include "job.h"
#include "str-utils.h"
#include "pint-servreq.h"
#include "pvfs2-attr.h"
#include "acache.h"
#include "pvfs2-internal.h"

extern void PINT_free_object_attr(PVFS_object_attr *attr);
static int setlayout_inject(PVFS_object_ref refn, void* layout);

/* DH: This is the entry function for the state machine */
PVFS_error PVFS_sys_setlayout(
    PVFS_object_ref ref,
    void* layout,
    PVFS_credentials *credentials)
{
    int ret = -PVFS_EINVAL;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "PVFS_sys_setlayout entered\n");

    if ((ref.handle == PVFS_HANDLE_NULL) ||
	(ref.fs_id == PVFS_FS_ID_NULL)) {
	gossip_err("PVFS_sys_setlayout: invalid (NULL) required argument\n");
	return ret;
    }

    gossip_debug(GOSSIP_CLIENT_DEBUG,
		 "Doing setlayout on handle %llu on fs %d\n",
		 ref.handle, ref.fs_id);

    ret = setlayout_inject(ref, layout);

    return ret;
}

/****************************************************************/

/* Deserialize the layout (dfiles and dist) and set in the
 * cache attribute struct
*/
static int deserialize_layout(char* layout, PVFS_metafile_attr* meta)
{
    int blob_size=0, fs_id=0;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: Begin\n");

    /* Size of entire opaque object */
    decode_int32_t(&layout, &blob_size);
    /* Size of entire opaque object */
    decode_int32_t(&layout, &fs_id);
    gossip_debug(GOSSIP_CLIENT_DEBUG,
		 "deserialize_layout: #bs:%d fsid:%d\n",
		 blob_size, fs_id);

    /* Deserialize dfile array */
    decode_PVFS_metafile_attr_dfiles(&layout, meta);
    gossip_debug(GOSSIP_CLIENT_DEBUG,
		 "deserialize_layout: #dfiles: %d dfile #0: %llu \n",
		 meta->dfile_count, llu(meta->dfile_array[0]));

    /* Deserialize distribution struct */
    decode_PVFS_metafile_attr_dist(&layout, meta);
    gossip_debug(GOSSIP_CLIENT_DEBUG,
		 "deserialize_layout: #ds: %d\n",
		 meta->dist_size);
    PINT_dist_dump(meta->dist);

    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: End\n");
    return 0;
}

/* DH: Place layout in cache */
static int setlayout_inject(PVFS_object_ref refn, void* layout)
{
    int ret = -PVFS_EINVAL;
    PVFS_object_attr attr;
    PVFS_size tempsz = 0;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_inject: Begin\n");

    if (layout == NULL)
    {
	gossip_err("setlayout_inject: Layout is NULL!\n");
	ret = -EIO;
	goto out;
    }

    /* Set (mostly false) attributes on the cached inode */
    attr.mask = (PVFS_ATTR_META_ALL | PVFS_ATTR_COMMON_ALL);
    /*    attr.mask &= !(PVFS_ATTR_SYMLNK_TARGET); */
    attr.objtype = PVFS_TYPE_METAFILE;

    /* Decode the blob layout */
    if ((ret = deserialize_layout((char*)layout, &attr.u.meta)))
    {
	gossip_err("setlayout_inject: Could not deserialize layout %d!\n",ret);
	goto out;
    }

    ret = PINT_acache_update(refn, &attr, &tempsz);
    if (ret)
    {
	gossip_err("setlayout_inject: Could not set layout in cache %d!\n",ret);
    }

    PINT_free_object_attr(&attr);
out:
    gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_inject: End\n");
    return ret;
}

/*
 * Local variables:
 *  mode: c
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 *
 * vim: ft=c ts=8 sts=4 sw=4 noexpandtab
 */
