/*
 * Dean Hildebrand
 *
 * This state machine injects the required information into the PVFS2
 * client cache to perform I/O for pNFS.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#define __PINT_REQPROTO_ENCODE_FUNCS_C
#include "gossip.h"
#include "client-state-machine.h"
#include "state-machine-fns.h"
#include "pvfs2-debug.h"
#include "job.h"
#include "str-utils.h"
#include "pint-servreq.h"
#include "pvfs2-attr.h"

extern job_context_id pint_client_sm_context;

static int setlayout_inject(
    PINT_client_sm *sm_p, job_status_s *js_p);
static int setlayout_failure(
    PINT_client_sm *sm_p, job_status_s *js_p);
static int setlayout_cleanup(
    PINT_client_sm *sm_p, job_status_s *js_p);


static union PINT_state_array_values ST_setlayout_inject[];
static union PINT_state_array_values ST_setlayout_failure[];
static union PINT_state_array_values ST_cleanup[];

struct PINT_state_machine_s pvfs2_client_setlayout_sm =
{
	ST_setlayout_inject,
	"pvfs2_client_setlayout_sm"
};

static union PINT_state_array_values ST_setlayout_inject[] = {
(union PINT_state_array_values) 0,
(union PINT_state_array_values) setlayout_inject,
(union PINT_state_array_values) 0,
(union PINT_state_array_values) ST_cleanup,
(union PINT_state_array_values) -1,
(union PINT_state_array_values) ST_setlayout_failure
};

static union PINT_state_array_values ST_setlayout_failure[] = {
(union PINT_state_array_values) 0,
(union PINT_state_array_values) setlayout_failure,
(union PINT_state_array_values) -1,
(union PINT_state_array_values) ST_cleanup
};

static union PINT_state_array_values ST_cleanup[] = {
(union PINT_state_array_values) 0,
(union PINT_state_array_values) setlayout_cleanup,
(union PINT_state_array_values) -1,
(union PINT_state_array_values) 7
};

/* DH: Kicks off state machine */
PVFS_error PVFS_isys_setlayout(
    PVFS_object_ref ref,
    void* layout,
    PVFS_credentials *credentials,
    PVFS_sys_op_id *op_id,
    void *user_ptr)
{
    PVFS_error ret = -PVFS_EINVAL;
    PINT_client_sm *sm_p = NULL;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "PVFS_isys_setlayout entered\n");

    if ((ref.handle == PVFS_HANDLE_NULL) ||
	(ref.fs_id == PVFS_FS_ID_NULL))
    {
	gossip_err("invalid (NULL) required argument\n");
	return ret;
    }

    sm_p = (PINT_client_sm *)malloc(sizeof(*sm_p));
    if (sm_p == NULL)
    {
        return -PVFS_ENOMEM;
    }
    memset(sm_p, 0, sizeof(*sm_p));

    PINT_init_msgarray_params(&sm_p->msgarray_params, ref.fs_id);
    PINT_init_sysint_credentials(sm_p->cred_p, credentials);
    sm_p->object_ref = ref;
    sm_p->u.setlayout.refn = ref;
    sm_p->u.setlayout.layout = layout;
    sm_p->error_code = 0;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "Doing setlayout on handle %Lu on fs %d\n",
		 Lu(ref.handle),
                 ref.fs_id);

    return PINT_client_state_machine_post(
        sm_p, PVFS_SYS_SETLAYOUT, op_id, user_ptr);
}

/* DH: This is the entry function for the state machine */
PVFS_error PVFS_sys_setlayout(
    PVFS_object_ref ref,
    void* layout,
    PVFS_credentials *credentials)
{
    int ret = -PVFS_EINVAL;
    PVFS_error error;
    PVFS_sys_op_id op_id;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "PVFS_sys_setlayout entered\n");

    ret = PVFS_isys_setlayout(ref, layout, credentials, &op_id, NULL);
    if (ret)
    {
        PVFS_perror_gossip("PVFS_isys_setlayout call", ret);
        error = ret;
    }
    else
    {
        ret = PINT_sys_wait(op_id, "setlayout", &error);
        if (ret)
        {
            PVFS_perror_gossip("PVFS_sys_wait call", ret);
            error = ret;
        }
    }

    PINT_sys_release(op_id);
    return error;
}

/****************************************************************/

/* Deserialize the layout (dfiles and dist) and set in the
 * cache attribute struct
*/
static int deserialize_layout(void* layout, PVFS_metafile_attr* meta)
{
    int blob_size=0, dfile_count=0, fs_id=0;
    PVFS_size df_array_size;
    char* lay_temp;
    int i;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: Begin\n");

    /* Free previous layout info if it existed */
    if (meta->dfile_count > 0)
    {
	if (meta->dfile_array)
	{
	    free(meta->dfile_array);
	}
    }
    PINT_dist_free(meta->dist);

    /* Size of entire opaque object */
    decode_int32_t(&layout, &blob_size);

    /* Size of entire opaque object */
    decode_int32_t(&layout, &fs_id);

    /* Number of dfiles */
    decode_int32_t(&layout, &dfile_count);

    /* Deserialize dfile array */
    df_array_size = dfile_count * sizeof(PVFS_handle);
    meta->dfile_array = malloc(df_array_size);
    if (!meta->dfile_array)
    {
	return -ENOMEM;
    }

    for (i=0; i < dfile_count; i++)
        decode_PVFS_handle(&layout, &meta->dfile_array[i]);

    meta->dfile_count = dfile_count;
    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: #bs:%d #df:%d\n", blob_size, dfile_count);
    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: dfile handle %Lu \n", Lu(meta->dfile_array[0]));

    /* Deserialize distribution struct */
    lay_temp = (char*)layout;
    decode_PVFS_metafile_attr_dist(&lay_temp, meta);

    PINT_dist_dump(meta->dist);
    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: #ds: %d\n", meta->dist_size);
    gossip_debug(GOSSIP_CLIENT_DEBUG, "deserialize_layout: End\n");
    return 0;
}

/* DH: Place layout in cache */
static int setlayout_inject(PINT_client_sm *sm_p,
			    job_status_s *js_p)
{
    int ret = -PVFS_EINVAL;
    PVFS_object_attr *attr = NULL;
    int release_required = 1;
    int status = 0;

    gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_inject: Begin\n");

    js_p->error_code = 0;
    if (sm_p->u.setlayout.layout == NULL)
    {
	gossip_err("setlayout_inject: Layout is NULL!\n");
	js_p->error_code = -EIO;
	goto out;
    }

    /* See if file exists in the cache.  If it does, we replace the layout */
    PINT_pinode *pinode = PINT_acache_lookup(sm_p->object_ref, &status, NULL);
    if (!pinode)
    {
	gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_inject: Creating new cache object\n");
	pinode = PINT_acache_pinode_alloc();
	if(!pinode)
        {
            js_p->error_code = -PVFS_ENOMEM;
            return 1;
        }
	release_required = 0;
    }
    pinode->refn = sm_p->u.setlayout.refn;
    attr = &pinode->attr;

    /* Set (mostly false) attributes on the cached inode */
    attr->mask |= (PVFS_ATTR_META_ALL | PVFS_ATTR_COMMON_ALL);
    attr->objtype = PVFS_TYPE_METAFILE;

    /* Decode the blob layout */
    if ((ret = deserialize_layout(sm_p->u.setlayout.layout, &attr->u.meta)))
    {
	gossip_err("setlayout_inject: Could not deserialize layout %d!\n",ret);
	js_p->error_code = ret;
	goto out;
    }

    gossip_debug(GOSSIP_CLIENT_DEBUG,
		 "setlayout_inject: new cache object fsid:%d handle:%Lu\n",
		 pinode->refn.fs_id,
		 Lu(pinode->refn.handle));
    PINT_acache_set_valid(pinode);
    if (release_required)
    {
	PINT_acache_release(pinode);
    }

out:
    gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_inject: End\n");
    return 1;
}

static int setlayout_failure(PINT_client_sm *sm_p,
					  job_status_s *js_p)
{
    gossip_debug(GOSSIP_CLIENT_DEBUG,
        "(%p) setlayout state: setlayout_failure\n", sm_p);

    if (js_p->error_code != 0)
    {
        gossip_err("setlayout: Failed with unexpected error code %d\n",
                   js_p->error_code);
    }
    return 1;
}

static int setlayout_cleanup(PINT_client_sm *sm_p,
                           job_status_s *js_p)
{
    gossip_debug(GOSSIP_CLIENT_DEBUG, "setlayout_cleanup (%d)\n",js_p->error_code);

    sm_p->error_code = js_p->error_code;
    sm_p->op_complete = 1;
    return 0;
}

/*
 * Local variables:
 *  mode: c
 *  c-indent-level: 4
 *  c-basic-offset: 4
 * End:
 *
 * vim: ft=c ts=8 sts=4 sw=4 noexpandtab
 */
