/*
 * COPYRIGHT    2002
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <libnet.h>
#include <err.h>
#include <stdio.h>
#include <signal.h>
#ifdef __linux__
/* Includes needed for the "local" option. */
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
/* #include <linux/if.h> */
#include <linux/if_tun.h>
#endif

/* Given a list of size/count pairs (specified by -s and -c on the
   command-line), spit attempts to send count packets per second of
   each corresponding size. */

/* There's one spitter_struct for each size/count pair: */
struct spitter_struct {
  u_char * packet;
  int size;
  int target_rate; /* number of packets/second to send, given by -c on the
		      command line */
  int tried, sent; /* tried, sent are reset to zero after
		      target_rate packets are sent*/
  int total_sent; /* total sent during execution of this process */
  int source_spoof_range;
  int dest_spoof_range;
  int src_addr;
  int dst_addr;
  struct spitter_struct * next;
};

struct spitter_struct * spitter_head = NULL;
#ifdef __OpenBSD__
char * device;
#else
u_char *device;
#endif

int tap0_fd;
int local = 0;

struct libnet_link_int *network;

u_char enet_src[6] = {0x00, 0x01, 0x02, 0x03, 0x04, 0x05};
u_char enet_dst[6] = {0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b};

void usage(char * prog) {
  fprintf(stderr,
	  "usage: %s [-i device] -s size -c count"
#ifdef __linux__
	  " [-l]"
#endif
	  "\n"
	  "       [-m source_spoof_range] [-n dest_spoof_range] ...\n"
	  "Attempts to send count packets per second of the given size.\n"
	  "(The size includes the 14-byte ethernet header.)  The source IP\n"
	  "addresses will cycle through source_spoof_range different\n"
	  "addresses of the form 10.x.x.x; dest_spoof_range works similarly.\n"
	  "A spoof_range of 0 means 2^24; default is 1, making the addresses\n"
	  "constant.  Arbitrarily many different packet sizes, each with\n"
	  "different -c, -m, and -n options, may be specified.\n\n"
	  "Note that the packets sent are not actually legal IP packets, but\n"
	  "they look like IP packets to the vault.\n"
#ifdef __linux__
	  "On linux, the -l option creates a virtural ethernet device tap0\n"
	  "which packets will be written to, never appearing on any physical\n"
	  "network.  This will only work if tun/tap support is compiled into\n"
	  "the kernel.  This option appears to use a lot of CPU, and\n"
	  "probably isn't actually very useful.\n"
#endif
	  ,
	  prog);
  exit(1);
}

#ifdef __linux__
void open_tap() {
   struct ifreq ifr;

   if ((tap0_fd = open("/dev/net/tun", O_RDWR)) < 0 )
     err(1,"open failed");
   
   memset(&ifr, 0, sizeof(ifr));
   ifr.ifr_flags = IFF_TAP | IFF_NO_PI;
   
   if (ioctl(tap0_fd, TUNSETIFF, (void *) &ifr) < 0) {
     close(tap0_fd);
     err(1,"ioctl failed");
   }
   
   printf("created device %s\n",ifr.ifr_name);
}
#endif

void spitter() {
  struct timeval then, now;
  float elapsed = 0;
  struct spitter_struct * this_spitter;
  float measured_rate;
  int total_count = 0;
  int delay_loop;
  int j;
  int rv;

  for (this_spitter = spitter_head;
       this_spitter != NULL;
       this_spitter = this_spitter->next) {
    this_spitter->src_addr = 1;
    this_spitter->dst_addr = 1;
    ((struct ip *)(this_spitter->packet+LIBNET_ETH_H))->ip_src.s_addr
      = htonl((this_spitter->src_addr & 0x00FFFFFF) | 0x0a000000);
    ((struct ip *)(this_spitter->packet+LIBNET_ETH_H))->ip_dst.s_addr
      = htonl((this_spitter->dst_addr & 0x00FFFFFF) | 0x0a000000);
    this_spitter->total_sent = 0;
    total_count += this_spitter->target_rate;
  }

  /* initial wild guess at delay loop: */
  delay_loop = 10000000/total_count;
  gettimeofday(&then,NULL);
  while (1) {
    for (this_spitter = spitter_head;
	 this_spitter != NULL;
	 this_spitter = this_spitter->next) {
      this_spitter->sent = this_spitter->tried = 0;
      while (this_spitter->sent < this_spitter->target_rate) {
	this_spitter->tried++;
	if (local) {
	    rv = write(tap0_fd,this_spitter->packet,this_spitter->size);
	} else {
	    rv = libnet_write_link_layer(network, device, this_spitter->packet,
					 this_spitter->size);
	}
	if (rv == this_spitter->size) { /* succesful send: */
	  this_spitter->total_sent++;
	  this_spitter->sent++;
	  /* Spoof addresses if necessary: */
	  if (this_spitter->source_spoof_range != 1) {
	    if (this_spitter->src_addr == this_spitter->source_spoof_range)
	      this_spitter->src_addr = 1;
	    else
	      this_spitter->src_addr++;
	    ((struct ip *)(this_spitter->packet+LIBNET_ETH_H))->ip_src.s_addr
	      = htonl((this_spitter->src_addr & 0x00FFFFFF) | 0x0a000000);
	  }
	  if (this_spitter->dest_spoof_range != 1) {
	    if (this_spitter->dst_addr == this_spitter->dest_spoof_range)
	      this_spitter->dst_addr = 1;
	    else
	      this_spitter->dst_addr++;
	    ((struct ip *)(this_spitter->packet+LIBNET_ETH_H))->ip_dst.s_addr
	      = htonl((this_spitter->dst_addr & 0x00FFFFFF) | 0x0a000000);
	  }
	}
	for (j=0; j<delay_loop; j++)
	  ;
      }
    }
    gettimeofday(&now,NULL);
    /* Print statistics and adjust delay loop: */
    for (this_spitter = spitter_head;
	 this_spitter != NULL;
	 this_spitter = this_spitter->next) {
      elapsed = (now.tv_sec - then.tv_sec)
	+ (now.tv_usec - then.tv_usec)/1000000.0;
      measured_rate = this_spitter->sent/elapsed;
      printf("%.0f packets/second sent of size %d (%.2f Mb/s),",
	     measured_rate,
	     this_spitter->size,
	     measured_rate * (this_spitter->size) * 8.0 /1024 /1024);
      printf("%.0f%% success rate\n",
	     100 * (float)this_spitter->sent / this_spitter->tried);
    }
    delay_loop = delay_loop / elapsed;
    memcpy(&then,&now,sizeof(now));
  }
}

void cleanup(int signal) {
  struct spitter_struct * this_spitter;

  printf("packets sent:\n");
  for (this_spitter = spitter_head;
	 this_spitter != NULL;
	 this_spitter = this_spitter->next) {
    printf("%d packets of size %d\n",
	   this_spitter->total_sent, this_spitter->size);
  }
  if (local) {
      close(tap0_fd);
  } else {
      if (libnet_close_link_interface(network) == -1)
	  libnet_error(LN_ERR_WARNING,
		       "libnet_close_link_interface couldn't close"
		       "the interface");
  }
  exit(0);
}

int main(int argc, char *argv[]) {
  char err_buf[LIBNET_ERRBUF_SIZE];
  extern int optind;
  extern char *optarg;
  char optch;
  struct spitter_struct * this_spitter;
  
  device = NULL;
  
  while ((optch = getopt(argc, argv, "c:i:lm:n:s:")) != -1) {
    switch (optch) {
    case 'c':
      if (spitter_head == NULL)
	errx(1,"-c rate must follow corresponding -s size");
      spitter_head->target_rate = atoi(optarg);
      break;
    case 'i':
      device = optarg;
      break;
    case 'l':
#ifdef __linux__
	local = 1;
	break;
#else
	errx(1,"The -l option is currently only available on Linux.");
#endif
    case 'm':
      if (spitter_head == NULL)
	errx(1,"-m source_spoof_range must follow corresponding -s size");
      spitter_head->source_spoof_range = strtol(optarg, NULL, 0);
      break;
    case 'n':
      if (spitter_head == NULL)
	errx(1,"-n dest_spoof_range must follow corresponding -s size");
      spitter_head->dest_spoof_range = strtol(optarg, NULL, 0);
      break;
    case 's':		/* size of packet to send (incl. ether header) */
      this_spitter = (struct spitter_struct *)
	malloc(sizeof(struct spitter_struct));
      this_spitter->next = spitter_head;
      spitter_head = this_spitter;
      spitter_head->size = strtol(optarg, NULL, 0);
      if (spitter_head->size < 60)
	errx(1, "packet size too small: %s", optarg);
      if (spitter_head->size > 1514)
	errx(1, "packet size too large: %s", optarg);
      /* set defaults: */
      spitter_head->target_rate = 1;
      spitter_head->source_spoof_range = 1;
      spitter_head->dest_spoof_range = 1;
      break;
    default:
      usage(argv[0]);
    }
  }

  if (spitter_head == NULL)
    usage(argv[0]);

  if (local) {
#ifdef __linux__
      open_tap();
#endif
  } else {
      if (device == NULL) {
	  struct sockaddr_in sin;
	  if (libnet_select_device(&sin, &device, err_buf) == -1)
	      libnet_error(LIBNET_ERR_FATAL,
			   "libnet_select_device failed: %s\n",
			   err_buf);
	  printf("assuming device %s\n", device);
      } else
	  printf("using device %s\n", device);
      if ((network = libnet_open_link_interface(device, err_buf)) == NULL)
	  libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface: %s\n",
		       err_buf);
  }
  for (this_spitter = spitter_head;
       this_spitter != NULL;
       this_spitter = this_spitter->next) {
    if (libnet_init_packet(this_spitter->size,&this_spitter->packet) == -1)
      libnet_error(LIBNET_ERR_FATAL,"libnet_init_packet failed\n");
    libnet_build_ethernet(enet_dst,
			  enet_src,
			  ETHERTYPE_IP,
			  NULL,
			  0,
			  this_spitter->packet);
  }  

  if (signal(SIGINT,cleanup) == SIG_ERR)
    errx(1,"Failed to set SIGINT handler");

  spitter();

  /* spitter has an infinite loop, so we should never end up here: */
  exit(1);
}

