/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 * 
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 * 
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <sys/ipc.h>   
#include <sys/sem.h>   
#include <sys/shm.h>   
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#ifdef __linux__
#define IPC_R 000400
#define IPC_W 000200
#endif

/********************************************************************
 * Structure definitions
 ********************************************************************/

typedef struct apv_slot {
    int volid;			/* The volid this slot represents */
    short status;		/* The current status of this slot */
    short retries;		/* The number of times a consumer has
				   failed to process this volid */
    struct apv_slot * next;	/* Next on the list */
} apv_slot;

/*
 * Values for apv_slot status field
 */
#define APVSLOT_STATUS_AVAILABLE	1
#define APVSLOT_STATUS_FILLED		2
#define APVSLOT_STATUS_DRAINING		3

#define APV_MAX_RETRIES			3

typedef struct apv_list_anchor {
    apv_slot * head;
    apv_slot * tail;
    apv_slot * free;
} apv_list_anchor;
apv_list_anchor * apv_list = NULL;

/*
 * REQ_APV_LIST_ADDR is the address where we want the
 * apv_list shared memory to be mapped.  We need
 * a consistent address, since we are using pointers
 * within that memory to manage linked lists.
 */
#define REQ_APV_LIST_ADDR 0x00500000

typedef struct apv_control_anchor {
    int semid;
    int shmid;
} apv_control_anchor;
apv_control_anchor * apv_cntl = NULL;

/*
 * The keys used for the different IPC pieces
 */
#define APV_CNTL_SHM_KEY	77
#define APV_LIST_SEM_KEY	88
#define APV_LIST_SHM_KEY	99

extern int errno;

#define SEMPERM 0600		/* Permissions for the semaphores, RW for the owner */
#define TRUE 1
#define FALSE 0

#define APV_SEM_MUTEX 0		/* Protects the shared memory */
#define APV_SEM_FILLED 1	/* How many slots are filled */
#define APV_SEM_AVAIL 2		/* How many slots are available to be filled */
#define APV_NUM_SEMS 3

/*
 * The number of slots and (tape) drives may be specified at compile-time
 * by defining APV_SLOTS and APV_DRIVES appropriately.
 */
#define APV_DEFAULT_NUM_SLOTS 35	/* Total number of slots available */
#define APV_DEFAULT_NUM_DRIVES 1	/* Total number of tape drives available */

#if defined(APV_SLOTS)
int num_slots = APV_SLOTS;
#else
int num_slots = APV_DEFAULT_NUM_SLOTS;
#endif

#if defined(APV_DRIVES)
int num_drives = APV_DRIVES;
#else
int num_drives = APV_DEFAULT_NUM_DRIVES;
#endif

int    debug = 0;

/********************************************************************
 * Function prototypes
 ********************************************************************/
int    apv_init();
int    apv_fini();
int    apv_destroy();
int    apv_produce(int volid);
int    apv_consume_get();
int    apv_consume_complete(int volid);
int    apv_consume_retry(int volid);

/* Shared memory routines */
int    shm_get(int key);
char * shm_attach(int shmid, char * fixed_addr);
int    shm_detach(char * shmaddr);
int    shm_delete(int shmid);
int    shm_list_initialize(int shmid);
int    shm_control_initialize(int shmid);

/* Semaphore routines */
int    sem_delete(int semid);
int    sem_get(int key);
int    sem_post(int semid, int whichsem);
int    sem_trywait(int semid, int whichsem);
int    sem_wait(int semid, int whichsem);

/* List processing routines */
int    list_add_tail(int volid);
apv_slot * list_find_filled();
apv_slot * list_find_slot_by_volid(int volid);
int    list_free_slot(apv_slot *slot);
int    list_print_status();

/* General utility routines */
void   usage(char *progname);
char * status_to_string(int status);
char * sem_to_string(int whichsem);

/********************************************************************
 * Semaphore routines
 ********************************************************************/

char * sem_to_string(int whichsem)
{
    if (whichsem == APV_SEM_MUTEX)
	return "MUTEX";
    else if (whichsem == APV_SEM_FILLED)
	return "FILLED";
    else if (whichsem == APV_SEM_AVAIL)
	return "AVAIL";
    else
	return "-----";
}

/*
 * NOTE: because of the use of SEM_UNDO, if you invoke sem_wait,
 * you get the semaphore.  But when your program ends, it is
 * automatically released.  So when you run it again, it gets it
 * again and then again automagically releases it when the program
 * ends.  We always specify SEM_UNDO for the the mutex semaphore
 * since we would deadlock if someone went away holding that.
 * The other semaphores need to live across the life of this
 * program.  If SEM_UNDO was specified for them, then any
 * changes to the filled or avail semaphores would be nullified.
 */

int sem_wait(int semid, int whichsem)    /* wait on a semaphore */
{
    struct sembuf p_buf;

    p_buf.sem_num = whichsem;
    p_buf.sem_op = -1;
    if (whichsem == APV_SEM_MUTEX) 
	p_buf.sem_flg = SEM_UNDO;
    else
	p_buf.sem_flg = 0;

    if (debug) fprintf(stderr, "About to wait on semid %d, sem %-8s\n", 
		semid, sem_to_string(whichsem));

    if (semop(semid, &p_buf, 1) == -1) {
	perror("sem_wait() failed");
	exit(1);
    } else  
	return (0);
}

int sem_trywait(int semid, int whichsem) /* get a semaphore, but don't wait */
{
    struct sembuf p_buf;

    p_buf.sem_num = whichsem;
    p_buf.sem_op = -1;
    if (whichsem == APV_SEM_MUTEX) 
	p_buf.sem_flg = SEM_UNDO | IPC_NOWAIT;
    else
	p_buf.sem_flg = IPC_NOWAIT;

    if (semop(semid, &p_buf, 1) == -1) {
	if (errno == EWOULDBLOCK)
	    return(-1);
	perror("sem_trywait() failed");
	exit(1);
    } else {
	return (0);
    }
}

int sem_post(int semid, int whichsem)    /* post to a semaphore */
{
    struct sembuf v_buf;

    v_buf.sem_num = whichsem;
    v_buf.sem_op = 1;
    if (whichsem == APV_SEM_MUTEX) 
	v_buf.sem_flg = SEM_UNDO;
    else
	v_buf.sem_flg = 0;

    if (debug) fprintf(stderr, "About to post on semid %d, sem %-8s\n", 
		semid, sem_to_string(whichsem));

    if (semop(semid, &v_buf, 1) == -1) {
	perror("sem_post() failed");
	exit(1);
    } else {
	return (0);
    }
}

/*
 *
 */
int sem_get(int key)
{
    int semid;
    int rc;

    if ((semid = semget(key, APV_NUM_SEMS, 0)) < 0) {
	if (errno == ENOENT) {
	    if ((semid = semget(key, APV_NUM_SEMS,
			IPC_R | IPC_W | IPC_CREAT | IPC_EXCL)) < 0) {
		return -1;
	    }

	    /*
	     * Initialize the semaphores with an initial value
	     */
	    rc = semctl(semid, APV_SEM_MUTEX, SETVAL, 1);
	    if (rc) {
		perror("sem_get: SETVAL failed for APV_SEM_MUTEX");
		return -1;
	    }
	    rc = semctl(semid, APV_SEM_AVAIL, SETVAL, num_slots);
	    if (rc) {
		perror("sem_get: SETVAL failed for APV_SEM_AVAIL");
		return -1;
	    }
	    rc = semctl(semid, APV_SEM_FILLED, SETVAL, 0);
	    if (rc) {
		perror("sem_get: SETVAL failed for APV_SEM_FILLED");
		return -1;
	    }
	}
	else {
	    return -1;
	}
    }
    if (debug) fprintf(stderr, "The returned semid is %d\n", semid);
    return semid;
}

int sem_delete(int semid)
{
    int rc;

    if ((rc = semctl(semid, APV_NUM_SEMS, IPC_RMID, 0)) < 0) {
	fprintf(stderr, "sem_delete: removing semid %d failed\n", semid);
	perror("sem_delete: REMOVE failed");
	return -1;
    }
    return 0;
}

int sem_print(int semid)
{
    int rc;
    unsigned short semvals[APV_NUM_SEMS];

    rc = semctl(semid, APV_NUM_SEMS, GETALL, &semvals);
    if (rc) {
	perror("sem_print: GETALL failed for APV_SEM_MUTEX");
	return -1;
    }
    fprintf(stdout, "The semaphore values are: MUTEX %d, AVAIL %d, FILLED %d\n",
	semvals[APV_SEM_MUTEX], semvals[APV_SEM_AVAIL],
	semvals[APV_SEM_FILLED]);

    return 0;
}

/********************************************************************
 * Shared Memory routines
 ********************************************************************/

int shm_get(int key)
{
    int shmid;
    int size;
    int allocated = 0;

    /*
     * First determine (by the desired key) what size of
     * shared memory we should be asking for.
     */
    switch (key) {
    case APV_CNTL_SHM_KEY:
	size = sizeof(apv_control_anchor);
	break;
    case APV_LIST_SHM_KEY:
	size = sizeof(apv_list_anchor) + sizeof(apv_slot) * num_slots;
	break;
    default:
	fprintf(stderr, "shm_get: unknown key\n");
	return -1;
	break;
    }

    /*
     * Now try to get the shared memory.  If it fails the
     * first time because it doesn't exist, try to create it.
     */
    if ((shmid = shmget(key, size, 0)) < 0) {
	if (errno == ENOENT) {
	    if ((shmid = shmget(key, size,
			SHM_R | SHM_W | IPC_CREAT | IPC_EXCL)) < 0) {
		return -1;
	    }
	    allocated = 1;
	}
	else {
	    return -1;
	}
    }

    /*
     * We need to do initialization.  If we just allocated the
     * control shared memory, then we need to initialize it.
     * Otherwise, we need to attach to the control shared memory
     * and set apv_cntl with the correct address
     *
     * Note that all functions need to call apv_init()
     * (which calls this function) initially to set up apv_cntl
     * and apv_list.
     *
     * After performing their function, they should call
     * apv_fini() which will detach the control shared
     * memory.
     */
    switch (key) {
    case APV_CNTL_SHM_KEY:
	if (allocated) {
	    if (shm_control_initialize(shmid)) {
		shm_delete(shmid);
		return -1;
	    }
	}
	else {
	    if (((char *)apv_cntl = shm_attach(shmid, 0)) == (char *) -1) {
		return -1;
	    }
	    if (((char *)apv_list = shm_attach(apv_cntl->shmid,
				(char *)REQ_APV_LIST_ADDR)) == (char *) -1) {
		shm_detach((char *)apv_cntl);
		return -1;
	    }
	}
	break;
    case APV_LIST_SHM_KEY:
	if (allocated) {
	    if (shm_list_initialize(shmid)) {
		shm_delete(shmid);
		return -1;
	    }
	}
	break;
    }
    if (debug) fprintf(stderr, "The returned shmid for key %d, size %d, is %d\n",
	key, size, shmid);
    return shmid;

}

/*
 * We just created the control shared memory; we need to get/allocate
 * the semaphores and the list shared memory.  (And keep track of their
 * ids in the control shared memory.)
 */
int shm_control_initialize(int shmid)
{
    int rc = 0;

    if (((char *)apv_cntl = shm_attach(shmid, 0)) == (char *) -1)
	return -1;
    if ((apv_cntl->semid = sem_get(APV_LIST_SEM_KEY)) != -1) {
	if ((apv_cntl->shmid = shm_get(APV_LIST_SHM_KEY)) == -1) {
	    sem_delete(apv_cntl->semid);
	    rc = -1;
	}
    } else {
	rc = -1;
    }
    return rc;
}

int shm_list_initialize(int shmid)
{
    int size;
    apv_slot * slot;
    int i;

    if (((char *)apv_list = shm_attach(shmid, (char *)REQ_APV_LIST_ADDR)) == (char *) -1)
	return -1;
    
    /* Init the whole thing to zeros */
    size = sizeof(apv_list_anchor) + sizeof(apv_slot) * num_slots;
    memset(apv_list, '\0', size);

    /*
     * Now lets carve up the rest into a list and put them on the free list
     */
    apv_list->free = (apv_slot *) ((char *) apv_list + sizeof(apv_list_anchor));
    slot = apv_list->free;
    for (i = num_slots - 1; i > 0; i--) {
	slot->status = APVSLOT_STATUS_AVAILABLE;
	slot->next = ++slot;
    }
    /* Get the last one */
    slot->status = APVSLOT_STATUS_AVAILABLE;
    slot->next = NULL;

    return 0;
}


int shm_delete(int shmid)
{
    int rc;
    if ((rc = shmctl(shmid, IPC_RMID, (struct shmid_ds *)NULL)) < 0) {
	fprintf(stderr, "shm_delete: removing shmid %d failed\n", shmid);
	perror("shm_delete: REMOVE failed");
	return -1;
    }
    return 0;
}

/*
 * The second parameter specifies whether the memory need be
 * attached at a (the) fixed address.  If it is zero, then
 * it can be attached at any address.
 */
char * shm_attach(int shmid, char * fixed_addr)
{
    char * shm_addr;

    if ((shm_addr = shmat(shmid, fixed_addr, 0)) == (char *)-1) {
	perror("shm_attach failed");
	return (char *)-1;
    }
    if (fixed_addr && fixed_addr != shm_addr) {
	fprintf(stderr, "shm_attach: requested mapping at 0x%08x, "
			"but mapped at 0x%08x\n",
			(unsigned int) fixed_addr, (unsigned int) shm_addr);
	return (char *)-1;
    }

    if (debug) fprintf(stderr, "The shared memory has been attached at 0x%08x\n",
	(unsigned int) shm_addr);

    return shm_addr;
} 

int shm_detach(char * shmaddr)
{
    int rc;
    if ((rc = shmdt(shmaddr)) == -1) {
	perror("shm_detach failed");
	return -1;
    }
    return rc;
}

/********************************************************************
 * General APV routines
 ********************************************************************/

/*
 * This routine should be called at the beginning of each operation
 * to make sure that apv_cntl is properly initialized and that all
 * the required shared memory and semaphores have been created.
 * If things haven't been created, this routine will ensure that
 * they get created (or will return -1).
 */
int apv_init()
{
    int shmid;

    if ((shmid = shm_get(APV_CNTL_SHM_KEY)) == -1) {
	fprintf(stderr, "apv_init: Unable to set up shared memory space!\n");
	return -1;
    }
    return shmid;
}

/*
 * This routine should be called at the end of each operation
 * to clean things up that were set from the call to apv_init.
 */
int apv_fini()
{
    if (apv_list) {
	shm_detach((char *) apv_list);
    }
    if (apv_cntl) {
	shm_detach((char *) apv_cntl);
    }
    return 0;
}

/*
 * This routine is called to completely destroy all the shared
 * memory and semaphores.  It ignores error returns and just
 * tries to delete the list semaphores and shared memory,
 * then it deletes the control shared memory.
 * This should be called to clean up things to a known state
 * if problems are encountered.
 */
int apv_destroy()
{
    int shmid;

    /* Set apv_cntl and get shmid for control shared memory */
    if ((shmid = apv_init()) == -1) {
	fprintf(stderr, "apv_destroy: Unable to initialize\n");
	return -1;
    }
    shm_delete(apv_cntl->shmid);
    sem_delete(apv_cntl->semid);
    apv_fini();
    shm_delete(shmid);

    return 0;
}

/*
 * This routine is called to make a volume available to a
 * consumer (an archiver).  It must wait until there is a
 * slot available and then obtains the list protection
 * mutex.  It takes a "slot" off the free list and puts it
 * on the tail of the process list.
 */
int apv_produce(int volid) {
    int shmid;

    if ((shmid = apv_init()) == -1) {
	return -1;
    }

    if (debug) fprintf(stderr, "apv_produce: volid %d\n", volid);
    sem_wait(apv_cntl->semid, APV_SEM_AVAIL);
    sem_wait(apv_cntl->semid, APV_SEM_MUTEX);
    if (list_add_tail(volid) == -1)
    {
	fprintf(stderr, "apv_produce: error adding volid %d to the list!\n",
		volid);
	return -1;
    }
    sem_post(apv_cntl->semid, APV_SEM_MUTEX);
    sem_post(apv_cntl->semid, APV_SEM_FILLED);

    apv_fini();
    return 0;
}

/*
 * This routine is called by a consumer (archiver) to
 * obtain a volume to work on.  It must wait until there
 * is a filled volume to work on.  Then it gets the list
 * mutex, finds the first full volume on the list, and
 * returns the volid that needs to be processed.  After
 * it completes processing, the consumer should call
 * apv_consume_complete to remove the volume from the
 * process list, and add the slot back to the free list.
 */
int apv_consume_get() {
    int shmid;
    int volid = 0;
    apv_slot * slot;

    if ((shmid = apv_init()) == -1) {
	return -1;
    }

    if (debug) fprintf(stderr, "apv_consume_get: entered\n");
    sem_wait(apv_cntl->semid, APV_SEM_FILLED);
    sem_wait(apv_cntl->semid, APV_SEM_MUTEX);

    slot = list_find_filled();
    if (slot == NULL) {
	fprintf(stderr, "apv_consume_get: could not find a filled slot?\n");
	return -1;
    }

    slot->status = APVSLOT_STATUS_DRAINING;
    slot->retries = 0;
    volid = slot->volid;
    sem_post(apv_cntl->semid, APV_SEM_MUTEX);

    apv_fini();
    fprintf(stdout, "apv_consume_get: returning volid %d\n", volid);
    return volid;
}

/*
 * This routine is called by a consumer (archiver) when
 * it has completed archiving a volume.  The slot for
 * this volume can be removed from the process list and
 * added back to the free list.
 */
int apv_consume_complete(int volid) {
    int shmid;
    apv_slot * slot;

    if ((shmid = apv_init()) == -1) {
	return -1;
    }

    if (debug) fprintf(stderr, "apv_consume_complete: volid %d\n", volid);
    sem_wait(apv_cntl->semid, APV_SEM_MUTEX);

    slot = list_find_slot_by_volid(volid);
    if (slot == NULL) {
	fprintf(stderr, "apv_consume_complete: could not find slot for volid %d!\n", volid);
	return -1;
    }

    if (slot->status != APVSLOT_STATUS_DRAINING) {
	fprintf(stderr, "apv_consume_complete: slot for volid %d has status %s!\n", volid, status_to_string(slot->status));
	return -1;
    }

    if (list_free_slot(slot) != 0) {
	fprintf(stderr, "apv_consume_complete: error freeing slot!\n");
	return -1;
    }

    sem_post(apv_cntl->semid, APV_SEM_MUTEX);
    sem_post(apv_cntl->semid, APV_SEM_AVAIL);

    apv_fini();
    return 0;
}

/*
 * This routine is called by a consumer (archiver) when
 * it has failed for some reason to consume a volume.
 * If this consumer has tried APV_MAX_RETRIES to consume
 * the volume, it should give up.  In that case, it
 * changes the status of the slot and posts anyone
 * waiting for a filled volume.  If the consumer gets
 * back a zero, it should stop processing.  A one
 * indicates that it is okay to perform any necessary
 * recovery and try to process the volume again.
 */
int apv_consume_retry(int volid) {
    int shmid;
    apv_slot * slot;
    int giving_up = 0;

    if ((shmid = apv_init()) == -1) {
	return -1;
    }

    if (debug) fprintf(stderr, "apv_consume_retry: volid %d\n", volid);
    sem_wait(apv_cntl->semid, APV_SEM_MUTEX);
    slot = list_find_slot_by_volid(volid);
    if (slot == NULL) {
	fprintf(stderr, "apv_consume_retry: couldn't find volid %d?\n", volid);
	return -1;
    }

    if (++slot->retries >= APV_MAX_RETRIES) {
	slot->retries = 0;
	slot->status = APVSLOT_STATUS_FILLED;
	giving_up = 1;
    }
    sem_post(apv_cntl->semid, APV_SEM_MUTEX);
    if (giving_up) {
    	sem_post(apv_cntl->semid, APV_SEM_FILLED);
    }	

    apv_fini();
    if (debug) fprintf(stderr, "apv_consume_retry: returning for volid %d, %s\n",
		volid, giving_up ? "GIVING UP!" : "trying again...");
    return (giving_up);
}

int apv_status()
{
    int shmid;

    if ((shmid = apv_init()) == -1) {
	return -1;
    }

    sem_print(apv_cntl->semid);
    list_print_status();

    apv_fini();

    return 0;
}
/********************************************************************
 * List Processing routines
 *
 * These routines manage a pair of lists.
 * The free list is a SHST list of free slots.
 * The process list is a DHST list of slots being processed.
 *
 * NOTE that all list_* functions assume they are called with
 * the MUTEX semaphore held.
 *
 ********************************************************************/

/*
 * Moves a slot entry from the free list to the "process" list.
 * Returns 0 on success.
 * Returns -1 if an error is encountered.
 */
int list_add_tail(int volid)
{
    apv_slot *slot;

    if (apv_list->free == NULL) {
	fprintf(stderr, "list_add_tail: free list is empty??\n");
	return -1;
    }
    /* take one off the free list */
    slot = apv_list->free;
    apv_list->free = slot->next;

    /* set all the fields correctly */
    slot->volid = volid;
    slot->status = APVSLOT_STATUS_FILLED;
    slot->retries = 0;
    slot->next = NULL;

    /* if head is null, tail better be null! i.e. an empty list */
    if (apv_list->head == NULL) {
	apv_list->head = apv_list->tail = slot;
    }
    else {
	apv_list->tail->next = slot;
	apv_list->tail = slot;
    }
    return 0;
}

/*
 * Locates a slot of a "filled" volume.
 * Returns a pointer to a slot entry on success.
 * Returns NULL if no filled volumes are found,
 * or an unexpected error is encountered.
 */
apv_slot * list_find_filled()
{
    apv_slot *slot;
    if (apv_list->head == NULL) {
	fprintf(stderr, "list_find_filled: process list is empty??\n");
	return NULL;
    }
    /* traverse list looking for slot marked as filled */
    for (slot = apv_list->head; slot; slot = slot->next) {
	if (slot->status == APVSLOT_STATUS_FILLED) break;
    }
    if (slot == NULL) {
	fprintf(stderr, "list_find_filled: no slots are filled??\n");
    }
    return slot;
}

/*
 * Return a slot pointer given a volid.
 * Returns a pointer to a slot entry on success.
 * Returns NULL if the volid is not found,
 * or an unexpected error is encountered.
 */
apv_slot * list_find_slot_by_volid(int volid)
{
    apv_slot *slot;
    if (apv_list->head == NULL) {
	fprintf(stderr, "list_find_slot_by_volid: process list is empty??\n");
	return NULL;
    }
    /* traverse list looking for slot with the specified volid */
    for (slot = apv_list->head; slot; slot = slot->next) {
	if (slot->volid == volid) break;
    }
    if (slot == NULL) {
	fprintf(stderr, "list_find_slot_by_volid: no slots with volid %d\n",
		volid);
    }
    return slot;
}


/*
 * Move a given slot from the "process" list to the free list.
 * Returns 0 on success, or -1 if an unexpected error is encountered.
 */
int list_free_slot(apv_slot *slot)
{
    apv_slot * c_slot, *p_slot;

    if (apv_list->head == NULL) {
	fprintf(stderr, "list_free_slot: process list is empty??\n");
	return -1;
    }
    c_slot = p_slot = NULL;

    /*
     * first check the special case where the desired slot is the
     * only slot on the process list, otherwise traverse list looking for
     * the specified slot (actually we need the slot pointing to
     * the specified slot)
     */
    if (apv_list->head == slot && apv_list->tail == slot ) {
	apv_list->head = NULL;
	apv_list->tail = NULL;
	c_slot = slot;
    }
    else {
	for (c_slot = apv_list->head; c_slot;
				p_slot = c_slot, c_slot = c_slot->next) {
	    if (c_slot == slot) {
		if (p_slot == NULL) {
		    /* It is at the head of the list */
		    apv_list->head = c_slot->next;
		}
		else {
		    p_slot->next = c_slot->next;
		}
		/* If we're removing from the tail, fix the tail pointer */
		if (apv_list->tail == c_slot)
		    apv_list->tail = p_slot;	    
		break;
	    }
	}
    }
    if (c_slot == NULL) {
	fprintf(stderr, "list_free_slot: node 0x%08x not found!\n",
		(unsigned int) slot);
	return -1;
    }

    /*
     * We've unlinked it from the process list,
     * now we need to put it on the free list.
     * c_slot should be pointing at the one we're putting on
     */
    c_slot->volid = 0;
    c_slot->status = APVSLOT_STATUS_AVAILABLE;
    c_slot->retries = 0;
    c_slot->next = apv_list->free;
    apv_list->free = c_slot;

    return 0;
}

char * status_to_string(int status)
{
    if (status == APVSLOT_STATUS_AVAILABLE)
	return "AVAILABLE";
    else if (status == APVSLOT_STATUS_FILLED)
	return "FILLED";
    else if (status == APVSLOT_STATUS_DRAINING)
	return "DRAINING";
    else
	return "*INVALID*";
}

int slot_print_status(apv_slot * slot, int index)
{
    fprintf(stdout, "(%03d) 0x%08x volid %010d retries %02d "
		    "status %-10s next 0x%08x\n",
		index, (unsigned int) slot, slot->volid, slot->retries,
		status_to_string(slot->status), (unsigned int) slot->next);
    return 0;
}

int list_print_status()
{
    apv_slot * slot;
    int i;
    fprintf(stdout, "The list pointers head 0x%08x, tail 0x%08x, free 0x%08x\n",
		(unsigned int) apv_list->head,
		(unsigned int) apv_list->tail,
		(unsigned int) apv_list->free);

    fprintf(stdout, "The process list:\n");
    for (i = 1, slot = apv_list->head; slot; slot = slot->next, i++) {
	slot_print_status(slot, i);
    }

    fprintf(stdout, "The free list:\n");
    for (i = 1, slot = apv_list->free; slot; slot = slot->next, i++) {
	slot_print_status(slot, i);
    }

    return 0;
}

/********************************************************************
 * General Utility routines
 ********************************************************************/

void usage(char *progname)
{
	fprintf(stderr, "Usage: \n");
	fprintf(stderr, "%s\n[\n"
			"\tinit | \n"
		        "\tdestroy | \n"
		        "\tstatus | \n"
		        "\tproduce <volid> | \n"
		        "\tconsume_get | \n"
		        "\tconsume_complete <volid> | \n"
		        "\tconsume_retry <volid> \n"
			"]\n", progname);
	exit(1);
}

/********************************************************************
 * Main
 ********************************************************************/

/*
 * Okay, this is shoddy argument parsing, but it works...
 */

int main(int argc, char *argv[])
{
    if (argc < 2)
	usage(argv[0]);

    if (!strcmp(argv[1], "init")) {
	return apv_init();
    }
    else if (!strcmp(argv[1], "destroy")) {
	return apv_destroy();
    }
    else if (!strcmp(argv[1], "status")) {
	return apv_status();
    }
    else if (!strcmp(argv[1], "consume_get")) {
	return apv_consume_get();
    }

    if (argc < 3)
	usage(argv[0]);
	
    if (!strcmp(argv[1], "produce")) {
	return apv_produce(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "consume_complete")) {
	return apv_consume_complete(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "consume_retry")) {
	return apv_consume_retry(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "semget")) {
	return sem_get(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "semdelete")) {
	return sem_delete(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "semwait")) {
	return sem_wait(atoi(argv[2]), APV_SEM_MUTEX);
    }
    else if (!strcmp(argv[1], "sempost")) {
	return sem_post(atoi(argv[2]), APV_SEM_MUTEX);
    }
    else if (!strcmp(argv[1], "shmget")) {
	return shm_get(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "shmdelete")) {
	return shm_delete(atoi(argv[2]));
    }
    else if (!strcmp(argv[1], "shmattach")) {
	return (int) shm_attach(atoi(argv[2]), (char *)REQ_APV_LIST_ADDR);
    }
    else if (!strcmp(argv[1], "shmdetach")) {
	return shm_detach((char *)atoi(argv[2]));
    }
    else {
	usage(argv[0]);
    }
    return 1;
}
