#!/usr/bin/perl

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

package Archiver;

use strict;
use English;
use Cwd;
use Getopt::Long;
use File::Basename;

#------------------------------------------------------------------------
# FIRST THING -- Set up the correct working directory!
#------------------------------------------------------------------------
my $topdir = `dirname $0`;
chomp($topdir);
chdir($topdir);
$topdir = cwd();

#
# This has to be "require", otherwise compilation will
# if not invoked from the pilot directory
#
require Changer;

#------------------------------------------------------------------------
# Global variable declarations
#------------------------------------------------------------------------

#-----------------------------------------------
# These variables may be customized as required
#-----------------------------------------------

my $DEFAULT_MYSQL_TABLE = "firstgen";	# This is the default name of the
					# mySQL DB table (if --db_mysql)

my $DEFAULT_FILEDB_NAME = "/scratch0/fileDB/firstgen";
					# This is the default name of the
					# flat file DB (if --db_file)

my $DEFAULT_CHANGER_DEVICE = "/dev/ch0";
					# Default name of the changer device

my $DEFAULT_TAPE_DRIVE_DEVICE = "/dev/nrst";
					# Default name of the tape drive device
					# (w/o the device number)

my $DEFAULT_TAPE_DRIVE_NUMBER = 0;
					# Default drive number within the
					# changer

#------------------------------------
# These shouldn't need to be changed
#------------------------------------

my $debug = 0;
my $db_file = "";
my $table_name = "";
my $db_mysql = 0;
my $changer_device = "";
my $tape_device_name = "";
my $tape_device_number = 0;
my $full_tape_device = "";
my %opts;

#----------------------------------------------------------------
# The following are used only if we are using the mysql database
#----------------------------------------------------------------
my $dbh;

# SQL statements and their handles when "prepared"
my ($sql_update_status_to_archiving, $usta_sth);
my ($sql_update_status_to_complete, $ustc_sth);
my ($sql_select_by_status_full, $sbsf_sth);

#-----------
# Constants
#-----------
my $LOCK_SHARED = 1;
my $LOCK_EXCLUSIVE = 2;
my $LOCK_NOBLOCK = 4;
my $LOCK_UNLOCK = 8;

#########################################################################
#############  M A I N   E X E C U T I O N   P O I N T  #################
#########################################################################

#
# Parse input options.
#

GetOptions( \%opts,	"help",
			"debug:i",
			"db_file:s",
#			"db_mysql:s",
			"changer=s",
			"drive=s",
			"drivenum=i", 
	  );

#
# If --help was specified, just spit back the options available ...
#

if ( exists $opts{help} ) {
	print "\nUsage: $0 takes the following options:
	--help			- produce this help text
	--debug[=<level>]	- produce debugging output
				  (debugging level optional)
	--db_file[=<dbfile>]	- use file DB, optionally specify file to use
	--changer=<chngr_dev>	- the tape changer device name
	--drive=<drive_dev>	- the tape drive device name
				  (w/o the drive number - i.e. '/dev/nrst')
	--drivenum=<dev_num>	- the tape drive number ( i.e. '0', '1', etc)
\n";
	exit 1;
}

# mysql currently not fully supported...
#	--db_mysql[=<table>]	- use mySQL DB, optionally specify the
#				  table name to use


#
# If --debug was specified, but $debug is zero, then set $debug to 1
#
if ( exists $opts{debug} ) {
	if ( $opts{debug} == 0 ) { $debug = 1; }
	else { $debug = $opts{debug} }
}
else { $debug = 0 }

#
# If both db_file and db_mysql are specified, report an error.
# If neither is specified, default to db_file
#

if ( exists $opts{db_file} && exists $opts{db_mysql} ) {
	die "Specify only one of db_file or db_mysql\n";
}
elsif ( !exists $opts{db_file} && !exists $opts{db_mysql} ) {
	$opts{db_file} = "";
}

if ( exists $opts{db_file} ) {
	if ( $opts{db_file} eq '' ) { $db_file = $DEFAULT_FILEDB_NAME }
	else { $db_file = $opts{db_file} }
}
else { $db_file = 0 }

if ( exists $opts{db_mysql} ) {
	$db_mysql = 1;
	if ( $opts{db_mysql} eq '' ) { $table_name = $DEFAULT_MYSQL_TABLE }
	else { $table_name = $opts{db_mysql} }
}
else { $db_mysql = 0 }

#
# Allow command-line options to specify the changer device and
# the tape drive to be used for this archiver.
# If not specified, use the default values.
#

if ( exists $opts{changer} ) { $changer_device = $opts{changer} }
else { $changer_device = $DEFAULT_CHANGER_DEVICE }

if ( exists $opts{drive} ) { $tape_device_name = $opts{drive} }
else { $tape_device_name = $DEFAULT_TAPE_DRIVE_DEVICE }

if ( exists $opts{drivenum} ) { $tape_device_number = $opts{drivenum} }
else { $tape_device_number = $DEFAULT_TAPE_DRIVE_NUMBER }

$full_tape_device = $tape_device_name . $tape_device_number;

#
# Tell the Changer code to debug if we're debugging
#

if ($debug) {
	$Changer::debug = $debug;
}

my $changer_ref;

#
# Instantiate a Changer and get a reference to it
#
$changer_ref = Changer->new($changer_device, $tape_device_name);

#
# Initialize the use of a database
#
database_init($changer_ref);


#
# Get the status of the contents of the changer
#

print "\n";
$changer_ref->print_content_status($dbh);
print "\n";


#########################################################################
#############           S U B R O U T I N E S           #################
#########################################################################

#------------------------------------------------------------------------
# Connect to the database, die if it fails
#------------------------------------------------------------------------
sub database_connect {
	my $db_spec = "DBI:mysql:vault";
	my $db_user = 'root';
	my $db_password = 'mysql';


	$dbh = DBI->connect($db_spec, $db_user, $db_password,
			{ RaiseError => 1, AutoCommit => 1 })
		or die "database_connect: Could not open database $db_spec: $DBI::errstr; stopped";
}

#------------------------------------------------------------------------
# Prepare all the SQL statements and get handles to them
#------------------------------------------------------------------------
sub sql_prepare {

	#
	# Update a record to ARCHIVING status, fill in the tapeid
	#
	$sql_update_status_to_archiving = "UPDATE $table_name SET tapeid=?,status='ARCHIVING' WHERE volid=?";

	#
	# Update a record to COMPLETE status
	#
	$sql_update_status_to_complete = "UPDATE $table_name SET status='COMPLETE' WHERE volid=?";

	#
	# This will select all fields of all records waiting to be filled.
	#
	$sql_select_by_status_full= "SELECT volid FROM $table_name WHERE status='FULL'";


	#
	# Prepare all the statements
	#

	$usta_sth = $dbh->prepare_cached($sql_update_status_to_archiving);
	$ustc_sth = $dbh->prepare_cached($sql_update_status_to_complete);
	$sbsf_sth = $dbh->prepare_cached($sql_select_by_status_full);
}

#------------------------------------------------------------------------
# Initialize use of the database (whichever database is being used)
#------------------------------------------------------------------------
sub database_init {
	if ($db_mysql) {
		require DBI;
		my $rv;

		# Tell the changer we're using mySQL
		$changer_ref->db_mysql($db_mysql);

		# Initialize use of the database
		database_connect();
		sql_prepare();

		# Do changer-specific intialization
		$changer_ref->db_init($dbh, $table_name);
	}
	elsif ($db_file) {
		my $rv;

		# Tell the changer we're using a flat file DB
		$changer_ref->db_file($db_file);
	}
	else {
		die "Some form of database must be selected!\n";
	}
}
