#!/usr/bin/perl

# 
#  COPYRIGHT    2000
#  THE REGENTS OF THE UNIVERSITY OF MICHIGAN
#  ALL RIGHTS RESERVED
#  
#  Permission is granted to use, copy, create derivative works
#  and redistribute this software and such derivative works
#  for any purpose, so long as the name of The University of
#  Michigan is not used in any advertising or publicity
#  pertaining to the use of distribution of this software
#  without specific, written prior authorization.  If the
#  above copyright notice or any other identification of the
#  University of Michigan is included in any copy of any
#  portion of this software, then the disclaimer below must
#  also be included.
#  
#  THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
#  FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
#  PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
#  MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
#  WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
#  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
#  REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
#  FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
#  CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
#  OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
#  IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
#  SUCH DAMAGES.
#
 
#########################################################################
#                                                                       #
#  Retrieve contents from a tape.                                       #
#                                                                       #
#  Assumes that the desired tape is already mounted in the drive.       # 
#                                                                       #
#########################################################################

package Retrieve;

use strict;
use English;
use Cwd;
use Getopt::Long;
use Time::Local;
use File::Basename;
use locale;


#------------------------------------------------------------------------
# FIRST THING -- Set up the correct working directory
# to reference other programs.
#------------------------------------------------------------------------
my $topdir = `dirname $0`;
chomp($topdir);
chdir($topdir);
$topdir = cwd();

require Changer;

#------------------------------------------------------------------------
# Global variable declarations
#------------------------------------------------------------------------

#-----------------------------------------------
# These variables may be customized as required
#-----------------------------------------------

my $decrypt_dir = "../decrypt";		# Location of the decrypt program
					# directory (where the decrypt
					# program may be found)

my $DEFAULT_FILEDB_NAME = "/scratch0/fileDB/firstgen";
					# This is the default name of the
					# flat file DB (if --db_file)

my $DEFAULT_CHANGER_DEVICE = "/dev/ch0";
					# Default name of the changer device

my $DEFAULT_TAPE_DRIVE_DEVICE = "/dev/st";
					# Default name of the tape drive device

my $DEFAULT_TAPE_DRIVE_NUMBER = 0;	# Default drive number within the
					# changer

my $DEFAULT_BASE_DIRECTORY = "/scratch0/retrieved";
					# Default test directory

my $tar_pgm = "tar";			# Location of a 'working' tar program
					# (The default tar shipped with
					# OpenBSD does not properly return
					# an indication of failures.)

my $mt_pgm = "/bin/mt";			# Location of the mt (magnetic
					# tape) program

my $asymm_pgm = "/usr/local/bin/gpg";	# Location of the GNU Privacy Guard
					# asymmetric encryption program

my $sysname = `uname -s`;
chomp($sysname);

if ($sysname eq "FreeBSD") {
	$mt_pgm = "/usr/bin/mt";	# Location of the mt command
	$tar_pgm = "/usr/bin/tar";	# Location of a 'working' tar program
	$DEFAULT_TAPE_DRIVE_DEVICE = "/dev/nsa";
}
elsif ($sysname eq "OpenBSD") {
	$mt_pgm = "/bin/mt";		# Location of the mt command
	$tar_pgm = "/usr/local/bin/tar";# Location of a 'working' tar program
					# (The default tar shipped with
					# OpenBSD does not properly return
					# an indication of failures.)
	$DEFAULT_TAPE_DRIVE_DEVICE = "/dev/nrst";
}
elsif ($sysname eq "Linux") {
	$mt_pgm = "/bin/mt";		# Location of the mt command
	$tar_pgm = "/bin/tar";		# Location of a 'working' tar program
	$DEFAULT_TAPE_DRIVE_DEVICE = "/dev/nst";
	$DEFAULT_CHANGER_DEVICE = "/dev/sch0";
	$asymm_pgm = "/usr/bin/gpg";	# Location of GNU Privacy Guard
}
else {
	die "Need to figure out sysname...\n";
}

#------------------------------------
# These shouldn't need to be changed
#------------------------------------

my $decrypt_pgm = $topdir . "/" . $decrypt_dir . "/decrypt";
my $debug = 0;
my $db_file = "";
my $changer_device = "";
my $tape_device_name = "";
my $tape_device_number = 0;
my $full_tape_device = "";
my %opts;
my @segfilelist;
my $segfile;
my $cmdline_tapeid = "";	# tapeid was specified on command line
my $current_tapeid = "";	# tapeid we're currently processing
my $first_volume;
my $last_volume;
my $base_directory;
my $pass_phrase;		# Holds asymmetric key pass phrase
my $tempdirname = "=TEMP=";	# Temporary directory name
my $retcode;
my $time_to_quit = 0;
my $changer_ref;


#########################################################################
#############  M A I N   E X E C U T I O N   P O I N T  #################
#########################################################################

# 
# Make sure we are running as root
# 
verify_we_are_root();

#
# Parse input options.
#
process_arguments();

#
# Check the tar command to see if it properly reports errors
#
die "Fix tar before trying to run this program\n"
	if ( tar_command_is_broken() ) ;

#
# Tell the Changer code to debug if we're debugging
#
if ($debug) {
	$Changer::debug = $debug;
}

#
# Instantiate a Changer and get a reference to it
#
$changer_ref = Changer->new($changer_device, $tape_device_name);

#
# Initialize the use of a database
#
database_init();

#
# Create a directory to work in and go there
#
if (! -d $base_directory) {
	print "'$base_directory' doesn't exist.  Creating it...\n";
	if ( mkdir($base_directory, 0755) != 1 ) {
		die "cannot mkdir $base_directory: $!";
	}
}
chdir $base_directory || die "cannot chdir $base_directory";

#
# Prompt for the passphrase for (possibly many) decryptions
#
die "Could not determine pass phrase\n"
	if ( get_pass_phrase() );

#
# Set up an interrupt handler so we can clean up when we are done.
#
$SIG{INT} = \&SIGINT_Handler;

#
# If a tapeid was specified on the command line, process it.
# Otherwise, we loop prompting for tapeid, firstvol and lastvol
#
if ( $current_tapeid ne "" ) {
	$retcode = process_tape_retrieval( $current_tapeid,
			$first_volume, $last_volume);
	print scalar(localtime), ": Unloading tape drive...\n";
	$changer_ref->unload_drive($tape_device_number);
}
else {
	while ( !$time_to_quit ) {
		$retcode = 1;
		last if ( prompt_for_tapeid( \$current_tapeid, \$first_volume,
						\$last_volume ) );
		last if ( $retcode = process_tape_retrieval( $current_tapeid,
			$first_volume, $last_volume) );
	}
}

exit $retcode;



#########################################################################
#############           S U B R O U T I N E S           #################
#########################################################################

#------------------------------------------------------------------------
# Verify we are running as root
#------------------------------------------------------------------------
sub verify_we_are_root {
	if ($UID != 0  || $EUID != 0) {
		print "Sorry, this program must be run as root.\n";
		print "(UID is $UID EUID is $EUID)\n";
		exit 1;
	}
}

#------------------------------------------------------------------------
# Process command-line arguments
#------------------------------------------------------------------------
sub process_arguments {
	GetOptions( \%opts,	"help",
				"debug:i",
				"db_file=s",
				"changer=s",
				"drive=s",
				"drivenum=i",
				"basedir=s",
				"firstvol=i",
				"lastvol=i",
				"tapeid=s",
		  );

	#
	# If --help was specified, just spit back the options available ...
	#

	if ( exists $opts{help} ) {
		print "\nUsage: ", $0, " takes the following options:
	--help			- produce this help text
	--debug[=<level>]	- produce debugging output
				  (debugging level optional)
	--db_file=<dbfile>	- specify the file to use for the database
	--changer=<chngr_dev>	- the tape changer device name
	--drive=<drive_dev>	- the tape drive device name
				  (w/o the drive number - i.e. '/dev/nrst')
	--drivenum=<dev_num>	- the tape drive number ( i.e. '0', '1', etc)
	--basedir=<path>	- the base path to which volumes
				  are to be retrieved
	--firstvol=<###>	- the sequence number of the first volume
				  to retrieve from the tape (default is 1)
	--lastvol=<###>		- the sequence number of the last volume
				  to retrieve from the tape (default is 100)
	--tapeid=<tapeid>	- the tapeid of the tape containing the
				  volumes to be retrieved

	Note: If tapeid is not specified, you'll be prompted for tapeids
	      until told to quit.  In this case, values for firstvol and
	      lastvol will also be requested.

		\n";
		exit 1;
	}

	#
	# If --debug was specified, but $debug is zero, then set $debug to 1
	#
	if ( exists $opts{debug} ) {
		if ( $opts{debug} == 0 ) { $debug = 1; }
		else { $debug = $opts{debug} }
	}
	else { $debug = 0 }

	if ( exists $opts{db_file} ) {
		if ( $opts{db_file} eq '' ) { $db_file = $DEFAULT_FILEDB_NAME }
		else { $db_file = $opts{db_file} }
	}

	#
	# Allow command-line options to specify the changer device and
	# the tape drive to be used.  If not specified, use the default
	# values.
	#

	if ( exists $opts{changer} ) { $changer_device = $opts{changer} }
	else { $changer_device = $DEFAULT_CHANGER_DEVICE }

	if ( exists $opts{drive} ) { $tape_device_name = $opts{drive} }
	else { $tape_device_name = $DEFAULT_TAPE_DRIVE_DEVICE }

	if ( exists $opts{drivenum} ) { $tape_device_number = $opts{drivenum} }
	else { $tape_device_number = $DEFAULT_TAPE_DRIVE_NUMBER }

	$full_tape_device = $tape_device_name . $tape_device_number;


	#
	# Allow specifiecation of base directory, volume sequence, and
	# tapeid
	#

	if ( exists $opts{basedir} ) { $base_directory = $opts{basedir} }
	else { $base_directory = $DEFAULT_BASE_DIRECTORY }

	if ( exists $opts{firstvol} ) { $first_volume = $opts{firstvol} }
	else { $first_volume = 1 }

	if ( exists $opts{lastvol} ) { $last_volume = $opts{lastvol} }
	else { $last_volume = 100 }

	if ( $first_volume > $last_volume ) {
		die "Invalid volume number specification(s)\n";
	}

	if ( exists $opts{tapeid} ) { $cmdline_tapeid = uc($opts{tapeid}) }

	if ( $cmdline_tapeid ne "" ) { $current_tapeid = $cmdline_tapeid }

	#
	# Print out the results of command-line option processing
	#

	print scalar(localtime),
		": Starting retrieve with the following options:\n
	debug		'$debug'
	db_file		'$db_file'
	changer		'$changer_device'
	drive		'$tape_device_name'
	drivenum	'$tape_device_number'
	basedir		'$base_directory'
	firstvol	'$first_volume'
	lastvol		'$last_volume'
	tapeid		'$cmdline_tapeid'
	\n";
}


#------------------------------------------------------------------------
# Interrupt handler routine.  Allows us to clean up and unload the
# tape drive if we're looping for tapeid's and they hit Ctrl-C.
#------------------------------------------------------------------------
sub SIGINT_Handler {
	print scalar(localtime), ": Caught Ctrl-C (SIGINT)\n";
	print scalar(localtime), ": Unloading tape drive...\n";
	$changer_ref->unload_drive($tape_device_number);
	print scalar(localtime), ": Exiting ...\n";
	exit 1;
}

#------------------------------------------------------------------------
# Process a tape.  Extracting and decrypting the specified volumes.
#------------------------------------------------------------------------
sub process_tape_retrieval {
	my ($tapeid, $first_vol, $last_vol, @args) = @_;

	my $rc;

	#
	# Get the tape mounted and online
	#

	print scalar(localtime), ": Loading tape $tapeid into drive ...\n";
	$rc = $changer_ref->mount_specific($tape_device_number, $tapeid);
	if ($rc == -1) {
		$time_to_quit = 1;
		return 1;
	}
	
	#
	# Rewind the tape
	#
	print scalar(localtime), ": Rewinding tape $tapeid ... \n";
	die "Cannot do initial rewind of tape" if tape_rewind();

	if ($first_vol > 1) {
		print scalar(localtime),
				": Forward spacing to file $first_vol ...\n";
		die "Failed to forward space"
				if tape_forward_space_files($first_vol-1);
	}

	my $vol_count = $last_vol - $first_vol + 1;
	my $current_vol = $first_vol;

	while ( $vol_count > 0 ) {

		my $true_vol_number = -1;
		my $new_vol_dir;

		my $vol_dir = $base_directory . '/' . $tempdirname;
		system("/bin/rm", "-rf", $vol_dir);
		if ( mkdir($vol_dir, 0755) != 1 ) {
			die "cannot mkdir $vol_dir: $!";
		}
		chdir $vol_dir || die "cannot chdir $vol_dir: $!";

		if ( process_volume_retrieval(\$true_vol_number) ) {
			$vol_count = 0;
			last;
		}
	
		#
		# Build desired name for the restored volume.  If we
		# didn't get the true volume number, or the directory
		# already exists, then we'll build a different name
		# with a current timestamp in it.
		#
		$new_vol_dir = $base_directory . '/' . $true_vol_number;
		if ( $true_vol_number == -1 || -d $new_vol_dir ) {
			my $date = timelocal( localtime() );
			my $prefix;

			if ( $true_vol_number != -1 ) {
				$prefix = '/' . $true_vol_number . '-';
			}
			else {
				$prefix = '/XXX-';
			}
			$new_vol_dir = $base_directory . $prefix . $date;
		}
		print scalar(localtime),
			": Renaming directory to $new_vol_dir\n";
		if ( !rename($vol_dir, $new_vol_dir) ) {
			print scalar(localtime),
			     ": Failed to rename $vol_dir to $new_vol_dir!\n";
		}

		tape_forward_space_files(1);
		$vol_count--;
		$current_vol++;
	}

	if ( $vol_count == 0 ) { return 0; }
	else { return 1; }
}


#------------------------------------------------------------------------
# Process a volume.  Extracting and decrypting all the segments.
# (Assumes we're already in the correct directory.)
# Returns (via reference variable) the true volume number
# by examining the contents of the "volnum" file.
#------------------------------------------------------------------------
sub process_volume_retrieval {
	my ($volnumref, @args) = @_;
	my $volnum;

	print scalar(localtime), ": Untarring volume from tape...\n";

	# read tar
	die "Failed to read tar file" if tape_read_tar();
	print scalar(localtime), ": Untar complete\n";

	my $counter = 1;

	if ( opendir(VOLDIR, "." ) ) {
		@segfilelist = grep { /^:/ } readdir(VOLDIR);
		closedir(VOLDIR);
	}
	else {
		print "process_volume_retrieval: Failed to open directory?\n";
		return 1;
	}

	my $num_segments = scalar(@segfilelist);
	if ($num_segments == 0) {
		print scalar(localtime),
			": Empty directory after tar -- End of Tape??\n";
		return 1;
	}

	# decrypt the symmetric keys
	print scalar(localtime), ": Decrypting symmetric keys.\n";
	if (decrypt_symmetric_key("volKey") != 0 ||
	    decrypt_symmetric_key("transKey") != 0) {
		print scalar(localtime),
			"Failed to decrypt volume's symmetric keys!";
		return 1;
	}

	# decrypt each segment
	foreach $segfile (@segfilelist) {
		print scalar(localtime),
			": Decrypting segment ", $counter++,
			" of $num_segments ($segfile) ... \n";

		decrypt_segment($segfile);

		print scalar(localtime),
			": Decrypt completed ($segfile)\n";
	}

	# get the volume number from the "volnum" file
	if ( open(VOLNUM, "< volnum") ) {
		$volnum = (<VOLNUM>);
		$$volnumref = $volnum;
		close(VOLNUM);
	}
	else {
		print scalar(localtime),
			": Could not open volnum file: $!\n";
		$$volnumref = -1;
		$volnum = "???";
	}

	# delete the cleartext keys and "extra" files
	unlink("volKey") or print "Error unlinking volKey: $!\n";
	unlink("transKey") or print "Error unlinking transKey: $!\n";
	unlink("volKey.gpg") or print "Error unlinking volKey.gpg: $!\n";
	unlink("transKey.gpg") or print "Error unlinking transKey.gpg: $!\n";
	unlink("volnum") or print "Error unlinking volnum: $!\n";

	print scalar(localtime), ": Retrieval of volume $volnum complete!\n";
	return 0;
}

#------------------------------------------------------------------------
# Try a tar command that should fail.
# See if we properly get an error indication from it.  
#------------------------------------------------------------------------
sub tar_command_is_broken {
	my $rc;

	print "Testing the tar cmd for error indication ... \n";
	$rc = 0xffff & system("$tar_pgm -xf /dev/nst9 . >/dev/null 2>/dev/null");
	print "
	*******************************************************
	***            The tar command is broken!           ***
	***                                                 ***
	***    It failed to return an error indication.     ***
	*******************************************************
	\n" if ($rc == 0);
	print "tar_command_is_broken: return code was $rc\n" if ($debug) ;
	print "... tar correctly returned an error status\n\n" if ($rc != 0);
	return ! $rc;
}

#------------------------------------------------------------------------
# If we don't already have a pass phrase, prompt the user for it
#------------------------------------------------------------------------
sub get_pass_phrase {
	if ($pass_phrase eq "") {
		my $again;

		local $SIG{INT} = sub {
			print "\n";
			system("stty echo");
			exit 1;
		};

		system("stty -echo");
		while (1) {
			print "enter master key passphrase: ";
			$pass_phrase = <>;
			print "\nenter master key passphrase again: ";
			$again = <>;
			chop $pass_phrase;
			chop $again;
			if ($pass_phrase eq $again) {
				last;
			}
			print "\npassphrases don't match, try again\n"
		}
		print "\n";
		system("stty echo");
	}
	return 0;
}

#------------------------------------------------------------------------
# Prompt for a tapeid and volume range
# Gets references to places to return the tapeid,
# first_volume, and last_volume
#------------------------------------------------------------------------
sub prompt_for_tapeid {
	my ($tref, $fref, $lref, @args) = @_;

	my ($t, $f, $l);	# local holders for tapeid, first and last vol

	my $good_input;

	do {
		#
		# Install signal handler to allow them to use ctrl-c to
		# get out of this at a prompt
		#

		print "\n";
		print "Enter the tapeid to be retrieved: ";
		$t = <>;
		chop $t;
		$t = uc($t);

		print "Enter the beginning volume sequence number [$$fref]: ";
		$f = <>;
		chop $f;
		if ($f eq "") { $f = $$fref }

		print "Enter the ending volume sequence number [$$lref]: ";
		$l = <>;
		chop $l;
		if ($l eq "") { $l = $$lref }

		# Assume good input, then check it ...
		$good_input = 1;

		if ( $t eq "" ) {
			print "Invalid tapeid: '$t'\n";
			$good_input = 0;
		}

		if ( $l < $f ) {
			print "Invalid volume range '$f' ==> '$l'\n";
			$good_input = 0;
		}
	} until ($good_input);

	# Fill in the referenced variables with what we got from user
	$$tref = $t;
	$$fref = $f;
	$$lref = $l;

	return 0;
}

#------------------------------------------------------------------------
# Check the results of a system command
#------------------------------------------------------------------------
sub check_system_rc {
	my ($description, $rc, $dbg, @args) = @_;

	if ( $rc == 0) {
		print scalar(localtime), ": ", $description,
			" completed successfully!\n" if ($dbg);
		return 0;
	}
	elsif ( $rc == 0xff00 )  {
		print scalar(localtime), ": ", $description,
			" failed: !$\n";
		return 1;
	}
	elsif ( $rc > 0x80 ) {
		$rc >>= 8;
		print scalar(localtime), ": ", $description,
			" returned with status $rc\n";
		return 1;
	}
	else {
		my $coredump = 0;
		if ( $rc & 0x80 ) {
			$rc &= ~0x80;
			$coredump = 1; 
		}
		printf scalar(localtime), ": ", $description,
			" ended from signal %d (%s a core dump)\n",
			$rc, $coredump ? "with" : "without";
		return 1;
	}
}

#------------------------------------------------------------------------
# Initialize use of the database (whichever database is being used)
#------------------------------------------------------------------------
sub database_init {
	my $rv;
	    
	# Tell the changer we're using a flat file DB
	$changer_ref->db_file($db_file);
}

#------------------------------------------------------------------------
# Rewind the tape
#------------------------------------------------------------------------
sub tape_rewind {
	my $rc;

	$rc = 0xffff & system($mt_pgm, "-f", $full_tape_device, "rewind");
	return check_system_rc("mt rewind", $rc, 0);
}

#------------------------------------------------------------------------
# Skip forward "count" files
#------------------------------------------------------------------------
sub tape_forward_space_files {
	my ($count, @args) = @_;
	my $rc;

	$rc = 0xffff & system($mt_pgm, "-f", $full_tape_device, "fsf", $count);
	return check_system_rc("mt fsf", $rc, 0);
}

#------------------------------------------------------------------------
# Skip backward "count" files
#------------------------------------------------------------------------
sub tape_backward_space_files {
	my ($count, @args) = @_;
	my $rc;

	$rc = 0xffff & system($mt_pgm, "-f", $full_tape_device, "bsf", $count);
	return check_system_rc("mt bsf", $rc, 0);
}

#------------------------------------------------------------------------
# Read a tar file from the tape...
#------------------------------------------------------------------------
sub tape_read_tar {
	my $rc;

	# Before untarring anything, make sure directory is clean
	unlink <*>;

	$rc = 0xffff & system($tar_pgm, "-xf", $full_tape_device);
	return check_system_rc("tar extract", $rc, 0);
}

#------------------------------------------------------------------------
# Decrypt a volume's symmetric key
#------------------------------------------------------------------------
sub decrypt_symmetric_key {
	my ($keyfile, @args) = @_;

	my $rc;
	my $devname;

	print "decrypt_symmetric_key: decrypting $keyfile.gpg\n" if ($debug);

	# if the keyfile was already decrypted by a previous run, we're done
	if ( -f $keyfile ) {
		print "decrypt_symmetric_key: $keyfile already decrypted\n";
		return 0;
	}

	print "The decrypt command is 'echo ... | $asymm_pgm --quiet ",
	      "--no-tty --passphrase-fd 0 $keyfile.gpg'\n" if ($debug);

	$rc = 0xffff & system("echo '$pass_phrase' | $asymm_pgm --quiet --no-tty --passphrase-fd 0 $keyfile.gpg");

	if ( check_system_rc($asymm_pgm, $rc, 1) ) {
		return 1;
	}

	# make sure a decrypted keyfile was created
	if ( ! -f $keyfile ) {
		print "decrypt_symmetric_key: $asymm_pgm did not ",
		      "create decrypted keyfile $keyfile";
		return 1;
	}
	return 0;
}

#------------------------------------------------------------------------
# Run the decrypt program on a single segment ($segfile) and
# put the decrypted output into $decrypted_file
#------------------------------------------------------------------------
sub decrypt_segment {
	my ($segfile, @args) = @_;
	my $rc;

	my $decrypted_file = "DECRYPTED";
	my $sfile = "s" . $segfile;
	my $tfile = "t" . $segfile;
	my $Tfile = "transKey";
	my $Vfile = "volKey";
	my @decrypt_args = ($decrypt_pgm, "-o", $decrypted_file,
			    "-c", $segfile, "-s", $sfile, "-t", $tfile,
			    "-T", $Tfile, "-V", $Vfile);

	$rc = system(@decrypt_args);

	# removed the encrypted data and rename the decrypted data file
	unlink $segfile;
	unlink $sfile;
	unlink $tfile;
	link $decrypted_file, $segfile;
	unlink $decrypted_file;

	return check_system_rc("decrypt", $rc, 0);
}
