#!/usr/bin/perl

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

package Reconcile;

use strict;
use English;
use Changer;
use Cwd;
use Getopt::Long;
use File::Basename;
use Errno qw(:POSIX);


#------------------------------------------------------------------------
# Global variable declarations
#------------------------------------------------------------------------

#-----------------------------------------------
# These variables may be customized as required
#-----------------------------------------------

my $util_dir = "../util";		# Location of the utility program
					# directory (where the apvsync 
					# program may be found)

my $DEFAULT_VOLUME_PATH = "/scratch0/volumes";
					# Default path to locate volumes
					# to be written to tape

my $DEFAULT_MYSQL_TABLE = "firstgen";	# This is the default name of the
					# mySQL DB table (if --db_mysql)

my $DEFAULT_FILEDB_NAME = "/scratch0/fileDB/firstgen";
					# This is the default name of the
					# flat file DB (if --db_file)

#------------------------------------
# These shouldn't need to be changed
#------------------------------------

my $apvsync = cwd() . "/" . $util_dir . "/apvsync";
my $debug = 0;
my $db_file = "";
my $table_name = "";
my $db_mysql = 0;
my $volume_path_base = "";
my %opts;

#-----------
# Constants
#-----------
my $LOCK_SHARED = 1;
my $LOCK_EXCLUSIVE = 2;
my $LOCK_NOBLOCK = 4;
my $LOCK_UNLOCK = 8;

#########################################################################
#############  M A I N   E X E C U T I O N   P O I N T  #################
#########################################################################

#------------------------------------------------------------------------
# Attempt to reconcile the state of the shared memory with the state
# of the disk.  This is intended to be run during system start-up
# to reconcile the shared memory state in the case of a system crash.
# 
# WARNING: IT IS *NOT* SAFE TO RUN THIS SCRIPT AT OTHER TIMES!
#------------------------------------------------------------------------

#
# Make sure we are running as root
#
verify_we_are_root();

#
# Parse input options.
#
process_arguments();

print scalar(localtime), ": Reconciliation begins ...\n";

# First change directories into the volumes directory and see if there
# are subdirectories there (which would be volumes not yet processed
# and deleted by the archiver)

chdir($volume_path_base) || die "Unable to chdir to $volume_path_base: $!\n";

if ( opendir(VOLDIR, $volume_path_base) ) {
	my $subdir;
	my @process_list;

	#
	# Note!  We need to compile list of all the directories that
	# need processing before actually processing them since we
	# may be deleting entries from the directory we are reading,
	# which will screw up our readdir processing!
	#
	while ( defined ($subdir = readdir VOLDIR) ) {
		print "\nChecking $subdir ... all digits? ..." if ($debug);

		next if ($subdir =~ /\D/);	# skip names with non-digits
		print " dot file? ... " if ($debug);

		next if ($subdir =~ /^\./);	# skip all "dot" files
		print " directory? ... " if ($debug);

		next if (! (-d $subdir)); 	# skip regular files
		print "processing\n" if ($debug);

		push @process_list, $subdir;
	}
	closedir(VOLDIR);


	while ($subdir = pop @process_list) {
		reconcile_volume($subdir);
	}
}
print scalar(localtime), ": Reconciliation complete.\n";
exit 0;


#########################################################################
#############           S U B R O U T I N E S           #################
#########################################################################

#------------------------------------------------------------------------
# Check whether volume is in the database.
# If so, delete the data.
# If not, do the apvsync call to make the
# volume available to the archiver.
#------------------------------------------------------------------------
sub reconcile_volume {
	my ($volid, @args) = @_;
	print scalar(localtime), ": Reconciling volume $volid\n";

	if ( filedb_volume_has_db_entry($volid) ) {
		print scalar(localtime),
			": DB entry exists for volume $volid.",
			"\n\tRemoving data from the disk...\n";
		purge_data_from_disk($volid);
	}
	else {
		print scalar(localtime),
			": No DB entry exists for volume $volid.",
			"\n\tMaking volume available to archiver ...\n";
		semaphore_produce($volid);	
	}
}

#------------------------------------------------------------------------
# Process command-line arguments
#------------------------------------------------------------------------
sub process_arguments {
	GetOptions( \%opts,	"help",
				"debug:i",
				"db_file:s",
				"pathtovols=s",
		  );

	#
	# If --help was specified, just spit back the options available ...
	#

	if ( exists $opts{help} ) {
	print "\nUsage: reconcile takes the following options:
	--help			- produce this help text
	--debug[=<level>]	- produce debugging output
				  (debugging level optional)
	--db_file[=<dbfile>]	- use file DB, optionally specify file to use
	--pathtovols=<path>	- the pathname where the volume directories
				  created by the pkt_dumper can be found
	\n";

	}

	#
	# Redirect STDOUT and STDERR to a log file
	#

	open (LOGF, '>>/var/log/reconcile.log') ||
	 	die "Opening /var/log/reconcile.log: $!";
	*STDOUT = *LOGF;
	*STDERR = *LOGF;

	#
	# If --debug was specified, but $debug is zero, then set $debug to 1
	#
	if ( exists $opts{debug} ) {
		if ( $opts{debug} == 0 ) { $debug = 1; }
		else { $debug = $opts{debug} }
	}
	else { $debug = 0 }

	#
	# If both db_file and db_mysql are specified, report an error.
	# If neither is specified, default to db_file
	#

	if ( exists $opts{db_file} && exists $opts{db_mysql} ) {
		die "Specify only one of db_file or db_mysql\n";
	}
	elsif ( !exists $opts{db_file} && !exists $opts{db_mysql} ) {
		$opts{db_file} = "";
	}

	if ( exists $opts{db_file} ) {
		if ( $opts{db_file} eq '' ) { $db_file = $DEFAULT_FILEDB_NAME }
		else { $db_file = $opts{db_file} }
	}
	else { $db_file = 0 }

	if ( exists $opts{db_mysql} ) {
		$db_mysql = 1;
		if ( $opts{db_mysql} eq '' )
			{ $table_name = $DEFAULT_MYSQL_TABLE }
		else
			{ $table_name = $opts{db_mysql} }
	}
	else { $db_mysql = 0 }

	#
	# Allow specification of the path where we should look for volumes
	#

	if ( exists $opts{pathtovols} )
		{ $volume_path_base = $opts{pathtovols} }
	else
		{ $volume_path_base = $DEFAULT_VOLUME_PATH }

	#
	# Print out the results of command-line option processing
	#

	print scalar(localtime), ": Starting reconciliation with the following options:\n
		debug		'$debug'
		db_file		'$db_file'
		pathtovols	'$volume_path_base'
	\n";
}

#------------------------------------------------------------------------
# Verify we are running as root 
#------------------------------------------------------------------------
sub verify_we_are_root {
	if ($UID != 0  || $EUID != 0) {
		print "Sorry, this program must be run as root.\n";
		print "(UID is $UID EUID is $EUID)\n";
		exit 1;
	}
}


#------------------------------------------------------------------------
# Return the directory path for the specified volid
#------------------------------------------------------------------------
sub volume_path {
	my ($volid, @args) = @_;

	return $volume_path_base . "/" . $volid;
}


#------------------------------------------------------------------------
# Purge a volume's data from the disk
#------------------------------------------------------------------------
sub purge_data_from_disk {
	my ($volid, @args) = @_;

	my $volpath = volume_path($volid);

	print "purge_data_from_disk: Purging data in $volpath\n" if ($debug);

	if ( ! chdir $volpath ) {
		print "purge_data_from_disk: Unable to cd to $volpath";
		return 1;
	}
	my @failed = grep {not unlink} <*>;
	if (@failed) {
		print "purge_data_from_disk: Unable to remove @failed from $volpath: $!\n";
		return 1;
	}

	if ( ! rmdir($volpath) ) {
		print "purge_data_from_disk: Unable to rmdir $volpath: $!\n";
		return 1;
	}

	return 0;
}

#------------------------------------------------------------------------
# Check to see if there is a DB entry for the given volume
#------------------------------------------------------------------------
sub filedb_volume_has_db_entry {
	my ($volid, @args) = @_;

	my $found = 0;
	my ($fullid, $line, @lines);

	if ( ! open(DBFILE, "$db_file") ) {
		die "filedb_volume_has_db_entry: Unable to open $db_file: $!\n";
	}
	flock DBFILE, $LOCK_EXCLUSIVE;

	@lines = <DBFILE>;		# Read entire file into an array XXX

	$fullid = sprintf "%08d", $volid;


	#
	# Process the lines backwards.  That way, if there is an entry
	# we'll find it faster since it would have been recently added.
	# If there is not entry, we have to go through each line anyway.
	#
	while ( $line = pop @lines ) {
		print "Checking for $fullid in line '$line'\n" if ($debug > 10);
		if ( $line =~ /^$fullid\t/ ) {
			$found = 1;
			last;
		}
	}

	flock DBFILE, $LOCK_UNLOCK;
	close DBFILE;

	return $found;
}


#------------------------------------------------------------------------
# Use the apvsync program to inform others that we're done with a volume
#------------------------------------------------------------------------
sub semaphore_produce {
	my ($volid, @args) = @_;
	my @produce_args = ($apvsync, "produce", $volid);
	my $rc;

	$rc = 0xffff & system (@produce_args);

	if ($rc) {
		print "semaphore_produce: sync call failed ($rc)\n";
	}
}
