#!/usr/bin/perl

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

package Pilot;

use strict;
use English;
use Cwd;
use Getopt::Long;
use File::Basename;
use Errno qw(:POSIX);
use IO::Handle;
use Time::HiRes qw(time);


#========================================================================
#
# background tasks for listening to the net.
#
# Start by initializing the two keys:
# volume key, and the translation key.
#
# We want to create `volumes' of data to be tar'ed
# onto a tape drive...
#
# Each volume shouldn't exceed 1G, so for sanity's sake,
# We may yet decide to up this based on disk drive sizes...
#
# here's how it works....
# we sniff the net using the listen command.
# it makes files in a directory called. 'raw'.
#
# The pkt_dump command then is used to encrypt the data in the
# sniffed data.  It generates three files from each data file.
# The first file is called the data file, the second the `s' file
# and the third the `t' file.  The s file contains th substituted
# ip addresses, and the `t' file contains the substituted ip/pair
# addresses and their associated payload keys.
#
#========================================================================

#------------------------------------------------------------------------
# Global variable declarations
#------------------------------------------------------------------------

#-----------------------------------------------
# These variables may be customized as required
#-----------------------------------------------

my $rawMount   = "/mfs";
my $rawDir     = "/mfs/raw/";		# Note: trailing '/' required!
my $ufsMount   = "/scratch0";
my $ufsDir     = "$ufsMount/volumes";
my $pilotDir   = dirname($PROGRAM_NAME);
my $listenDir  = "$pilotDir/../listen";
my $dumpDir    = "$pilotDir/../dump";
my $utilDir    = "$pilotDir/../util";
my $volDir     = "$pilotDir/volumes";
my $volInfoDir = "$pilotDir/volInfo";

# 
# The following can all be specified via command-line option as
# segsize=??, segtime=??, mfssize=??, volsize=??, interface=??
# segsize, volsize, and mfssize are all specified in MB
#    (i.e. --segsize=20 means 20MB segments, --mfssize=256 means 256MB MFS)
# segtime is in seconds
# 

# segSize is when we start worrying about a new segment, size-wise...
# segTime is when we start worrying about a new segment, time-wise...
my $segSize    = 16*1024*1024;
my $segTime    = 60;

my $sysname = `uname -s`;
chomp($sysname);

if ($sysname eq "FreeBSD") {
}
elsif ($sysname eq "OpenBSD") {
}
elsif ($sysname eq "Linux") {
}
else {
        die "Need to figure out sysname...\n";
}

# mfsSize is the size of the memory file system
# This is the number of 512-byte sectors.
# For 256MB of MFS, we need 256*2*1024 512-byte sectors.
my $mfsSize;
if ($sysname eq "Linux") {
	$mfsSize = "256M";
}
else {
	$mfsSize   = 256*2*1024;
}

# maxVolSize is when we start worrying about a new volume...
my $maxVolSize  = 1000*1024*1024;
 
# mfsThreshold is when we refuse to store more bytes into MFS...
# ufsThreshold is when we refuse to store more bytes into UFS...
my $mfsThreshold = 32;
my $ufsThreshold = 1024;

# interface is whence the vault fetches packets
my $interface;

my $buffaloTest = 0;			# Set to 1 for testing only !!

if ($buffaloTest) {
	$interface  = "xl0";		# For testing only!
}
elsif ($sysname eq "Linux") {
	$interface = "eth0";
}
elsif ($sysname eq "OpenBSD") {
	$interface  = "fxp0";
}
elsif ($sysname eq "FreeBSD") {
	$interface  = "unknown0";
}

#------------------------------------
# These shouldn't need to be changed
#------------------------------------

my $time_to_quit = 0;
my $pid;
my $volume;
my $volumeSize;
my($this_vol, $last_vol); # used to keep track of time to produce a volume

my $highVol    = $volInfoDir . "/highVolNum";
my $incrHighVol= $volInfoDir . "/incrVolNum";
my $resetVol   = $volInfoDir . "/resetVolNum";

my $highVolSet = $volInfoDir . "/highVolSetNum";
my $incrHighVolSet = $volInfoDir . "/incrVolSetNum";

my $apvsync    = $utilDir . "/apvsync";

# Default values for things changable by command-line arguments
my $debug      = 0;
my $cleartext  = 0;
my $noarchive  = 0;
my $cryptlvl   = 3;
my $cryptfmt   = 1;


#########################################################################
#############           S U B R O U T I N E S           #################
#########################################################################

#------------------------------------------------------------------------
# Logging routines.  Adds timestamp.
#------------------------------------------------------------------------
sub apvLog {
        my (@args) = @_;
	print scalar(localtime), ": ", @args;
}

sub apvLogError {
	my (@args) = @_;
	print stderr scalar(localtime), ": ", @args;
}

#------------------------------------------------------------------------
# The main processing loop.
#------------------------------------------------------------------------
sub processFiles {
	my $sleeptime;

	do {
		local $SIG{INT} = sub {
			apvLog "Caught signal in processFiles\n";
			$time_to_quit = 1;
		};

		processBatch();

		# Sleep for 1/4 second and then look again...
		$sleeptime = 0.25;
		select(undef, undef, undef, $sleeptime);

	} until $time_to_quit;
	
	# We've been told to quit.  Allow the listener to finish writing
	# to MFS, and process the last segment stored there.
	sleep(1);
	apvLog("Processing last batch of MFS files\n");
	processBatch();

	# Make the last (partial) volume available to the archiver
	system($apvsync, "produce", $volume);
}

#------------------------------------------------------------------------
# Process a batch of segments stored in MFS by the listener.
#------------------------------------------------------------------------
sub processBatch {
	my($this_seg, $last_seg, $processing_start);

	$this_seg = $this_vol = time();

	while (<$rawDir:*>) {
		my $file = $_;
		$processing_start = time();
		processSingleFile($file);

		volSizeCheck();
		$last_seg = $this_seg;
		$this_seg = time();
		print("segment dt = ",
		      sprintf("%.2fs",$this_seg - $last_seg),
		      sprintf("(%.2fs idle)",
			      $processing_start - $last_seg),"\n");
	}
}

#------------------------------------------------------------------------
# Process a single file.
#------------------------------------------------------------------------
sub processSingleFile {
	my ($fullfile, @args) = @_;

	my $i = basename($fullfile);
	my $d = dirname($fullfile);

	my $xin = "X" . $i;
	my $fullxin = "$d/$xin";

	my $cleartextOption = "";
	my $p = vPath($volume);
	
	my ($pid, $rc);

	apvLog "Processing file $fullfile\n" if ($debug);
	$cleartextOption = "-c $p/c$i" if ($cleartext);
	apvLog "clearTextOption is '$cleartextOption'\n" if ($debug);

 	my $pkt_dump_cmd = "$dumpDir/pkt_dump.sh";
	my $pkt_dump_args = "-b2 -y -e $cryptlvl " .
			"-V $p/volKey -T $p/transKey " .
			"-o $p/$i -s $p/s$i -t $p/t$i " .
			"-w $fullxin -x $p/x$i -z $cryptfmt " .
			"$cleartextOption $fullfile";

	apvLog "Running $pkt_dump_cmd $pkt_dump_args\n" if ($debug);

	#
	# Run pkt_dump in a separate process. (Ignoring interrupt signals.)
	#

	if ($pid = fork) {
		# -- PARENT --
		local $SIG{INT} = sub {
			apvLog "Caught signal while running pkt_dump\n";
			$time_to_quit = 1;
		};
		waitpid($pid, 0);
		$rc = $?;
	}
	else {
		# -- ERROR -- (if $pid is undefined)
		die "Unable to fork: $!" unless defined $pid;

		# -- CHILD --

		$SIG{INT} = "IGNORE";
		exec("$pkt_dump_cmd $pkt_dump_args");
	}

	if ( $rc != 0 ) {
		if ( $rc == 0xff00 ) {
			apvLog "pkt_dump failed!\n";
		}
		elsif ( $rc > 0x80 ) {
			$rc >>= 8;
			apvLog "pkt_dump returned with $rc\n";
		}
		else {
			$rc &= ~0x80 if ( $rc & 0x80 );
			apvLog "pkt_dump ended from signal $rc\n";
		}
		apvLog "Terminating after pkt_dump failure!\n";
		$time_to_quit = 1;
	}

	# Remove the input files from MFS
	unlink($fullfile) or die "Cannot unlink $fullfile: $!";
	unlink($fullxin) or die "Cannot unlink $fullxin: $!";

	# Update the current volume size
	$volumeSize += (-s "$p/$i");
	$volumeSize += (-s "$p/s$i");
	$volumeSize += (-s "$p/t$i");
	$volumeSize += (-s "$p/c$i") if ($cleartext);
}

#------------------------------------------------------------------------
# Interrupt signal handler
#------------------------------------------------------------------------
sub SIGINT_Handler {
	apvLogError "Caught SIGINT, cleaning up!\n";
	$time_to_quit = 1;
	$SIG{INT} = 'IGNORE';
}


#------------------------------------------------------------------------
# Process command-line options 
#------------------------------------------------------------------------
sub processOptions {
	my %opts;

	GetOptions( \%opts,	"help",
				"debug:i",
				"interface=s",
				"crypt=s",
				"format=s",
				"segsize=i",
				"segtime=i",
				"volsize=i", 
				"mfssize=i",
				"mfsthreshold=i",
				"ufsthreshold=i",
				"ufsmount=s",
				"cleartext",
				"noarchive",
		  );

	#
	# If --help was specified, just spit back the options available ...
	#

	if ( exists $opts{help} ) {
		print "\nUsage: archiver takes the following options:
		--help			- Display this help text
		--debug[=<level>]	- Turn on debugging and set debug level
					  (debug level is one if not specified)
		--interface=<intf>	- Specify the ethernet interface to use
		--crypt[=<none|desx|aes|aes2>
					- Specify encryption method
					  (default is aes2, Gladman's asm aes)
		--format=<prototype or openheader|conversation|endpoint>
					- Specify encryption format
					  (default is conversation)
		--segsize=<size>	- Specify max segment size in MB
		--segtime=<size>	- Specify max segment time in seconds
		--volsize=<size>	- Specify volume size in MB
		--mfssize=<size>	- Specify MFS file system size in MB
		--mfsthreshold=<size>	- Specify MFS threshold in MB
		--ufsthreshold=<size>	- Specify UFS threshold in MB
		--ufsmount=<mtpoint>	- Specify where UFS space is mounted
		--noarchive		- Specify that volumes should just be
					  deleted and not given to the archiver
		\n";
		exit 1;
	}

	#
	# Process the options...
	#

	#
	# If --debug was specified, but is zero, then set $debug to 1
	#
	if ( exists $opts{debug} ) {
		if ( $opts{debug} == 0 ) { $debug = 1; }
		else { $debug = $opts{debug} }
	}
	else { $debug = 0 }

	#
	# If --interface was specified, then use the specified value
	#

	if ( exists $opts{interface} ) {
		$interface = $opts{interface };
	}

	#
	# Allow specification of the crypt
	#

	if ( exists $opts{crypt} ) {
		if    ( $opts{crypt} eq "none" ) { $cryptlvl = 0 }
		elsif ( $opts{crypt} eq "desx" ) { $cryptlvl = 1 }
		elsif ( $opts{crypt} eq "aes" )  { $cryptlvl = 2 }
		elsif ( $opts{crypt} eq "aes2" ) { $cryptlvl = 3 }
		else  { die "invalid crypt option: $opts{crypt}" }
	}

	#
	# Allow specification of encryption format
	#

	if ( exists $opts{format} ) {
		if    ( $opts{format} eq "prototype" )    { $cryptfmt = 0 }
		elsif ( $opts{format} eq "openheader" )    { $cryptfmt = 0 }
		elsif ( $opts{format} eq "conversation" ) { $cryptfmt = 1 }
		elsif ( $opts{format} eq "endpoint" )     { $cryptfmt = 2 }
		else  { die "invalid format option: $opts{format}" }
	}

	#
	# Allow specification of segment size
	#

	if ( exists $opts{segsize} ) {
		$segSize = ($opts{segsize} * 1024 * 1024);
	}

	#
	# Allow specification of segment time
	#

	if ( exists $opts{segtime} ) {
		$segTime = $opts{segtime};
	}

	#
	# Allow specification of volume size
	#

	if ( exists $opts{volsize} ) {
		$maxVolSize = ($opts{volsize} * 1024 * 1024);
	}

	#
	# Allow specification of memory file system
	# mfs size is in 512-byte sectors, need to
	# multiply by 2 and 1024 to get MB
	#

	if ( exists $opts{mfssize} ) {
		$mfsSize = ($opts{mfssize} * 2 * 1024);
	}

	#
	# Allow specification of MFS threshold
	# If the remaining space in the MFS goes
	# below this value, in megabytes, we terminate
	#

	if ( exists $opts{mfsthreshold} ) {
		$mfsThreshold = ($opts{mfsthreshold});
	}

	#
	# Allow specification of UFS threshold
	# If the remaining space in the UFS goes
	# below this value, in megabytes, we terminate
	#

	if ( exists $opts{ufsthreshold} ) {
		$ufsThreshold = ($opts{ufsthreshold});
	}

	#
	# Allow specification of UFS mountpoint
	#

	if ( exists $opts{ufsmount} ) {
		$ufsMount = ($opts{ufsmount});
		$ufsDir = "$ufsMount/volumes";
	}

	#
	# Allow specification of writing cleartext files
	# (SHOULD BE USED FOR TESTING ONLY!)
	#

	if ( exists $opts{cleartext} ) {
		apvLog "WARNING: Will be writing cleartext files!!!\n";
		$cleartext = 1;
	}

	if ( exists $opts{noarchive} ) {
		apvLog "WARNING: Will *NOT* be archiving data!!!\n";
		$noarchive = 1;
	}

	#
	# Print out the results of command-line option processing
	#

	apvLog "Starting $PROGRAM_NAME with the following options:\n
		debug		'$debug'
		interface	'$interface'
		crypt		'$cryptlvl'
		format		'$cryptfmt'
		segsize		'$segSize'
		segtime		'$segTime'
		volsize		'$maxVolSize'
		mfssize		'$mfsSize'
		mfsthreshold	'$mfsThreshold'
		ufsthreshold	'$ufsThreshold'
		ufsmount	'$ufsMount'
		cleartext	'$cleartext'
		noarchive	'$noarchive'
	\n";
	return 0;
}

#------------------------------------------------------------------------
# Verify we are running as root 
#------------------------------------------------------------------------
sub verify_we_are_root {
	if ($UID != 0  || $EUID != 0) {
		apvLogError "Sorry, this program must be run as root.\n";
		apvLogError "(UID is $UID EUID is $EUID)\n";
		exit 1;
	}
}

#------------------------------------------------------------------------
# return the volume path given the volume number.
#------------------------------------------------------------------------
sub vPath {
	my ($volname, @args) = @_;
	return join("/", $volDir, $volname);
}

#------------------------------------------------------------------------
# return the file path given the volume number and file name.
#------------------------------------------------------------------------
sub vPathFile {
	my ($volname, $filename, @args) = @_;
	return join("/", $volDir, $volname, $filename);
}

#------------------------------------------------------------------------
# create a random key using the srandom device.
#------------------------------------------------------------------------
sub makeKey {
	my ($keyfile, @args) = @_;
	my $random;
	my $rc;

	if (-r "/dev/srandom") {
		$random = "/dev/srandom";
	}
	elsif (-r "/dev/urandom") {
		$random = "/dev/urandom";
	}
	else {
		die "Need random source -- no /dev/srandom or /dev/urandom!\n";
	}

        $rc = 0xffff & system("dd", "if=$random", "of=$keyfile", "bs=8", "count=3");
	if ($rc) {
		apvLogError "Error in makeKey for $keyfile!!\n";
		return 1;
	}
	return 0;
}

#------------------------------------------------------------------------
# make new volume and translation table keys.
#------------------------------------------------------------------------
sub makeNewKeys {
	my ($rc);

	apvLog "Making new vol/trans keys\n";
 
	my $volkeyFile = vPathFile($volume, "volKey");
	makeKey($volkeyFile);
	
	apvLog "Done with volkey now do transKey\n";
	
	my $transkeyFile = vPathFile($volume, "transKey");
	makeKey($transkeyFile);
	
	apvLog "Done with transkey\n";
}

#------------------------------------------------------------------------
# check if volume size exceeded.  switch to new volume if so.
#------------------------------------------------------------------------
sub volSizeCheck {
	my $rounded;
	$rounded = sprintf("%.3f", $volumeSize/(1024*1024));
	apvLog "Current volume size: $volumeSize ($rounded M)\n";
	if ($volumeSize >= $maxVolSize) {
		apvLog "volumeSize $volumeSize exceeds maxVolSize $maxVolSize\n";
		if ($noarchive) {
			my $vtodelete = vPath($volume);
			apvLog "Deleting volume $volume ($vtodelete)\n";
			system("/bin/rm", "-rf", $vtodelete);
		}
		else {
			system($apvsync, "produce", $volume);
		}

		makeVolume();
		$last_vol = $this_vol;
		$this_vol = time();
		print("total timing for this volume: ",
		      sprintf("%.2f",$this_vol-$last_vol),"\n");
	}
}

#------------------------------------------------------------------------
# create a new ufs volume
#------------------------------------------------------------------------
sub makeVolume {

	#
	# Start the next volume
	#
	$volumeSize = 0;
	$volume = `$incrHighVol $volInfoDir`;
	chop($volume);
	my $newVolumeDir = vPath($volume);
	mkdir($newVolumeDir, 0755) ||
		die "cannot create volume directory $newVolumeDir: $!";

	#
	# and make new keys
	#
	makeNewKeys();
}

#########################################################################
#############  M A I N   E X E C U T I O N   P O I N T  #################
#########################################################################

#
# Make STDOUT non-buffered
#
STDOUT->autoflush;

#
# Make sure we are running as root
#
verify_we_are_root();


#
# Process the command-line options
#
processOptions();

apvLog "$PROGRAM_NAME begins...\n";

#
# Unmount any MFS filesystems currently mounted ...
#
my $umount = join("/", $pilotDir, "umount.sh");
apvLog "Calling '$umount $rawMount'\n" if ($debug);
system($umount, $rawMount) == 0
	or die "'$umount $rawMount' failed: $?";

#
# Touch a file under the mount point to make sure it disappears
#
my $hiddenFile = "$rawMount/ThisFileShouldBeHidden";
system("touch", $hiddenFile) == 0
	or die "'touch $hiddenFile' failed: $?";

#
# Create and Mount a new MFS filesystem
#
if ($sysname eq "Linux") {
	apvLog "/bin/mount -t tmpfs -o size=$mfsSize /dev/shm $rawMount\n"
		if ($debug);
	system("/bin/mount", "-t", "tmpfs", "-o", "size=$mfsSize",
			"/dev/shm", $rawMount) == 0
		or die "'mount of mfs failed: $?";
}
else {
	apvLog "mount_mfs -s $mfsSize swap $rawMount\n" if ($debug);
	system("mount_mfs", "-s", $mfsSize, "swap", $rawMount) == 0
		or die "'mount_mfs -s $mfsSize swap $rawMount' failed: $?";
}

#
# Wait for the hidden file to disappear
#
while (-e $hiddenFile) {
	select(undef, undef, undef, 0.1);
}

apvLog "/bin/rm -rf $rawDir\n" if ($debug);
system("/bin/rm", "-rf", $rawDir) == 0
	or die "'/bin/rm -rf $rawDir' failed: $?";
apvLog "mkdir $rawDir\n" if ($debug);
mkdir($rawDir, 0755) == 1
	or die "Error creating directory $rawDir: $!";

if ($buffaloTest) {
        apvLogError "*** Skipping mount of $ufsMount ***\n";
}
else {
	my $mount = join("/", $pilotDir, "mount.sh");
	apvLog "$mount $ufsMount\n" if ($debug);
	system($mount, $ufsMount) == 0
		or die "'$mount $ufsMount' failed: $?";
}

unlink($volDir);
symlink($ufsDir, $volDir) || die "Error linking $ufsDir as $volDir";

#
# Get the next volume number to be used and create a new volume
#
$volume = `$incrHighVol $volInfoDir`;
chop($volume);

apvLog "New high volume number: $volume\n";

my $newVolumeDir = vPath($volume);
apvLog "Creating volume directory '$newVolumeDir'...\n";
mkdir($newVolumeDir, 0755) ||
		die "Cannot create volume directory $newVolumeDir: $!";

$volumeSize = 0;
makeNewKeys();

#
# Set up signal handler for Ctrl-C a.k.a. SIGINT
#
$SIG{INT} = \&SIGINT_Handler;

#
# Start the listener program by forking off another process
# and exec'ing the listener (the child).  We (the parent)
# continue on and process files that it produces.
#

if ($pid = fork) {
	#
	# -- PARENT --
	#
	processFiles();
	apvLog "Back from processFiles.\n";
	kill 'INT', $pid;		# Tell listener to stop
	apvLog "Waiting for listener to finish ...\n";
	waitpid $pid, 0;
	apvLog "Listener terminated.  Continuing with termination.\n";

} elsif (defined $pid) {
	#
	# -- CHILD --
	#
	my $listener = join("/", $listenDir, "listen.sh");

        exec($listener, "-f", $rawDir, "-s", $segSize,
			"-t", $segTime, "-u", $ufsMount,
			"-M", $mfsThreshold, "-U", $ufsThreshold,
			$interface)
	or die "Error exec'ing listener!!!: $!\n";

} else {
	die "Unable to fork the listener: $!\n";
}

#
# Before going away, umount the MFS so we don't leave
# any unencrypted data hanging around
#

apvLog "Unmounting $rawMount.\n";
apvLog "Calling '$umount $rawMount'\n" if ($debug);
system($umount, $rawMount);

apvLog "$PROGRAM_NAME exiting...\n";
exit 0;
