#!/usr/bin/perl -w

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

use strict;

package Changer;
use vars qw(@ISA @EXPORT $VERSION);
use Exporter ();
@ISA = qw(Exporter);
@EXPORT = qw( $debug $db_file $db_mysql );

require ChangerElement;

use English;
use Errno qw(:POSIX);
use IPC::SysV qw(IPC_PRIVATE IPC_RMID S_IRWXU IPC_CREAT IPC_EXCL
		 IPC_NOWAIT SEM_UNDO SETVAL GETVAL GETALL SETALL);
$VERSION = 0.1;


##########################################################################
# Changer is an aggregate of pickers, drives, slots, and portals
# within a tape changer unit
##########################################################################

#========================================================================#
# Global/Exportable symbols...
#========================================================================#

my $debug = 0;
my $db_file = "";
my $db_mysql = 0;

#========================================================================#
# Local configuration options
#========================================================================#

use Cwd;
my $topdir = cwd();		# We expect to be invoked in
				# the right directory

my $mt_pgm = "mt";		# Location of the mt command

my $chio_pgm = "/bin/chio";	# Location of the chio command

my $mount_sleep_time = 10;	# Number of seconds to wait after loading
				# a tape before expecting it to be ready
				# to be written to.

my $sysname = `uname -s`;
chomp($sysname);
my $status_cmd = "ERROR";

if ($sysname eq "FreeBSD") {
	$status_cmd = "status -v";	# chio status with voltag option
	$mt_pgm = "/usr/bin/mt";	# location of the mt command
}
elsif ($sysname eq "OpenBSD") {
	$status_cmd = "status";		# chio status command
	$mt_pgm = "/bin/mt";		# location of the mt command
}
elsif ($sysname eq "Linux") {
	$status_cmd = "status";		# chio status command
	$chio_pgm = $topdir . "/chio-mover";	# chio pgm location
	$mt_pgm = "/bin/mt";		# location of the mt command
}
else {
	die "Unknown sysname...";
}

#========================================================================#
# End of Local configuration options
#========================================================================#

my $LOCK_SHARED = 1;
my $LOCK_EXCLUSIVE = 2;
my $LOCK_NOBLOCK = 4;
my $LOCK_UNLOCK = 8;

my $UNKNOWN_TAPEID = "????????";

#=========================================================================
# Logging routines.  Adds timestamp.
#=========================================================================
sub apvLog {
	my ($drive_num, @args) = @_;
	print scalar(localtime), ": [$drive_num] ", @args;
}

sub apvLogError {
	my ($drive_num, @args) = @_;
	print stderr scalar(localtime), ": [$drive_num] ", @args;
}

#=========================================================================
# Semaphore routines for synchronization with other processes
#=========================================================================
my $SEM_LOCK = -1;
my $SEM_WAIT = 0;
my $SEM_UNLOCK = 1;
my $LOCK_KEY = 8675309;		# Jenny! :-)

sub enter_critical_section {
	my $semid;
	my $opstring;

	$semid = semget($LOCK_KEY, 0, 0);
	if (!defined($semid)) {
		if ($ERRNO == ENOENT) {
#			apvLog "?", "Creating new semaphore for critical section\n";
			$semid = semget($LOCK_KEY, 1, IPC_CREAT | IPC_EXCL | 0666);
			if (!defined($semid)) {
				die "enter_critical_section: error from semget (create): $!";
			}
			# Initialize the semaphore with the value one
			semctl($semid, 0, SETVAL, 1);
		}
		else {
			die "enter_critical_section: error from semget: $!";
		}
	}

	# Now get/lock the semaphore.  The SEM_UNDO makes sure that
	# the mutex is released when our process goes away unexpectedly

	$opstring = pack('sss', 0, $SEM_LOCK, SEM_UNDO);     # Decrement number allowed in

	semop($semid, $opstring)
		or die "enter_critical_section: error obtaining lock: $!";
}

sub exit_critical_section {
	my $semid;
	my $opstring;

	$semid = semget($LOCK_KEY, 0, 0);
	if (!defined($semid)) {
		die "exit_critical_section: Unexpected error from semget: $!";
	}

	# Now put/unlock the semaphore.  We must undo all of these when we go
	# away too.  Otherwise, only the decrements will be undone...

	$opstring = pack('sss', 0, $SEM_UNLOCK, SEM_UNDO);      # Increment number allowed in

	semop($semid, $opstring)
		or die "exit_critical_section: error releasing lock: $!";

	# This may not be entirely necessary, but better safe than sorry.
	# Without this during testing, one process kept hogging the
	# critical section and the others waited forever...

	sleep(1);
}

#=========================================================================
# Instantiate a new Changer
#=========================================================================

my ($drives_ref, $slots_ref, $portals_ref, $pickers_ref);

sub new {
	apvLog "?", "Changer::new: Creating new Changer ...\n" if ($debug);
	my $type = shift;	
	my $chgrdev = shift;
	my $tapedev = shift;
	my $self = {@_};

	$self->{chgrdev} = $chgrdev;
	$self->{tapedev} = $tapedev;
	$self->{db_handle} = undef;
	$self->{db_table} = undef;

	if ($debug) {
		my ($mypackage, $callerpackage);
		$mypackage = __PACKAGE__;
		$callerpackage = caller();
		apvLog "?", "Changer::new: mypackage '$mypackage' caller '$callerpackage'\n";
	}
	printf "Changer::new: Adding elements with '%s' and '%s'\n", "slot", $self->{chgrdev} if ($debug);
	$slots_ref = add_elements_type($self, "slot");
	printf "Changer::new: Adding elements with '%s' and '%s'\n", "drive", $self->{chgrdev} if ($debug);
	$drives_ref = add_elements_type($self, "drive");
	printf "Changer::new: Adding elements with '%s' and '%s'\n", "portal", $self->{chgrdev} if ($debug);
	$portals_ref = add_elements_type($self, "portal");
	printf "Changer::new: Adding elements with '%s' and '%s'\n", "picker", $self->{chgrdev} if ($debug);
	$pickers_ref = add_elements_type($self, "picker");
	return bless $self, $type;
}

#=========================================================================
# General methods that do not deal with a particular object
#=========================================================================

#-------------------------------------------------------------------------
# intervention_required
#-------------------------------------------------------------------------

sub intervention_required {
	my ($aref, @args) = @_; 
	my ($prompt, $junk);
	my $tty = 1;
	local *TTY;

	open (TTY, "> /dev/tty") or $tty = 0;
	while ($prompt = shift @$aref) {
		print STDERR $prompt;
		print TTY $prompt if ($tty);
	}
	close TTY if ($tty);

	eval {
		local $SIG{INT} = sub {
			die "received interrupt";
		};

		# Wait for something to be entered...
		$junk = <STDIN>;
	};
	if ( $@ ) {
		if ($@ !~ /received interrupt/ ) { die }
		else { return -1; }
	}
	return 0;
}

#=========================================================================
# Methods dealing with the drives array
#=========================================================================

sub num_drives {
	return scalar(@$drives_ref);
}

sub drive_full {
	my ($self, $i, @args) = @_;
	return @$drives_ref[$i]->full();
}

sub drive_contents {
	my ($self, $i, @args) = @_;
	apvLog $i, "drive_contents: request for drive $i returning '", @$drives_ref[$i]->contents(), "'\n"
		if ($debug);
	return @$drives_ref[$i]->contents();
}

#=========================================================================
# Methods dealing with the slots array
#=========================================================================

sub num_slots {
	return scalar(@$slots_ref);
}

sub slot_full {
	my ($self, $i, @args) = @_;
	return @$slots_ref[$i]->full();
}

sub slot_contents {
	my ($self, $i, @args) = @_;
	apvLog "?", "slot_contents: request for slot $i returning '", @$slots_ref[$i]->contents(), "'\n"
		if ($debug);
	return @$slots_ref[$i]->contents();
}

#=========================================================================
# Methods dealing with the portals array
#=========================================================================

sub num_portals {
	return scalar(@$portals_ref);
}

sub portal_full {
	my ($self, $i, @args) = @_;
	return @$portals_ref[$i]->full();
}

sub portal_contents {
	my ($self, $i, @args) = @_;
	apvLog "?", "portal_contents: request for portal $i returning '", @$portals_ref[$i]->contents(), "'\n"
		if ($debug);
	return @$portals_ref[$i]->contents();
}

#=========================================================================
# Methods dealing with the changer in general
#=========================================================================

#-------------------------------------------------------------------------
# The contents of the entire library doesn't make sense
# (This *could* be re-written to return an array of the
# contents of each element?)
#-------------------------------------------------------------------------

sub contents {
	return "N/A";
}

#-------------------------------------------------------------------------
# Returns the device name for the changer
#-------------------------------------------------------------------------

sub changer_device {
	my ($self, @args) = @_;
	return $self->{chgrdev};
}

#-------------------------------------------------------------------------
# Returns the device name for the tape drive
#-------------------------------------------------------------------------

sub tape_device_name {
	my ($self, @args) = @_;
	return $self->{tapedev};
}

#-------------------------------------------------------------------------
# Get/Set the debug variable.
#-------------------------------------------------------------------------

sub debug {
	my $self = shift;
	if (@_) { $debug = shift; }
	return $debug;
}

#-------------------------------------------------------------------------
# Get/Set the db_mysql variable.
#-------------------------------------------------------------------------

sub db_mysql {
	my $self = shift;
	if (@_) { $db_mysql = shift; }
	return $db_mysql;
}

#-------------------------------------------------------------------------
# Get/Set the db_file variable.
#-------------------------------------------------------------------------

sub db_file {
	my $self = shift;
	if (@_) { $db_file = shift; }
	return $db_file;
}

#-------------------------------------------------------------------------
# Initializes the use of the Database to determine whether
# a given tape is a scratch tape or not.
#-------------------------------------------------------------------------

sub db_init {
	if ($db_mysql) {
		my ($self, $dbh, $table, @args) = @_;

		$self->{db_handle} = $dbh;
		$self->{db_table} = $table;

		my $sql_statement = "SELECT COUNT(*) FROM $self->{db_table} WHERE tapeid=?";
		apvLog "?", "db_init: preparing: $sql_statement\n" if ($debug);
		$self->{sbt_sth} = $self->{db_handle}->prepare_cached($sql_statement)
				|| die $self->{db_handle}->errstr;
	}
	elsif ($db_file) {
		apvLog "?", "Using a flat file DB\n";
	}
	else {
		die "You must specify some kind of database to use!";
	}
}

#-------------------------------------------------------------------------
# Print contents and status of each element
#-------------------------------------------------------------------------

sub print_content_status {
	my ($self, @args) = @_;

	my ($i, $type, $count, $full, $contents, $scratch, $status);
	my $scratch_count = 0;

	#
	# Print heading
	#

	apvLog "?", "Current contents of ", $self->changer_device(), " :\n";
	printf "%-9s   %-10s %s\n", " Element", "Contents", "Status";
	printf "%-9s   %-10s %s\n", "---------", "--------", "-------";

	#
	# for each element get it's contents and status (scratch or not)
	#

	# Process drives ...
	$type = "Drive";
	$count = num_drives();
	for ($i = 0; $i < $count; $i++) {
		$full = @$drives_ref[$i]->full();
		if ($full) {
			$contents = @$drives_ref[$i]->contents();
			$scratch = is_scratch($self, $contents);
			$status = $scratch ? "SCRATCH" : "WRITTEN";
		}
		else {
			$contents = "N/A";
			$scratch = 0;
			$status = "N/A";
		}
		if ($scratch) { $scratch_count++ }
		printf "%-6s %2d   %-10s %s\n", $type, $i, $contents, $status;
	}

	# Process slots ...
	$type = "Slot";
	$count = num_slots();
	for ($i = 0; $i < $count; $i++) {
		$full = @$slots_ref[$i]->full();
		if ($full) {
			$contents = @$slots_ref[$i]->contents();
			$scratch = is_scratch($self, $contents);
			$status = $scratch ? "SCRATCH" : "WRITTEN";
		}
		else {
			$contents = "N/A";
			$scratch = 0;
			$status = "N/A";
		}
		if ($scratch) { $scratch_count++ }
		printf "%-6s %2d   %-10s %s\n", $type, $i, $contents, $status;
	}

	# Process portals ...
	$type = "Portal";
	$count = num_portals();
	for ($i = 0; $i < $count; $i++) {
		$full = @$portals_ref[$i]->full();
		if ($full) {
			$contents = @$portals_ref[$i]->contents();
			$scratch = is_scratch($self, $contents);
			$status = $scratch ? "SCRATCH" : "WRITTEN";
		}
		else {
			$contents = "N/A";
			$scratch = 0;
			$status = "N/A";
		}
		if ($scratch) { $scratch_count++ }
		printf "%-6s %2d   %-10s %s\n", $type, $i, $contents, $status;
	}

	apvLog "?", "\nThere are currently $scratch_count scratch tapes in the changer.\n";
	if ($scratch_count <= 1) {
		apvLog "?", "Think about adding some scratch tapes!!!\n"
	}
}

#-------------------------------------------------------------------------
# Returns 1 (true) if a given tapeid is a scratch tape
#-------------------------------------------------------------------------

sub is_scratch {
	my ($self, $tapeid, @args) = @_;
	my $rows;
	my $fgrep_cmd;
	my $rv;

	# Special case; bad tapeid or no tapeid is never scratch
	if ($tapeid eq $UNKNOWN_TAPEID || $tapeid eq "") {
		return 0;
	}

	if ($db_mysql) {
		die "is_scratch called before db_init!" if ($self->{db_handle} == undef);

		# Execute the select command.  If we get back any entries, then
		# the specified tapeid is NOT scratch.  We can get back "undef"
		# if there was nothing to return OR there was an error.  If we
		# get back "undef" and there was an error, then we die.  Otherwise,
		# it means there was no entry and we return true.

		apvLog "?", "is_scratch: requested tapeid was $tapeid\n" if ($debug);

		$rv = $self->{sbt_sth}->execute($tapeid) || die $self->{sbt_sth}->errstr;
		$rows = $self->{sbt_sth}->rows();
		die $self->{sbt_sth}->errstr if ($rows == undef);
		return 1 if $rows > 0;
	}
	elsif ($db_file) {
		###------------------------------------------------------
		###  This method takes at least around 5 seconds to find
		###  a scratch tape in a file with a million records
		###------------------------------------------------------
		# open DBFILE, $db_file or die "Unable to open $db_file: $!";
		# while (<DBFILE>) {
		# 	if (/\b$tapeid\b/) {
		# 		close DBFILE;
		# 		return 0;
		# 	}
		# }
		# close DBFILE;
		# return 1;

		###------------------------------------------------------
		###  This method takes at least around 3 seconds to find
		###  a scratch tape in a file with a million records
		###------------------------------------------------------
		if (! open(DBFILE, $db_file ) ) {
			# Ignore error if the file doesn't exist yet
			if ($ERRNO == ENOENT) {
				die "File $db_file doesn't exist? : $!";
			}
			die "Unable to open $db_file: $!";
		}
		flock DBFILE, $LOCK_SHARED;

		$fgrep_cmd = "fgrep -i -w \"$tapeid\" $db_file | wc ";
		#$fgrep_cmd = "fgrep -i -E \"\\<$tapeid\\>\" $db_file | wc ";
		my $grepout = `$fgrep_cmd`;

		# Get rid of newline and trim leading whitespace
		chomp($grepout);
		$grepout =~ s/^\s*//;

		my ($lines, $words, $chars);

		($lines, $words, $chars) = split(/\s+/, $grepout);
		printf "is_scratch: query for '%s' returned %d lines, %d words, and %d chars from '%s'\n",
			$tapeid, $lines, $words, $chars, $grepout  if ($debug);

		flock DBFILE, $LOCK_UNLOCK;
		close DBFILE;
		return ($lines == 0);

	}
	else {
		die "You must specify some kind of database to use!";
	}

	return 0;
}

#-------------------------------------------------------------------------
# Find an empty element (slot or portal, actually)
#-------------------------------------------------------------------------

sub find_empty_element {
	my ($self, @args) = @_;
	my ($i, $count, $empty_element);
	my $time_to_quit = 0;
	my ($type, $full);

	# update our status before continuing
	$self->update_elements_all();

	#
	# Check elements to find an empty one
	#

	$empty_element = "";

	while ( !$time_to_quit && $empty_element eq "") {
		# Check slots first ...
		$type = "slot";
		$count = num_slots();
		for ($i = 0; $i < $count; $i++) {
			$full = @$slots_ref[$i]->full();
			if (! $full) {
				$empty_element = $type . " " . $i;
				last;
			}
		}
		if ($empty_element eq "") {
			$type = "portal";
		$count = num_portals();
			for ($i = 0; $i < $count; $i++) {
				$full = @$portals_ref[$i]->full();
				if (! $full) {
					$empty_element = $type . " " . $i;
					last;
				}
			}
		} 
		if ($empty_element eq "") {
			my @int_msg;
			push @int_msg, "**** There are no empty elements in the tape library    ****\n";
			push @int_msg, "**** Manually remove one or more tapes from the library ****\n";
	
			if ( intervention_required(\@int_msg) ) {
				$empty_element = -1;
				$time_to_quit = 1;
			}
		}
	}
	return $empty_element;
}

#-------------------------------------------------------------------------
# Unloads a tape from the given drive.  This involves taking the tape
# offline and moving it from the drive back to the slot where it came
# from.  (If we don't know where it came from, then we try to find an
# empty slot or portal to move the tape to.
#-------------------------------------------------------------------------

sub unload_drive {
	my ($self, $drive_num, @args) = @_;
	my $contents = @$drives_ref[$drive_num]->contents();
	my $came_from = @$drives_ref[$drive_num]->came_from();
	my $drive_full = @$drives_ref[$drive_num]->full();
	my $status;


	if ($drive_full) {
		apvLog $drive_num, "unload_drive: unload $contents from drive $drive_num, which came from '$came_from'\n"
			if ($debug);
		if ($came_from eq "") {
			# Need to find an empty element to move this tape to
			apvLog $drive_num, "unload_drive: looking for a place to move tape currently in drive\n"
				if ($debug);
			$came_from = $self->find_empty_element();
			if ($came_from == -1) { return; }
			apvLog $drive_num, "unload_drive: moving $contents to empty element, '$came_from'\n"
				if ($debug);
		}

		# Issue correct mt command to take the tape drive offline
		my $device_name = $self->{tapedev} . $drive_num;
		apvLog $drive_num, "unload_drive: '$mt_pgm -f $device_name offline' ...\n"
			if ($debug);
		$status = system($mt_pgm, "-f", $device_name, "offline");
		apvLog $drive_num, "unload_drive: status from mt command was $status\n"
			if ($debug);

		# Issue correct chio command to move tape
		my @tn = split / /, $came_from;
		apvLog $drive_num, "unload_drive: Moving $contents (back) to $came_from\n"
			if ($debug);

	retry_move:
		apvLog $drive_num, "unload_drive: '$chio_pgm -f $self->{chgrdev} move drive $drive_num $tn[0] $tn[1]' ...\n" if ($debug);
		$status = system($chio_pgm, "-f", $self->{chgrdev}, "move",
					"drive", $drive_num, $tn[0], $tn[1]);
		if ($status) {
			apvLog $drive_num, "unload_drive: Error ($status) from move command, retrying...\n";
			sleep(1);
			goto retry_move;
		}
		# Update in-core status
		$self->update_elements_all();
	}
	else {
		apvLog $drive_num, "unload_drive: drive $drive_num is not full!\n" if ($debug);
	}
}

#-------------------------------------------------------------------------
# Load a tape from the given element into the given drive.
#-------------------------------------------------------------------------

sub load_drive {
	my ($self, $drive_num, $from_type, $from_num, @args) = @_;
	my $drive_full = @$drives_ref[$drive_num]->full();
	my $status;
	my $came_from = $from_type . " " . $from_num;

	apvLog $drive_num, "load_drive: requested to load drive $drive_num from $from_type $from_num\n" if ($debug);

	if ($drive_full) {
		$self->unload_drive($drive_num);
	}

	apvLog $drive_num, "load_drive: Moving from $from_type $from_num to drive $drive_num\n" if ($debug);
	$status = system($chio_pgm, "-f", $self->{chgrdev}, "move",
				$from_type, $from_num, "drive", $drive_num);
	die "$0: chio command failed!" if ($status);

	# Save where this tape came from
	@$drives_ref[$drive_num]->came_from($came_from);

	# Update in-core status
	$self->update_elements_all();
	sleep($mount_sleep_time);
}

#-------------------------------------------------------------------------
# Find and mount a scratch tape into the specified drive
#-------------------------------------------------------------------------

sub mount_scratch {
	my ($self, $desired_drive, @args) = @_;
	my ($i, $count, $scratch, $type, $contents);
	my $time_to_quit = 0;
	local $SIG{INT} = sub {
		apvLog $desired_drive, "mount_scratch: Caught SIGINT\n";
		$time_to_quit = 1;
	};

	apvLog $desired_drive, "mount_scratch: Entered with desired_drive $desired_drive\n"
		if ($debug);

	#
	# Only one archiver at a time allowed within here
	#
	enter_critical_section();

	$scratch = "";
	while ($scratch eq "") {

		#
		# See if we got an interrupt (Ctrl-C)
		#
		if ($time_to_quit) {
			exit_critical_section();
			return -1;
		}

		# update our status before continuing
		$self->update_elements_all();

		#
		# Check each element to find a scratch tape
		# already in the library
		#

		# Check only the $desired_drive since another
		# drive may have a scratch tape, but we can't
		# "steal" it from them...

		$type = "drive";
		$i = $desired_drive;
		apvLog $desired_drive, "mount_scratch: Checking drive $i for a scratch tape\n"
			if ($debug);
		$contents = $self->drive_contents($i);
		apvLog $desired_drive, "mount_scratch: drive $i contains '$contents'\n"
			if ($debug > 2);
		if ($contents ne "" && $self->is_scratch($contents)) {
			apvLog $desired_drive, "mount_scratch: '$contents' is a scratch!\n"
				if ($debug > 2);
			$scratch = $contents;
			last;
		}

		# ... then slots ...
		if ($scratch eq "") {
			$type = "slot";
			$count = num_slots();
			apvLog $desired_drive, "mount_scratch: Checking $count slots for a scratch tape\n" if ($debug);
			for ($i = 0; $i < $count; $i++) {
				$contents = $self->slot_contents($i);
				apvLog $desired_drive, "mount_scratch: slot $i contains '$contents'\n"
					if ($debug > 2);
				if ($contents ne "" && $self->is_scratch($contents)) {
					$scratch = $contents;
					apvLog $desired_drive, "mount_scratch: '$contents' is a scratch!\n"
						if ($debug > 2);
					last;
				}
			}
		}

		# ... then portals ...
		if ($scratch eq "") {
			$type = "portal";
			$count = num_portals();
			apvLog $desired_drive, "mount_scratch: Checking $count portals for a scratch tape\n" if ($debug);
			for ($i = 0; $i < $count; $i++) {
				$contents = $self->portal_contents($i);
				apvLog $desired_drive, "mount_scratch: portal $i contains '$contents'\n"
					if ($debug > 2);
				if ($contents ne "" && $self->is_scratch($contents)) {
					apvLog $desired_drive, "mount_scratch: '$contents' is a scratch!\n"
						if ($debug > 2);
					$scratch = $contents;
					last;
				}
			}
		}

		# If no scratch found in the above, we need more scratch tapes!!!!!!
		if ($scratch eq "") {
			my @int_msg;
			push @int_msg, "************************************************************\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "**** There are no SCRATCH tapes in the tape library     ****\n";
			push @int_msg, "**** Manually load more SCRATCH tapes into the library  ****\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "************************************************************\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "**** Press <ENTER> when this has been completed         ****\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "************************************************************\n";
	
			if ( intervention_required(\@int_msg) ) {
				$time_to_quit = 1;
			}
		}
	}

	apvLog $desired_drive, "Scratch tape, $contents, currently located in $type $i\n";

	#-----------------------------------------------------------------
	# Now do what is necessary to move that scratch tape into a drive
	#-----------------------------------------------------------------

	#
	# If the desired tape is not already in the desired drive,
	# load it.  The load_drive routine will deal with any
	# tape that might already be in the drive...
	#
	apvLog $desired_drive, "Checking '$type' against 'drive' and '$i' against '$desired_drive'\n" if ($debug);
	if ($type ne "drive" || $i != $desired_drive) {
		apvLog $desired_drive, "Loading $contents from $type $i into drive $desired_drive\n" if ($debug);
		$self->load_drive($desired_drive, $type, $i);
		apvLog $desired_drive,
			"Scratch tape, $contents, now in desired location, drive $desired_drive\n";
	}

	#
	# Leave the critical section before returning
	#
	exit_critical_section();

	return $contents;
}

#-------------------------------------------------------------------------
# Mount a specific tape into the specified drive
#-------------------------------------------------------------------------
sub mount_specific {
	my ($self, $desired_drive, $tape_id, @args) = @_;
	my ($i, $type, $count);
	my $time_to_quit = 0;
	my $found = 0;

	apvLog $desired_drive, "mount_specific: Entered with desired_drive $desired_drive ",
	      "desired_tapeid $tape_id\n" if ($debug);

	local $SIG{INT} = sub {
		apvLog $desired_drive, "mount_specific: Caught SIGINT\n";
		$time_to_quit = 1;
	};

	#
	# Only one archiver at a time in this routine
	#
	enter_critical_section();

	while ( !$found && !$time_to_quit ) {

		# update our status before continuing
		$self->update_elements_all();

		#
		# Check each element to find the desired tape
		# already in the library
		#

		# Check only the desired drive
		$type = "drive";
		$i = $desired_drive;
		apvLog $desired_drive, "mount_specific: Checking drive $i for tape $tape_id\n" if ($debug);
		if ( $tape_id eq $self->drive_contents($i) ) {
			apvLog $desired_drive, "mount_specific: tape found in $type $i\n" if ($debug);
			$found = 1;
			last;
		}

		# ... then slots ...
		if ( !$found ) {
			$type = "slot";
			$count = num_slots();
			apvLog $desired_drive, "mount_specific: Checking $count slots for tape $tape_id\n" if ($debug);
			for ($i = 0; $i < $count; $i++) {
				if ( $tape_id eq $self->slot_contents($i) ) {
					apvLog $desired_drive, "mount_specific: tape found in $type $i\n" if ($debug);
					$found = 1;
					last;
				}
			}
		}

		# ... then portals ...
		if ( !$found ) {
			$type = "portal";
			$count = num_portals();
			apvLog $desired_drive, "mount_specific: Checking $count portals for tape $tape_id\n" if ($debug);
			for ($i = 0; $i < $count; $i++) {
				if ( $tape_id eq $self->portal_contents($i) ) {
					apvLog $desired_drive, "mount_specific: tape found in $type $i\n" if ($debug);
					$found = 1;
					last;
				}
			}
		}

		if ( !$found ) {
			my @int_msg;
			push @int_msg, "************************************************************\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "**** Could not locate tapeid $tape_id in the library    ****\n";
			push @int_msg, "**** Manually load the tape into the library            ****\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "************************************************************\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "**** Press <ENTER> when this has been completed         ****\n";
			push @int_msg, "****                                                    ****\n";
			push @int_msg, "************************************************************\n";
	
			if ( intervention_required(\@int_msg) ) {
				$time_to_quit = 1;
			}
		}
	}
	if ( !$found ) {
		exit_critical_section();
		return -1;
	}

	apvLog $desired_drive, "Tape $tape_id currently located in $type $i\n";

	#-----------------------------------------------------------------
	# Now do what is necessary to move that tape into the drive
	#-----------------------------------------------------------------

	#
	# If the desired tape is not already in the desired drive,
	# load it.  The load_drive routine will deal with any
	# tape that might already be in the drive...
	#
	apvLog $desired_drive, "Checking '$type' against 'drive' and '$i' against '$desired_drive'\n" if ($debug);
	if ($type ne "drive" || $i != $desired_drive) {
		apvLog $desired_drive, "Loading $tape_id from $type $i into drive $desired_drive\n" if ($debug);
		$self->load_drive($desired_drive, $type, $i);
		apvLog $desired_drive,
			"Tape $tape_id now in desired location, drive $desired_drive\n";
	}

	#
	# Leave the critical section before returning
	#
	exit_critical_section();
	return 0;
}

#-------------------------------------------------------------------------
# parse_chio_status
#
# This routine parses the output lines of the 'chio status' command.
# Returning the parsed output in the references supplied.
#-------------------------------------------------------------------------

sub parse_chio_status {
	my ($lineref, $tref, $nref, $sref, $cref, $eref, $fullref) = @_;

	my ($ignore);

	$$tref = '';	# type
	$$nref = '';	# number
	$$sref = '';	# status
	$$cref = '';	# contents
	$$eref = '';	# errors
	$$fullref = 0;	# empty until proven full

	# eliminate newline
	chomp($$lineref);

	if ($sysname eq "FreeBSD" or $sysname eq "Linux") {

		apvLog "?", "parse_chio_status: Using FreeBSD/Linux syntax\n"
			if ($debug > 5);
		# strip out angle and square brackets to reduce confusion
		$$lineref =~ s/[<>\[\]]//g;

		# now split the string at whitespace
		($$tref, $$nref, $$sref, $ignore, $$cref, $$eref) = split(/\s+/, $$lineref);

		#
		# The output for the picker has fewer fields.
		# Shift things now to fix that.
		#
		if ("$$tref" eq "picker") {
			$$sref = "";
			$$cref = $ignore;
		} 

		# get rid of colon in device number
		$$nref =~ s/://g;

		# get rid of colon and serial number from contents
		($$cref, $ignore) = split(/:/, $$cref);
	}
	elsif ($sysname eq "OpenBSD") {
		apvLog "?", "parse_chio_status: Using OpenBSD syntax\n"
			if ($debug > 5);
		# strip out colon and (angle & square) brackets to reduce confusion
		$$lineref =~ s/[:<>\[\]]//g;

		# now split the string at whitespace
		($$tref, $$nref, $$sref, $$cref, $$eref) = split(/\s+/, $$lineref);

		# The changer can get into a state where it has an exception
		# with no contents information.  The exception info
		# always (?) begins with 'ASC=', check for that case...

		if ( $$cref =~ m/ASC=/) {
			apvLog "?", "Ignoring exception info in contents area!\n";
			$$eref = $$cref;
			$$cref = "";
		}

		# Strip off "PVT=" from the contents (if contents there at all)
		$$cref =~ s/PVT=//;
	}
	else {
		die "parse_chio_status: unknown system name -- quitting.";
	}

	apvLog "?", "parse_chio_status: After split: t '$$tref' n '$$nref' s '$$sref' c '$$cref' e '$$eref'\n"
		if ($debug > 5);

	# Check for FULL in the status area
	$$fullref = $$sref =~ m/full/i;

	# If status indicates full, make sure contents contains *something*
	$$cref = $UNKNOWN_TAPEID if ($$fullref && !$$cref);

	# Upper-case the first letter of the type
	$$tref = ucfirst(lc($$tref));
}

#-------------------------------------------------------------------------
# Update all the different types of elements...
#-------------------------------------------------------------------------

sub update_elements_all {
	my $self = shift;
	update_elements_type($self, "drive");
	update_elements_type($self, "slot");
	update_elements_type($self, "portal");
	update_elements_type($self, "picker");
}

#-------------------------------------------------------------------------
# This routine updates the current status of all the elements of
# the changer.  It ASSUMES (1) that add_elements_type has already been
# invoked to get the initial picture, and (2) that the configuration
# of elements has not changed (there are no more and no fewer elements
# than when add_elements_type was initially called).
#-------------------------------------------------------------------------

sub update_elements_type {
	my ($self, $type) = ($_[0], $_[1]);

	my ($i, $t, $n, $s, $c, $e, $ignore);
	my ($full, $chio);
	my (@chio);
	my $checkcmd;

	$checkcmd = "$chio_pgm -f $self->{chgrdev} $status_cmd $type";
	apvLog "?", "update_elements_type: chio command returned $?\n" if ($debug > 5);
	apvLog "?", "update_elements_type: issuing command '$checkcmd'\n" if ($debug > 5);
	@chio = `$checkcmd`;
	die "$0: chio command failed!" if ($#chio < 0);

	printf "update_elements_type: The number of elements is %d, last index is %d\n",
			scalar(@chio), $#chio
		if ($debug > 5); 

	for ($i = 0; $i <= $#chio; $i++) {

		parse_chio_status(\$chio[$i], \$t, \$n, \$s, \$c, \$e, \$full);

		printf "update_elements_type: Element %s%d is %s%s.\n", $t, $n,
			$full ? "filled with " : "empty",
			$full ? $c : "" if ($debug > 5);

		apvLog "?", "update_elements_type: *** Updating a '$t' ***\n" if ($debug > 5);

		if ($t eq "Drive") {
			@$drives_ref[$i]->status($s);
			@$drives_ref[$i]->contents($c);
			@$drives_ref[$i]->errors($e);
			@$drives_ref[$i]->full($full);
		} elsif ($t eq "Slot") {
			@$slots_ref[$i]->status($s);
			@$slots_ref[$i]->contents($c);
			@$slots_ref[$i]->errors($e);
			@$slots_ref[$i]->full($full);
		} elsif ($t eq "Portal") {
			@$portals_ref[$i]->status($s);
			@$portals_ref[$i]->contents($c);
			@$portals_ref[$i]->errors($e);
			@$portals_ref[$i]->full($full);
		} elsif ($t eq "Picker") {
			@$pickers_ref[$i]->status($s);
			@$pickers_ref[$i]->contents($c);
			@$pickers_ref[$i]->errors($e);
			@$pickers_ref[$i]->full($full);
		}
	}
}

#-------------------------------------------------------------------------
# add_elements_type requires the element type
#
# This routine returns a reference to a an array of references
# to objects of the type specified.
#-------------------------------------------------------------------------

sub add_elements_type {
	my ($self, $type) = ($_[0], $_[1]);

	my ($i, $t, $n, $s, $c, $e);
	my ($full, $chio, $checkcmd);
	my (@chio, @e_a);		# e_a is element array
	my $newcontents;

	$checkcmd = "$chio_pgm -f $self->{chgrdev} $status_cmd $type";
	apvLog "?", "add_elements_type: type '$type'\n" if ($debug);
	apvLog "?", "add_elements_type: issuing command '$checkcmd'\n" if ($debug > 5);

	@chio = `$checkcmd`;
	die "$0: chio command failed!" if ($#chio < 0);

	printf "add_elements_type: The number of elements is %d, last index is %d\n",
			scalar(@chio), $#chio
		if ($debug > 5); 

	for ($i = 0; $i <= $#chio; $i++) {

		parse_chio_status(\$chio[$i], \$t, \$n, \$s, \$c, \$e, \$full);

		printf "add_elements_type: Element %s%d is %s%s.\n", $t, $n,
			$full ? "filled with " : "empty",
			$full ? $c : "" if ($debug > 5);

		apvLog "?", "add_elements_type: *** Creating a new '$t' ***\n" if ($debug > 5);

		if ($t eq "Drive") {
			$e_a[$i] = Drive->new(number => $n, status => $s, contents => $c, errors => $e);
		} elsif ($t eq "Slot") {
			$e_a[$i] = Slot->new(number => $n, status => $s, contents => $c, errors => $e);
		} elsif ($t eq "Portal") {
			$e_a[$i] = Portal->new(number => $n, status => $s, contents => $c, errors => $e);
		} elsif ($t eq "Picker") {
			$e_a[$i] = Picker->new(number => $n, status => $s, contents => $c, errors => $e);
		}

		printf "add_elements_type: The new $t object has number '%s' status '%s' contents '%s' errors '%s'\n",
			$e_a[$i]->{number}, $e_a[$i]->{status},
			$e_a[$i]->{contents}, $e_a[$i]->{errors} if ($debug > 5);

		$newcontents = $e_a[$i]->contents();
		printf "add_elements_type: The contents of $t$n is '$newcontents' [$e_a[$i]->{contents}]\n" if ($debug > 5);

		$full = $e_a[$i]->full();
		printf "add_elements_type: This element %sfull\n", $full ? "is " : "is not " if ($debug > 5);
	}
	return \@e_a;
}


##########################################################################
# The main stuff!
##########################################################################

return 1;
