/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <err.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/stat.h>

#include <sys/param.h>
#include <sys/mount.h>

#include "bpf.h"

/*
 * Global data
 */
int dflag = 1;
int nflag = 0;
int verbose = 1;
int active = 0;
char * intf;
int seg_size = 4*1024*1024;	/* max seg size (B) */
int seg_time = 60;		/* max fill time (s) */
int mfs_threshold = 32;		/* min free space required in MFS (MB) */
int ufs_threshold = 1024;	/* min free space required in UFS (MB) */

/* Where to put the data... */
char * oFile = "data";
char tempFileName[128];
char fileName[128];
char fileNameX[130];

char *ufsMount;

int readIt()
{
    return 0;
}


void usage(char * prog)
{
    fprintf( stderr,
    "Usage:\t%s [-f file] [-s segment size] [-t segment fill time]\n\t"
    "[-u ufs_mount_point] [-M mfs_threshold] [-U ufs_threshold] [-n] intf \n", 
	     prog);
    exit(1);
}


char *
makeTempFileName( char * file )
{
    sprintf( tempFileName, "%s-fill", file );
    return(tempFileName);
}


char *
makeFileName( char * file )
{
    time_t now = time(0);
    static time_t then;
    static time_t suffix;

    if (now == then) {
        suffix++;
    } else {
        suffix = 0;
    }
    sprintf( fileName, "%s:%ld.%d", file, (long int) now, suffix);
    sprintf( fileNameX, "%sX:%ld.%d", file, (long int) now, suffix);
    then = now;

    /* printf( "File: %s\n", fileName ); */
    /* printf( "XFile: %s\n", fileNameX ); */
    return(fileName);
}


void intfClose2(int s)
{
    printf("Received signal %d\n", s);
    intfClose();
    exit(102);
}


int main( argc, argv )
    int argc;
    char *argv[];
{
    int err;
    struct statfs statfsBuf;
    int mfsFree;
    int rc = 0;
    int ch;
    extern int optind;
    extern char *optarg;

    while ((ch = getopt(argc, argv, "M:U:f:i:s:t:u:n")) != -1) {
      switch (ch) {
      case 'M':
	mfs_threshold = atoi(optarg);
	break;
      case 'U':
	ufs_threshold = atoi(optarg);
	break;
      case 'f':
	oFile = optarg;
	break;
      case 'i':
	intf = optarg;
	break;
      case 's':
	seg_size = atoi(optarg);
	break;
      case 't':
	seg_time = atoi(optarg);
	break;
      case 'u':
	ufsMount = optarg;
	break;
      case 'n':			/* don't create permanent MFS files */
	++nflag;
	break;
      default:
	usage(argv[0]);
      }
    }
    if (optind+1 == argc) {
        intf = argv[optind++];
    }
    if (optind != argc) {
        usage(argv[0]);
    }
    if (setlinebuf(stdout) == 0)
	warnx("setlinebuf() succeeded");
    else
	warnx("setlinebuf() FAILED");

    printf("segment: size=%d, fill time=%d\n", seg_size, seg_time);
    printf("threshold: MFS=%d MB, UFS=%d MB\n", mfs_threshold, ufs_threshold);

    /*
     * now work
     */

    intfInit( intf, makeTempFileName( oFile ) );

    if (oFile) {
        fprintf( stderr, "writing: %s\n", oFile );
        if (signal(SIGINT,(void*)&intfClose2) <0)
            warnx("signal SIGINT");
        if (signal(SIGTERM,(void*)&intfClose2) <0)
            warnx("signal SIGTERM");
    }

    intfPromiscOn();

    intfStat();		/* Get initial reading before entering loop */

    while(1) {
        extern int w_fd;
        int then = time(0);
        struct stat s[1];

        if (w_fd >= 0 && fstat( w_fd, s ) != 0) {
            perror( "fstat" );
            exit(1);
        }
        while ( s->st_mtimespec.tv_sec  < then + seg_time ) {
            intfRead( readIt );
            if (s->st_size > seg_size) {
                /* printf( "big File: %x\n", s->st_size ); */
                break;
            }
            if (w_fd >= 0 && fstat( w_fd, s ) != 0) {
                perror( "fstat" );
                exit(1);
            }
        }

	/* Get numbers for this segment (before doing file stuff) */
    	intfStat();

	if (w_fd >= 0) {
            /* create permanent file */
            if (!nflag) {
		err = link( makeTempFileName( oFile ), makeFileName( oFile ) );
	        if (err) {
		    perror( "link" );
		}
	    }
	    /* remove fill file */
            err = unlink( makeTempFileName( oFile ) );
            if (err) {
                perror( "unlink" );
            }
	    /* reopen new fill file */
            intfFileSave( makeTempFileName( oFile ) );
	}

	/*
	 * Note that makeFileName updates fileNameX, so
	 * we must call intfWriteXFile *after* calling
	 * makeFileName.  And before calling intfStatDump
	 * which will reset the numbers...
	 */
	if (!nflag)
	    intfWriteXFile(fileNameX);

	/*
	 * Dump (and reset) the statistics gathering
	 * mechanism for each segment.
	 */
        mfsFree = intfStatDump();

	/*
	 * Quit if free space in MFS below threshold.
	 */
	if (mfsFree < mfs_threshold) {
		printf("MFS free space below threshold of %d MB\n", mfs_threshold);
		rc = 100;
		break;
	}

	/*
	 * Really quit if free space in UFS below threshold.
	 */
	if (ufsMount) {
		if (statfs(ufsMount, &statfsBuf) < 0)
			warn("statfs failure" );
		else {
			int ufsFree;

			ufsFree = statfsBuf.f_bavail/1024*statfsBuf.f_bsize/1024;
			if (ufsFree < ufs_threshold) {
				printf("UFS free space %d MB below threshold of %d MB\n", ufsFree, ufs_threshold);
				rc = 101; 
				break;
			}
		}
	}
    }

    if (oFile) { /* close data stream */
        intfFileSave(0);
    }
    intfClose();
    /* sample_dmp(); */

    exit(rc);
}
