/*
 * Copyright (c) 1996, 1997
 *	The Regents of the University of California.  All rights reserved.
 * 
 * This software has been derived at the University of Michigan from software
 * originally created as described below.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors
 * nor the name of the University of Michigan may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <pcap.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#ifndef __linux__
#ifdef LOCAL_BPF_HEADERS
#include "../sys/net/bpf.h"
#else
#include <net/bpf.h>
#endif
#endif
#include "../bpf/parse_bpf.h"

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <net/if_arp.h>
#include <linux/if_ether.h>
#include <linux/if_packet.h>
#include <linux/socket.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <linux/poll.h>
#include <linux/mman.h>
#include <asm/system.h>
#include <err.h>

#define RING_BUFFER_MB 32

static int fd;
unsigned char * ringbuf, * ringbuf_end;
unsigned char * current_page;
static int page_size;
static int frame_size;
static int frames_per_page;
static int pages;
static int frames;


struct intf_stats {
    struct timeval time;
    long long int bytes_received;
    long long int packets_received;
    long long int packets_dropped;
};

static struct intf_stats this_stats, last_stats;

static float dt, Bps;
static long long int bytes_received;


void
init_intf(char * intf) {
    struct ifreq ifr;
    struct tpacket_req req;
    struct packet_mreq mr;
    
    page_size = getpagesize();
    frame_size = TPACKET_ALIGN(TPACKET_HDRLEN)
	+ TPACKET_ALIGN(1514);
    frames_per_page = page_size/frame_size;
    pages = (RING_BUFFER_MB*1024*1024)/page_size;
    frames = pages*frames_per_page;
    
    if (frames == 0)
	errx(1,"frames == 0");
    
    fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
    if (fd < 0)
	err(1,"socket failed");
    
    req.tp_frame_nr = frames;
    req.tp_frame_size = frame_size;
    req.tp_block_nr = pages;
    req.tp_block_size = page_size;
    if (setsockopt(fd, SOL_PACKET, PACKET_RX_RING, (void *)&req, sizeof(req)))
	err(1,"setsockopt failed");
    
    ringbuf = mmap(0, pages*page_size, PROT_READ|PROT_WRITE|PROT_EXEC,
		   MAP_SHARED, fd, 0);
    if ((long)ringbuf == -1L)
	err(1,"mmap failed");

    memset(&ifr,0,sizeof(ifr));
    strcpy(ifr.ifr_name, intf);
    if (ioctl(fd, SIOCGIFINDEX, &ifr) < 0)
	err(1,"failed to get device index");
    memset(&mr,0,sizeof(mr));
    mr.mr_ifindex = ifr.ifr_ifindex;
    mr.mr_type = PACKET_MR_PROMISC;
    if (setsockopt(fd,SOL_PACKET,PACKET_ADD_MEMBERSHIP,(char *)&mr,sizeof(mr)))
	err(1,"failed to configure interface");
    
    ringbuf_end = ringbuf + pages*page_size;
    current_page = ringbuf;

    bytes_received = 0;
}

void
intf_read(struct bpf_hdr * hdr, char * bpf_content) {
    static int current_frame = 0;

    while (1) {
	while ( ((struct tpacket_hdr *)(current_page+current_frame))->tp_status
		!= TP_STATUS_KERNEL) {
	    struct tpacket_hdr *h = (struct tpacket_hdr *)
		(current_page + current_frame);
	    unsigned char *etherstart = (unsigned char*)h + h->tp_mac;
	    
	    hdr->bh_tstamp.tv_sec = h->tp_sec;
	    hdr->bh_tstamp.tv_usec = h->tp_usec;
	    hdr->bh_datalen = h->tp_len;
	    hdr->bh_caplen = h->tp_snaplen;
	    hdr->bh_hdrlen = BPF_HDR_SIZE;
	    memcpy(bpf_content,etherstart,h->tp_snaplen);

	    bytes_received += h->tp_snaplen;

	    h->tp_status = TP_STATUS_KERNEL;
	    mb(); /* Huh? */
	    current_frame += frame_size;
	    if (current_frame + frame_size > page_size) {
		current_frame = 0;
		current_page += page_size;
		if (current_page >= ringbuf_end)
		    current_page = ringbuf;
	    }
	    return;
	}
	while ( ((struct tpacket_hdr *)(current_page+current_frame))->tp_status
		== TP_STATUS_KERNEL) {
	    struct pollfd pfd;
	    
	    pfd.fd = fd;
	    pfd.revents = 0;
	    pfd.events = POLLIN|POLLRDNORM|POLLERR;
	    
	    while (1) {
		if (poll(&pfd, 1, -1) < 0) {
		    if (errno != EINTR)
			err(1,"poll failed");
		}
		else break;
	    }
	    if (pfd.revents&POLLERR)
		errx(1,"poll error");
	}
    }
}

void
get_tpstats(struct tpacket_stats * stats) {
    int stats_len=sizeof(*stats);

    if (getsockopt(fd, SOL_PACKET, PACKET_STATISTICS,
		   (void *)stats, &stats_len))
	warn("unable to get interface socket statistics");
    /* Note: doesn't give us bytes received, bytes snapped, or bytes dropped.
       We have no other way of getting bytes dropped, so perhaps that should
       be added to the kernel. */
}

void
init_intf_stats() {

    memset(&this_stats,0,sizeof(this_stats));
    memset(&last_stats,0,sizeof(last_stats));
    gettimeofday(&this_stats.time,NULL);
}

void
update_intf_stats() {
    struct tpacket_stats tpstats;
    int tpstats_len = sizeof(tpstats);

    memcpy(&last_stats,&this_stats,sizeof(this_stats));
    if (getsockopt(fd, SOL_PACKET, PACKET_STATISTICS,
		   (void *)&tpstats, &tpstats_len))
	warn("unable to get interface socket statistics");
    gettimeofday(&this_stats.time,NULL);
    this_stats.packets_received += tpstats.tp_packets;
    this_stats.packets_dropped += tpstats.tp_drops;
    this_stats.bytes_received = bytes_received;
    dt = (this_stats.time.tv_sec - last_stats.time.tv_sec) 
	+ (this_stats.time.tv_usec - last_stats.time.tv_usec)/1000000.0;
    if (dt == 0)
	dt = 1/1000000.0;
    Bps = (this_stats.bytes_received-last_stats.bytes_received)/dt;
}

void
print_intf_stats() {
    struct tm * tm_p = localtime((time_t *)&this_stats.time.tv_sec);
    printf("intfStat:  %02d/%02d/%02d %02d:%02d:%.2f dt:%.3f pr:%llu(+%llu) "
	   "br:%llu(+%llu) pd:%llu(+%llu) Bps:%.0f "
	   "Mbps:%.2f ",
	   tm_p->tm_mon+1,
	   tm_p->tm_mday,
	   tm_p->tm_year-100,
	   tm_p->tm_hour,
	   tm_p->tm_min,
	   tm_p->tm_sec + this_stats.time.tv_usec/1000000.0,
	   dt,
	   this_stats.packets_received,
	   this_stats.packets_received - last_stats.packets_received,
	   this_stats.bytes_received,
	   this_stats.bytes_received - last_stats.bytes_received,
	   this_stats.packets_dropped,
	   this_stats.packets_dropped - last_stats.packets_dropped,
	   Bps,
	   (Bps*8)/(1024*1024));
}

void
write_Xfile(char * filename) {
    FILE * file;

    if ((file = fopen(filename,"w")) == NULL)
	errx(1,"unable to open statistics file %s for writing",filename);
    fprintf(file,"%lu %lu %lu %lu %llu %llu %llu %llu %llu %u\n",
	    last_stats.time.tv_sec,
	    this_stats.time.tv_sec,
	    last_stats.time.tv_usec,
	    this_stats.time.tv_usec,
	    this_stats.packets_received - last_stats.packets_received,
	    this_stats.packets_dropped - last_stats.packets_dropped,
	    this_stats.bytes_received - last_stats.bytes_received,
	    0ULL, /* byes dropped, bytes snapped not available. */
	    0ULL,
	    (int)Bps
	    );
    fclose(file);
}


