/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/vfs.h>
#include <sys/time.h>
#include <pcap.h>
#include <signal.h>
#include <err.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "segnames.h"
#include "lin_packet.h"
#include "../bpf/parse_bpf.h"

int timer_flag = 0;
int nflag = 0;
int seg_size = 16*1024*1024; /* bytes */
int seg_time = 60; /* seconds */
int mfs_threshold = 32; /* megs */
int ufs_threshold = 1024; /* megs */
int interrupted = 0;

void
usage(char * prog) {

    fprintf(stderr,
"Usage: %s [-f file_prefix] [-s segment size] [-t segment time]\n"
"[-u ufs mount point] [-U ufs threshold] [-M mfs threshold] [-n] intf\n"
"Sniffs packets from interface intf, writing them to a file in /mfs.\n"
"This file is renamed to a unique name after it reaches the given segment\n"
"size, or after the given segment time, whichever comes first.  Also\n"
"monitors free space in mfs and ufs, and exits if thresholds are passed.\n"
"The -n flag turns off creation of new files; instead, the same file will\n"
"be repeatedly rewritten.\n",prog);
    exit(1);
}

void
segment_alarm_handler(int signum) {
    timer_flag = 1;
}

void
set_timer(int seg_time) {
    static struct itimerval itv = {{0,0},{0,0}};

    itv.it_interval.tv_sec = seg_time;
    itv.it_value.tv_sec = seg_time;
    if (setitimer(ITIMER_REAL,&itv,NULL) == -1)
	err(1,"unable to set timer");
}
 
void
shutdown(int signum) {
    printf("Received signal %d\n",signum);
    interrupted = 1;
} 

int
get_free_megs(char * path) {
    struct statfs statfsbuf;

    if (statfs(path, &statfsbuf) < 0)
	err(1,"get_free_megs statfs failure");
    return (long long int)statfsbuf.f_bavail * statfsbuf.f_bsize / (1024*1024);
}

int
main(int argc, char * argv[]) {
    int ch;
    char * intf = NULL;
    char * ufsMount = NULL;
    extern int optind;
    extern char * optarg;

    set_segment_name_prefix("");
    while ((ch = getopt(argc, argv, "M:U:f:i:s:t:u:n")) != -1) {
	switch (ch) {
	case 'M':
	    mfs_threshold = atoi(optarg);
	    break;
	case 'U':
	    ufs_threshold = atoi(optarg);
	    break;
	case 'f':
	    set_segment_name_prefix(optarg);
	    break;
	case 'i':
	    intf = optarg;
	    break;
	case 's':
	    seg_size = atoi(optarg);
	    break;
	case 't':
	    seg_time = atoi(optarg);
	    break;
	case 'u':
	    ufsMount = optarg;
	    break;
	case 'n':     /* don't create permanent MFS files */
	    ++nflag;
	    break;
	default:
	    usage(argv[0]);
	}
    }
    if (optind+1 == argc) {
	intf = argv[optind++];
    } else {
	usage(argv[0]);
    }

    setlinebuf(stdout);

    printf("segment: size=%d, fill time=%d\n", seg_size, seg_time);
    printf("threshold: MFS=%d MB, UFS=%d MB\n", mfs_threshold, ufs_threshold);
    printf("writing: %s\n", temp_segment_name);

    if (signal(SIGINT,(void*)&shutdown) <0)
	warnx("signal SIGINT");
    if (signal(SIGTERM,(void*)&shutdown) <0)
	warnx("signal SIGTERM");

    if (signal(SIGALRM,segment_alarm_handler) == SIG_ERR)
	err(1,"unable to set signal handler");
    set_timer(seg_time);

    init_intf(intf);
    init_intf_stats(intf);

    while (1) {
	FILE * temp_file;
	int mfs_free_megs = -1, ufs_free_megs = -1;
	int seg_size_sofar = 0;

	timer_flag = 0;
	if ((temp_file = fopen(temp_segment_name,"w")) == NULL)
	    err(1,"unable to open %s",temp_segment_name);
	while (!timer_flag) {
	    char bpf_contents[MAX_PACKET_SIZE];
	    struct bpf_hdr bpf;

	    intf_read(&bpf,bpf_contents);
	    /*XXX approximate: rewrite write_bpf to return bytes written,
	      to get more accurate total.  Or just decide to use packet sizes
	      instead of file sizes to decide when to make new segments? */
	    seg_size_sofar += bpf.bh_caplen + 20;
	    if (write_bpf(temp_file,&bpf,bpf_contents,1) == -1)
		errx(1,"error writing packet");
	    if (seg_size_sofar >= seg_size || interrupted) {
		set_timer(seg_time);
		break;
	    }
	}
	fclose(temp_file);

	make_new_segment_names();
	if (!nflag) {
	    if (link(temp_segment_name,current_segment_name))
		err(1,"error moving temporary segment");
	}
	if (unlink(temp_segment_name))
	    err(1,"error deleting temporary segment");
	if (!nflag)
	    write_Xfile(current_Xfile_name);

	update_intf_stats();
	print_intf_stats();
	mfs_free_megs = get_free_megs(current_segment_name);
	printf("mf: %dM ",mfs_free_megs);
	if (ufsMount) {
	    ufs_free_megs = get_free_megs(ufsMount);
	    printf("uf: %dM",ufs_free_megs);
	}
	putchar('\n');

	if (mfs_free_megs < mfs_threshold)
	    errx(100,"MFS free space %d MB, below threshold of %d MB\n",
		mfs_free_megs, mfs_threshold);
	if (ufsMount && ufs_free_megs < ufs_threshold)
	    errx(101,"UFS free space %d MB, below threshold of %d MB",
		ufs_free_megs, ufs_threshold);

	if (interrupted)
	    exit(102);

    }
}
