/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#ifdef LOCAL_BPF_HEADERS
#include "../sys/net/bpf.h"
#else
#include "/usr/src/sys/net/bpf.h"
#endif
#include "../bpf/parse_bpf.h"
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <pcap.h>
#include <unistd.h>
#include <err.h>
#include <signal.h>
#include "print.h"
#include "../crypto/crypto.h"
#ifndef __linux__
#include "sample.h"
#endif

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

#include "cache.h"

/* This is the length of the fixed-length part of the IP header: */
#define IP_HDR_SIZE 20

int cryptalg = DEFAULT_ALGORITHM;
int dFlag = 0;
int nFlag = 0;
int yFlag = 0;
int zFlag = 1;

#ifdef INSECURE
FILE *clear_file = NULL;
#endif

FILE *oFile;

struct crypto_session tr_key_session;
struct crypto_session vol_key_session;

int totPackets = 0;		/* # packets read from input (MFS) file */
int totLen = 0;			/* # bytes read  from input (MFS) file */
int totWritePackets = 0;	/* # packets written to output file */
int totWriteLen = 0;		/* # bytes written to output file */
int ipEncryptPackets = 0;	/* # packets encrypted to output file */
int totEncryptLen = 0;		/* # bytes encrypted to output file */
int ipUnEncryptPackets = 0;	/* # packets unencrypted to output file */
int totUnEncryptLen = 0;	/* # bytes unencrypted to output file */
int ipClearPackets = 0;		/* # unencrypted packets output to debug file */
int totClearLen = 0;		/* # unencrypted bytes output to debug file */

void usage(char * prog) {
  printf(
"usage: %s [options] filename\n",prog);
  printf(
"where filename is the file to read raw packet data from.\n"
"options:\n"
"-T file: read the translation table key from file\n"
"-V file: read the volume key from file\n"
"-a: supress most debugging messages\n"
"-b: use 0) linked list, 1) btree, or 2) hash for s/t tables\n"
#ifdef INSECURE
"-c file: write cleartext packets to file\n"
#endif
"-d: print debugging messages\n"
"-e algorithm: encrypt using algorithm (see below)\n"
"-n num: stop after processing num packets\n"
"-o file: output encrypted packet data to file\n"
"-s filename: output translation table to file\n"
"-t filename: output list of conversations to file\n"
"-w filename: input statistics file\n"
"-x filename: output statistics file\n"
"-y: don't cache key schedules\n"
"-z num: specify file format:\n"
"-z0: file format from prototype vault\n"
"-z1: conversation format\n"
"-z2: endpoint format\n"
"-z3: segment single-key format\n"
"The argument to -e is one of the following integers\n");
  print_algorithms();
  exit(1);
}

/*
** Read the X file from the listener.  It has important info
** like packets_dropped and bytes_dropped that we could not know.
** We process the listener's X file and write one of our own.
*/

int processXfiles(FILE *xin, FILE *xout, double dt)
{
	int processed;
	u_int start_sec, end_sec;
	u_int start_usec, end_usec;
	u_int lis_pkts, lis_pkts_drop;
	u_int64_t lis_bytes, lis_bytes_snap, lis_bytes_drop, lis_bytes_per_sec;

	if ( (processed =
	      fscanf(xin, "%u %u %u %u %u %u %lu %lu %lu %lu\n",
		     &start_sec, &end_sec,
		     &start_usec, &end_usec,
		     &lis_pkts, &lis_pkts_drop,
		     (long int *)&lis_bytes, (long int *)&lis_bytes_snap,
		     (long int *)&lis_bytes_drop, 
		     (long int *)&lis_bytes_per_sec)) != 10 )
	{
		fprintf(stderr,
			"Only got %d values processing listener's X file\n",
			processed);
		exit(1);
	}
	fclose(xin);

	fprintf(xout, "starttime:%u.%06u\n", start_sec, start_usec);
	fprintf(xout, "endtime:%u.%06u\n", end_sec,end_usec);
	fprintf(xout, "lis_pkts:%u\n", lis_pkts);
	fprintf(xout, "lis_pkts_drop:%u\n", lis_pkts_drop);
	fprintf(xout, "lis_bytes:%lu\n", (long int)lis_bytes);
	fprintf(xout, "lis_bytes_snap:%lu\n", (long int)lis_bytes_snap);
	fprintf(xout, "lis_bytes_drop:%lu\n", (long int)lis_bytes_drop);
	fprintf(xout, "lis_bytes_per_sec:%lu\n", (long int)lis_bytes_per_sec);
	fprintf(xout, "dmp_pkts:%d\n", totPackets);
	fprintf(xout, "dmp_pkts_written:%d\n", totWritePackets);
	fprintf(xout, "dmp_bytes:%d\n", totLen);
	fprintf(xout, "dmp_bytes_written:%d\n", totWriteLen);
	/* Add start, end stamps for dump? Or just float dt for now.*/
	fprintf(xout, "dmp_bytes_per_sec:%.0f\n", totWriteLen/dt);
	fclose(xout);
	return 0;
}

struct crypto_session * process_ip_packet( struct ip * ip )
{
        static struct crypto_session * conv_session = NULL;
	char * conv_key;
	in_addr_t fake_src, fake_dst;
  
	/* Look up, or assign new, substituted IP addresses: */
	fake_src = obscure_address(ip->ip_src.s_addr);
	fake_dst = obscure_address(ip->ip_dst.s_addr);

	if (yFlag) {
	  /* Since conversation keys aren't kept in the list of
	     conversations, we have to allocate space for them here
	     the first time: */
	  if (!conv_session) {
	    conv_session =
	      (struct crypto_session *)malloc(sizeof(struct crypto_session));
	  }
	  conv_key = make_conv_key(&vol_key_session, ip, cryptalg);
	  get_crypto_session(conv_session, cryptalg, CBC_MODE, ENCRYPT,
			     conv_key,0);
	} else {
	  /* Look up, or create a new entry in, the list of
	     conversations, and return the appropriate conversation
	     key; this may require generating a new conversation key. */
	  conv_session = get_key_schedule(fake_src,fake_dst,ip,
					  &vol_key_session);
	}

	/* Replace the IP addresses in the packet by the substituted IP
	   addresses: */
	ip->ip_src.s_addr = fake_src;
	ip->ip_dst.s_addr = fake_dst;
	return(conv_session);
}

/* Format of output file with -z0 (default):
     format id, cryptalg, iv, packet 1, packet 2,....
   where format id is an int equal to 0, cryptalg is an int, and the format
   of a packet is:
     plaintext bpf header with timestamp, lengths, etc. (18 bytes),
     encrypted ethernet header + first two bytes of IP header (16 bytes),
     remaining fixed-length part of IP header in plaintext (18 bytes),
     remaining IP packet (with padding if necessary for the encryption
     algorithm)
   (We encrypt the first two bytes of the IP header to avoid having to
     pad the ethernet header out to the block length of the encryption
     algorithm.)
 */

/* Format of output file with -z1 option:
     format id, cryptalg, iv, packet 1, packet 2
   where format id is an int equal to 1, cryptalg is an int, and the format
   of a packet is:
     length (number of bytes to the end of the packet data, NOT including the
       padding added for encryption), in plaintext (2 bytes),
     the conversation key encrypted with the translation table key (16 bytes
       in the case of AES)
     the following, encrypted together with the conversation key:
       timestamp, taken from the original bpf header (8 bytes)
       ethernet header, IP header, etc., from original bpf contents, with
         padding as necessary for the encryption algorithm.
*/

/* Format of output file with -z2 option:
   Same as above, but the format id is 2, and after the length field that
   begins each mangled packet, there are two additional fields:
     the conversation key encrypted with a key derived from the source address
       (16 bytes in the case of AES)
     the conversation key encrypted with a key derived from the destination
       address (16 bytes, for AES)
*/

/* Format with -z3 option:
     format id, iv, file_length, data
   where foramt id is 3, and file_length is the length of the unpadded segment.
   Packet data is written (in bpf format) exactly as it was read.
*/

void bpf_dump_File( FILE * in )
{
	struct bpf_hdr * bpf;
	char * bpf_content;
	char * buffer;
	int padded_length, padded_ip_length;
	struct tm * tm;
	u_short ether_type;
	char * iv;
	struct ip * ip_packet;
	char * conv_key, * src_key, * dst_key;
	struct crypto_session * conv_session;
	struct crypto_session src_session, dst_session;
	int blocklen, keylen, padded_keylen;
	char * buffer_ptr, * encrypt_start, * pkt_end_ptr;
	short int * length_ptr;
	int i;
#ifdef INSECURE
	struct pcap_pkthdr pkt_hdr;

	if (clear_file) {
	  if (!write_pcap_header(clear_file,0)) {
	    warnx("error writing unencrypted file header");
	  }
	}
#endif

	if (zFlag) {
	  /* In this case, the conversation keys aren't cached in the list
	     of conversations, so we have to make space for a conversation
	     key here: */
	  if (!(conv_session = malloc(sizeof(*conv_session)))) {
	    perror("Unable to malloc space for conversation key session");
	    exit(1);
	  }
	  /* We don't use the bpf header without a lot of mangling, so it
	     needs its own space separate from the main buffer: */
	  if (!(bpf = (struct bpf_hdr *)malloc(sizeof(*bpf)))) {
	    perror("Unable to malloc space for bpf header");
	    exit(1);
	  }
	}

	/* Write crypto file header: */
	if (oFile) {
	  if (fwrite(&zFlag,sizeof(zFlag),1,oFile) !=1 ) {
	    perror("error writing file format to encrypted file");
	    exit(1);
	  }
	  if (fwrite(&cryptalg,sizeof(cryptalg),1,oFile) != 1) {
	    perror("error writing cryptalg to encrypted file");
	    exit(1);
	  }
	}
	
	blocklen = algorithms[cryptalg].block_length;
	keylen = algorithms[cryptalg].key_length;
	padded_keylen = get_encrypted_length(cryptalg,keylen);

	/* get a random iv */
	iv = (char *)malloc(blocklen);
	for (i = 0; i < blocklen; i += sizeof(long))
		*(long *)(iv + i) = random();
	
	if (fwrite(iv,1,blocklen,oFile) != blocklen) {
	  perror("error writing iv to file");
	  exit(1);
	}

	if (!(buffer = (char *)malloc(MAX_PACKET_SIZE))) {
	  perror("Unable to malloc space for buffer");
	  exit(1);
	}

	/* Set bpf_content to point to where we want the packet contents to
	   be put: */
	switch (zFlag) {
	case 0:
	  bpf = (struct bpf_hdr *)buffer;
	  bpf_content = buffer + BPF_HDR_SIZE;
	  break;
	case 1:
	  bpf_content = buffer + sizeof(short int) + padded_keylen
	    + sizeof(struct timeval);
	  break;
	case 2:
	  bpf_content = buffer + sizeof(short int) + 3*padded_keylen
	    + sizeof(struct timeval);
	  break;
	default:
	  printf("unknown argument to -z flag\n");
	  exit(1);
	}
  
	/* Process packets while there are more to process. */
	while (!read_bpf(in,bpf,bpf_content,1)) {

	  if (nFlag && (totPackets >= nFlag))
	    break;
	  totPackets++;
	  totLen += bpf->bh_caplen;

	  if (dFlag) {
	    tm = localtime((time_t *)&bpf->bh_tstamp.tv_sec);
	    printf("\nProcessing packet number %d\n", totPackets);
	    printf( "%.24s.%ld: %d:%d:%d\n",
		    asctime(tm), bpf->bh_tstamp.tv_usec,
		    bpf->bh_caplen,
		    bpf->bh_datalen,
		    bpf->bh_hdrlen );
	    /* sample() prints some stats. */
#ifndef __linux__
	    sample((struct packet *)bpf_content,
		   BPF_HDR_SIZE+bpf->bh_caplen);
#endif
	  }

	  ether_type = ntohs(*(u_short *)(bpf_content
					  + ETHER_HDR_LEN - 2));

	  /* PROBLEM: Note we don't parse IEEE 802.2/802.3
	     packets correctly; see RFC 1042.*/
	  if (ether_type == ETHERTYPE_IP) {
	    ip_packet = (struct ip *)(bpf_content + ETHER_HDR_LEN);

#ifdef INSECURE
	    if (clear_file) {
	      pkt_hdr.ts.tv_sec = bpf->bh_tstamp.tv_sec;
	      pkt_hdr.ts.tv_usec = bpf->bh_tstamp.tv_usec;
	      pkt_hdr.len = bpf->bh_datalen;
	      pkt_hdr.caplen = bpf->bh_caplen;
	      if (fwrite(&pkt_hdr,sizeof(pkt_hdr),1,clear_file) != 1)
		warnx("error writing unencrypted file packet header\n");
	      if (fwrite(bpf_content,bpf->bh_caplen,1,
			 clear_file) != 1)
		warnx("error writing unencrypted file packet data\n");
	      ++ipClearPackets;
	      totClearLen += bpf->bh_caplen;
	    }
#endif
	    if (zFlag) { /* write -z1/-z2 file formats: */
	      buffer_ptr = buffer;
	      length_ptr = (short int *)buffer_ptr;
	      buffer_ptr += sizeof(short int);
	      
	      conv_key = make_conv_key(&vol_key_session,
				       ip_packet,cryptalg);
	      get_crypto_session(conv_session,cryptalg,
				 CBC_MODE,ENCRYPT,conv_key,0);

	      if (zFlag == 2) {
		src_key = make_src_key(&vol_key_session,
				       ip_packet, cryptalg);
		get_crypto_session(&src_session,cryptalg,
				   CBC_MODE,ENCRYPT,src_key,0);
		memcpy(buffer_ptr,conv_key,keylen);
		crypt_buffer(&src_session,buffer_ptr,keylen,iv);
		buffer_ptr += padded_keylen;

		dst_key = make_dst_key(&vol_key_session,
				       ip_packet, cryptalg);
		get_crypto_session(&dst_session,cryptalg,
				   CBC_MODE,ENCRYPT,dst_key,0);
		memcpy(buffer_ptr,conv_key,keylen);
		crypt_buffer(&dst_session,buffer_ptr,keylen,iv);
		buffer_ptr += padded_keylen;
	      }

	      memcpy(buffer_ptr,conv_key,keylen);
	      crypt_buffer(&tr_key_session,buffer_ptr,keylen,iv);
	      buffer_ptr += padded_keylen;

	      encrypt_start = buffer_ptr;
	      memcpy(buffer_ptr,&bpf->bh_tstamp,
		     sizeof(bpf->bh_tstamp));
	      buffer_ptr += sizeof(bpf->bh_tstamp);
	      
	      if (buffer_ptr != bpf_content) {
		printf("bpf_content - buffer_ptr = %d\n",
		       bpf_content-buffer_ptr);
		exit(1);
	      }
	      
	      pkt_end_ptr = bpf_content+bpf->bh_caplen;
	      padded_length =
		crypt_buffer(conv_session,encrypt_start,
			     pkt_end_ptr-encrypt_start,
			     iv);
	      buffer_ptr = encrypt_start + padded_length;
	      *length_ptr = pkt_end_ptr-(buffer+sizeof(short int));
	      
	      ++ipEncryptPackets;
	      totEncryptLen += buffer_ptr-encrypt_start;
	      /* What is totEncryptLen really supposed to measure? */
	      
	      if (oFile)
		{
		  PRINT(("Writing %d bytes for this packet\n",
			 buffer_ptr-buffer));
		  if (fwrite(buffer,1,buffer_ptr-buffer,oFile) 
		      != buffer_ptr-buffer)
		    errx(1, "write failed");
		  ++totWritePackets;
		  totWriteLen += buffer_ptr-buffer;
		}
	      
	    } else { /* zFlag == 0; the original file format: */

	      /* substitute IP addresses, update lists,
		       generate conversation key, etc.: */
	      conv_session = process_ip_packet(ip_packet);
	      
	      ++ipEncryptPackets;
	      totEncryptLen += bpf->bh_caplen;
	      
	      crypt_buffer(conv_session,bpf_content,ETHER_HDR_LEN,iv);

	      padded_ip_length =
		crypt_buffer(conv_session,
			     (char *)ip_packet+IP_HDR_SIZE,
			     bpf->bh_caplen-ETHER_HDR_LEN-IP_HDR_SIZE,iv);
	      padded_length = padded_ip_length
		+ETHER_HDR_LEN+IP_HDR_SIZE;

	      if (oFile)
		{
		  PRINT(("Writing %d bytes for this packet\n",
			 BPF_HDR_SIZE+padded_length));
		  /* bpf->bh_datalen is left alone, so it can
		     still be used by the decrypter to easily
		     determine the end of the packet data. */
		  bpf->bh_caplen = padded_length;
		  if (write_bpf(oFile,bpf,bpf_content,0))
		    errx(1, "write failed");
		  ++totWritePackets;
		  totWriteLen += BPF_HDR_SIZE+bpf->bh_caplen;
		}
	    }
	  }
	} 
	free(iv);
	free(buffer);
	printf("\nEnd of packet file\n");
}

void read_file(FILE * infile, char ** buffer, int * length) {
#define READ_INC 1024
  int read;

  *length = 0;
  if ((*buffer = malloc(READ_INC)) == NULL)
    errx(1,"unable to malloc space for read buffer");
  while ((read = fread(*buffer+*length,1,READ_INC,infile)) 
	 == READ_INC) {
    *length += read;
    if ((*buffer = realloc(*buffer,*length+READ_INC)) == NULL)
      errx(1,"unable to realloc space for read buffer");
  }
  if (!feof(infile)) {
    errx(1,"error reading from infile");
  } else {
    *length += read;
  }
}

int main( int argc, char * argv[] )
{
        struct timeval start, stop;
	double dt;
	struct tm *tm_p;
	int bFlag = 0;
	char * oName = NULL;
	char * vKeyFile = 0;
	char * tKeyFile = 0;
	char * sName = NULL;
	char * tName = NULL;
	FILE * xinFile = NULL;
	FILE * xoutFile = NULL;
	FILE * infile = NULL;
	char * infile_name = NULL;
	char key_translate_key[MAX_KEY_LENGTH];
	char key_volume_key[MAX_KEY_LENGTH];
	char * buffer;
	int length;
	extern char * optarg;
	extern int optind;
	char optch;

#if 0
	if (setlinebuf(stdout) == 0)
		warnx("setlinebuf() succeeded");
	else
		warnx("setlinebuf() FAILED");
#endif

	/*
	 * Ignore SIGINT to prevent pkt_dump exit until it has finished
	 * encrypting the current segment.
	 */
	if (signal(SIGINT, SIG_IGN) == SIG_ERR)
		err(1, "can't ignore SIGINT");

	gettimeofday(&start,NULL);

	/* random() is used for random numbers in a number of places; we
	   initialize the generator here so any functions we call from now
	   on will get decent random numbers from random(): */
#ifdef __linux__
	/* This is bad, replace it. */
	srandom(start.tv_usec);
#else
	srandomdev();
#endif

	if (argc == 1)
	  usage(argv[0]);

	while ((optch = 
		getopt(argc,argv,"V:T:b:c:de:n:o:s:t:w:x:yz:")) != -1) {
	  switch (optch) {
	  case 'T':		 /* file containing translation key */
	    tKeyFile = optarg;
	    break;
	  case 'V':		 /* file containing volume key */
	    vKeyFile = optarg;
	    break;
	  case 'b':               /* choose list, btree, hash for s/t tables */
	    bFlag = atoi(optarg);
	    break;
#ifdef INSECURE
	  case 'c':               /* cleartext packet file */
	    if ((clear_file = fopen(optarg,"w")) == NULL)
	      err(1, "failed to open %s",optarg);
	    break;
#endif
	  case 'd':		 /* print debugging messages */
	    dFlag++;
	    break;
	  case 'e':		 /* encryption algorithm */
	    cryptalg = atoi(optarg);
	    break;
	  case 'n':		 /* process only n packets */
	    nFlag = atoi(optarg);
	    break;
	  case 'o':		 /* output file */
	    oName = optarg;
	    if ((oFile = fopen( oName, "w+" )) == NULL ) {
	      perror( oName );
	      exit(1);
	    }
	    break;
	  case 's':		 /* substitution table */
	    sName = optarg;
	    break;
	  case 't':		 /* conversation list */
	    tName = optarg;
	    break;
	  case 'w':		 /* input statistics file */
	    if ((xinFile = fopen(optarg, "r")) == NULL )
	      errx(1,"unable to open input x file %s",optarg);
	    break;
	  case 'x':		 /* output statistics file */
	    if ((xoutFile = fopen(optarg, "w+")) == NULL )
	      errx(1,"unable to open output x file %s",optarg);
	    break;
	  case 'y':
	    yFlag++;
	    break;
	  case 'z':
	    zFlag = atoi(optarg);
	    break;
	  case '?':
	  default:
	    usage(argv[0]);
	  }
	}
	if (optind + 1 == argc) {
	  infile_name = argv[optind];
	  printf( "Reading %s\n", infile_name );
	  if ((infile = fopen(infile_name,"r")) == NULL)
	    errx(1,"unable to open %s",infile_name);
	} else {
	  usage(argv[0]);
	}
	init_substitution_table(bFlag);
	init_key_schedule_cache(bFlag);

	read_key_file(vKeyFile, key_volume_key, cryptalg);
	/* key schedule for conversation key generation */
	get_crypto_session(&vol_key_session,cryptalg,
			   CBC_MODE,ENCRYPT,
			   key_volume_key,0);

	read_key_file(tKeyFile, key_translate_key, cryptalg);
	/* key schedule for substitution table encryption */
	get_crypto_session(&tr_key_session,cryptalg,
			   CBC_MODE,ENCRYPT,
			   key_translate_key,0);

	if (zFlag == 3) {
	  char * seg_key;
	  struct crypto_session seg_session;
	  char * buffer;
	  int length;
	  
	  seg_key = make_segment_key(&vol_key_session,infile_name,cryptalg);
	  get_crypto_session(&seg_session,cryptalg,CBC_MODE,ENCRYPT,seg_key,0);
	  if (fwrite(&zFlag,sizeof(zFlag),1,oFile) !=1 )
	    errx(1,"error writing file format to encrypted file");
	  if (fwrite(&cryptalg,sizeof(cryptalg),1,oFile) != 1)
	    errx(1,"error writing cryptalg to encrypted file");
	  read_file(infile,&buffer,&length);
	  printf("encrypting entire segment of length %d\n",length);
	  totLen = totWriteLen = length;
	  write_cbc_encrypted_buffer(&seg_session,oFile,oName,
				     buffer,length);
	} else {
	  bpf_dump_File(infile);
	}
	if (infile)
	  fclose(infile);

	if (dFlag) {
	  printf("\nPacket statistics:\n");
#ifndef __linux__
	  sample_dmp();
#endif
	  print_substitution_table();
	  print_key_schedule_cache();
	}

	if (oFile)
	  fclose(oFile);

#ifdef INSECURE
	if (clear_file)
	  fclose(clear_file);
#endif

	/* We write two addional files:
	   First a file with ip and substituted ip address.
	   Then a file with "conversations" (ip/ip pairs).
	   Each file is encrypted with the translation table key.
	   In the case of -z1 or -z2, these files are unused.  (They're
	   still written, but are very short.)
	*/

	if (sName) {
	  get_substitution_table(totWritePackets,&buffer,&length);
	  save_cbc_encrypted_buffer(&tr_key_session,sName,buffer,length);
	  free(buffer);
	}
	if (tName) {
	  get_conversation_list(totWritePackets,&buffer,&length);
	  save_cbc_encrypted_buffer(&tr_key_session,tName,buffer,length);
	  free(buffer);
	}

	gettimeofday(&stop,NULL);
	tm_p = localtime((time_t *)&stop.tv_sec);
	dt = (stop.tv_sec - start.tv_sec)
	  + (stop.tv_usec - start.tv_usec)/1000000.0;
	if (dt == 0)
	  dt = 1/1000000.0;
	printf("pkt_dump: %02d/%02d/%02d %02d:%02d:%.3f dt:%.3f\n"
	       "total read:\n"
	       "packets:%d data:%d\n"
	       "total written:\n"
	       "packets:%d data:%d byte/s:%.02f mbps:%.02f\n"
	       "encrypted packets:%d data:%d\n"
	       "unencrypted packets:%d data:%d (should be zero)\n"
	       "cleartext packets:%d data:%d (should be zero unless debugging)\n"
	       "(using %s structure)\n",
	    tm_p->tm_mon+1,
	    tm_p->tm_mday,
	    tm_p->tm_year-100,
	    tm_p->tm_hour,
	    tm_p->tm_min,
	    tm_p->tm_sec + (stop.tv_usec/1000000.0),
	    dt,
	    totPackets,
	    totLen,
	    totWritePackets,
	    totWriteLen,
            totWriteLen/dt,
            totWriteLen/dt*8/(1024*1024),
	    ipEncryptPackets,
	    totEncryptLen,
	    ipUnEncryptPackets,
	    totUnEncryptLen,
	    ipClearPackets,
	    totClearLen,
	    bFlag == 2 ? "hash" : bFlag == 1 ? "btree" : "list");

	/* Write a third file to keep track of statistics */
	if ((xinFile) && (xoutFile))
	  processXfiles( xinFile, xoutFile, dt );
	exit(0);
}
