/*
 * COPYRIGHT    2002
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <err.h>
#include "../crypto/crypto.h"
#include "hash.h"

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

/* Size of the hash tables: (The code below will break if the
   following constants are not powers of two.) Note this is a hard upper
   bound on the number of elements we can be store.  These values should
   be sufficient for any segment produced as long as segsize is 16MB.*/
#define SUBS_SIZE 524288
#define KEYCACHE_SIZE 262144

/* increments by which to allocate memory for data stored in hash tables: */
#define INIT_SUBS 131072
#define INIT_KEYCACHE 65536

extern int cryptalg;

/*
  We use these memory pools for storing actual data.
*/

struct mempool {
  int total_blocks;
  int next_free_block;
  int block_size;
  struct slab * slab_head;
};

struct slab {
  void * data;
  struct slab * last;
};

void * get_pool(int start_blocks, int block_size) {
  struct mempool * new;

  if ((new = (void *)malloc(sizeof(*new))) == NULL)
    errx(1,"unable to allocate memory for memory pool");
  new->block_size = block_size;
  new->total_blocks = start_blocks;
  new->next_free_block = 0;
  if ((new->slab_head = malloc(sizeof(struct slab))) == NULL)
    errx(1,"unable to allocate memory for slab_head");
  if ((new->slab_head->data = malloc(start_blocks * block_size)) == NULL)
    errx(1,"unable to allocate initial %d %d-byte blocks for memory pool",
	 start_blocks,block_size);
  new->slab_head->last = NULL;
  return(new);
}

void destroy_pool(struct mempool * this) {
  struct slab * next_slab, * this_slab;

  this_slab = this->slab_head;
  while (this_slab != NULL) {
    next_slab = this_slab->last;
    free(this_slab->data);
    free(this_slab);
    this_slab = next_slab;
  }
  free(this);
}

void * get_block(struct mempool * this) {
  struct slab * old_slab;

  if (this->next_free_block == this->total_blocks) {
    old_slab = this->slab_head;
    if ((this->slab_head = malloc(sizeof(struct slab))) == NULL)
      errx(1,"unable to allocate memory for slab_head");
    if ((this->slab_head->data = malloc(this->total_blocks * this->block_size))
	== NULL)
      errx(1,"unable to allocate additional %d %d-byte blocks for memory pool",
	   this->total_blocks,this->block_size);
    this->slab_head->last = old_slab;
    this->next_free_block = 0;
  }
  return(this->slab_head->data + this->block_size*(this->next_free_block++));
}

/*
  hash table stuff:
*/

struct hash_table_element {
  int next;
  void * data;
};

struct hash_table {
  int (* cmp)(void * left, void * right); /* returns sign(left-right) */
  int (* f)(struct hash_table * this, void * data);
  int table_size;
  int elements;
  int occupied_top; /* All indices >= occupied_top are occupied. */
  struct hash_table_element * table;
};

struct hash_table * hash_init(int size) {
  struct hash_table * new;

  if ((new = calloc(1,sizeof(*new))) == NULL)
    errx(1,"unable to allocate memory for new hash_table struct");
  new->elements = 0;
  new->table_size = size;
  new->occupied_top = size; /* 1 higher than the top index */
  if ((new->table = calloc(size, sizeof(struct hash_table_element))) == NULL)
      errx(1,"unable to allocate memory for hash table");
  return(new);
}

/* Note that managing the memory that the actual data is stored in is the
   caller's responsibility. */
void destroy_hash_table(struct hash_table * this) {

  free(this->table);
  free(this);
}

void * hash_lookup(struct hash_table * this, void * data) {
  int index;

  index = this->f(this,data);
  if (this->table[index].data == NULL)
    return(NULL);
  for (; index != -1; index=this->table[index].next)
    if (this->cmp(this->table[index].data,data) == 0)
      return(this->table[index].data);
  return(NULL);
}

int hash_find_free_entry(struct hash_table * this) {
  if (this->occupied_top == 0)
    errx(1,"hash table overflow");
  this->occupied_top--;
  while (this->table[this->occupied_top].data != NULL)
    if (this->occupied_top == 0)
      errx(1,"Hash table overflow");
    else
      this->occupied_top--;
  return(this->occupied_top);
}

int hash_lookup_or_store(struct hash_table * this, void * key, void *** data) {
  int index, new;

  index = this->f(this,key);
  if (this->table[index].data == NULL) {
    *data = &this->table[index].data;
    this->table[index].next = -1;
    this->elements++;
    return(0);
  }
  while (1) {
    if (this->cmp(this->table[index].data,key) == 0) {
      *data = &this->table[index].data;
      return(1); /* success */
    }
    if (this->table[index].next == -1)
      break;
    index = this->table[index].next;
  }
  new = hash_find_free_entry(this);
  this->table[index].next = new;
  *data = &this->table[new].data;
  this->table[new].next = -1;
  this->elements++;
  return(0);
}

void hash_store(struct hash_table * this, void * data) {
  int index, new;

  index = this->f(this,data);
  if (this->table[index].data == NULL) {
    this->table[index].data = data;
    this->table[index].next = -1;
  } else {
    while (index != -1)
      index = this->table[index].next;
    /* Search for unoccupied spot in table: */
    new = hash_find_free_entry(this);
    this->table[index].next = new;
    this->table[new].data = data;
    this->table[new].next = -1;
  }
  this->elements++;
}

typedef void hash_element_handler(struct hash_table_element * this, void * user);

/* walks through each element of the hash, calling handle_element as it
   goes. */
void hash_walk(struct hash_table * this, hash_element_handler handle_element,
		 void * user) {
  int i;

  if (this->elements == 0)
      return;
  for (i = 0; i<this->table_size; i++) {
    if (this->table[i].data != NULL)
	handle_element(&this->table[i],user);
  }
}

/*
  substitution- and keycache- table specific stuff:
*/

struct hash {
    struct hash_table * table;
    struct mempool * data_pool;
};

struct substitution {
  in_addr_t real_addr;
  in_addr_t fake_addr;
};

struct conversation {
  in_addr_t fake_src;
  in_addr_t fake_dst;
  struct crypto_session session;
};

int cmp(int left, int right) {
  if (left<right)
    return -1;
  else if (left>right)
    return 1;
  else /* left==right */
    return 0;
}

int subs_cmp_by_real(void * left, void * right) {

  return(cmp(((struct substitution *)left)->real_addr,
	     ((struct substitution *)right)->real_addr));
}

int conv_cmp(void * left, void * right) {
  struct conversation * leftc, * rightc;

  leftc = (struct conversation *)left;
  rightc = (struct conversation *)right;
  if (cmp(leftc->fake_src,rightc->fake_src))
    return cmp(leftc->fake_src,rightc->fake_src);
  else
    return cmp(leftc->fake_dst,rightc->fake_dst);
}

/* log_2 of a power of 2: */
int stupid_log2(int n) {
  int i;

  for (i=-1; n>0; i++)
    n >>= 1;
  return(i);
}

int f_subs(struct hash_table * this, void * data) {
  static unsigned int m = 0;
  static int b;

  if (!m) {
    m = random() | 0x40000001U; /* a random odd number between 
				   2^30 and 2^31-1 */
    b = 8*sizeof(int) - stupid_log2(SUBS_SIZE);
  }
  return( ( ((struct substitution *)data)->real_addr * m ) >> b);
}

int f_keycache(struct hash_table * this, void * data) {
  static unsigned int m = 0;
  static unsigned int n = 0;
  static int b;

  if (!m) {
    m = random() | 0x40000001U;
    n = random() | 0x40000001U;
    b = 8*sizeof(int) - stupid_log2(KEYCACHE_SIZE);
  }
  return( ( ( ((struct conversation *)data)->fake_src * m ) ^
	    ( ((struct conversation *)data)->fake_dst * n )  ) >> b);
}

struct hash *
hash_init_substitution_table() {
  struct hash * new;

  if ((new = malloc(sizeof(*new))) == NULL)
      err(1,"unable to allocate memory for new substitution table");
  new->table = hash_init(SUBS_SIZE);
  new->table->cmp = subs_cmp_by_real;
  new->table->f = f_subs;
  new->data_pool = get_pool(INIT_SUBS,sizeof(struct substitution));
  return(new);
}

struct hash *
hash_init_key_schedule_cache() {
  struct hash * new;

  if ((new = malloc(sizeof(*new))) == NULL)
      err(1,"unable to allocate memory for new key schedule cache");
  new->table = hash_init(KEYCACHE_SIZE);
  new->table->cmp = conv_cmp;
  new->table->f = f_keycache;
  new->data_pool = get_pool(INIT_KEYCACHE,sizeof(struct conversation));
  return(new);
}

in_addr_t hash_obscure_address(struct hash * subs_table,
			       in_addr_t real_address) {
  struct substitution key;
  struct substitution ** new_substitution_ptr;

  key.real_addr = real_address;
  if (!hash_lookup_or_store(subs_table->table,&key,
			    (void ***)&new_substitution_ptr)) {
    *new_substitution_ptr = get_block(subs_table->data_pool);
    (*new_substitution_ptr)->real_addr = real_address;
    (*new_substitution_ptr)->fake_addr = subs_table->table->elements;
  }
  return((*new_substitution_ptr)->fake_addr);
}

int hash_reveal_address(struct hash * subs_table,
			in_addr_t * fake_address) {

  errx(1,"hash_reveal_address not implemented");
}

void hash_insert_substitution(struct hash * subs_table,
			      in_addr_t real_address, in_addr_t fake_address) {

  errx(1,"hash_insert_substitution not implemented");
}

struct crypto_session * 
hash_get_key_schedule(struct hash * keycache_table,
		      in_addr_t fake_src,
		      in_addr_t fake_dst,
		      struct ip * ip,
		      struct crypto_session * vol_key_session) {
  struct conversation ** new_conversation_ptr;
  struct conversation key;
  char * conv_key;

  key.fake_src = fake_src;
  key.fake_dst = fake_dst;
  if (!hash_lookup_or_store(keycache_table->table,&key,
			    (void ***)&new_conversation_ptr)) {
    *new_conversation_ptr = get_block(keycache_table->data_pool);
    (*new_conversation_ptr)->fake_src = fake_src;
    (*new_conversation_ptr)->fake_dst = fake_dst;
    conv_key = make_conv_key(vol_key_session, ip, cryptalg);
    get_crypto_session(&(*new_conversation_ptr)->session,
		       cryptalg,CBC_MODE,ENCRYPT,conv_key,1);
  }
  return(&(*new_conversation_ptr)->session);
}

struct sized_buffer {
  char * buffer;
  int length;
};

void copy_subs_to_buffer(struct hash_table_element * this, void * user) {
  struct sized_buffer * sbuff;
  struct substitution * this_subs;

  sbuff = (struct sized_buffer *)user;
  this_subs = (struct substitution *)this->data;
  memcpy(sbuff->buffer+sbuff->length,&this_subs->real_addr,sizeof(in_addr_t));
  sbuff->length += sizeof(in_addr_t);
  memcpy(sbuff->buffer+sbuff->length,&this_subs->fake_addr,sizeof(in_addr_t));
  sbuff->length += sizeof(in_addr_t);
}

void
hash_destroy(struct hash * this) {

    destroy_pool(this->data_pool);
    destroy_hash_table(this->table);
    free(this);
}

void
hash_get_substitution_table(struct hash * subs_table,
			    int max_table_size, char ** buffer,
			    int * length) {
  struct sized_buffer sbuff;

  if ((*buffer = malloc(subs_table->table->elements*2*sizeof(in_addr_t) 
			+ MAX_BLOCK_LENGTH)) == NULL)
    errx(1,"unable to allocate memory in hash_get_substitution_table");
  sbuff.buffer = *buffer;
  sbuff.length = 0;
  hash_walk(subs_table->table,copy_subs_to_buffer,&sbuff);
  *length = sbuff.length;
  hash_destroy(subs_table);
}

void copy_conv_to_buffer(struct hash_table_element * this, void * user) {
  struct sized_buffer * sbuff;
  struct conversation * this_conv;

  sbuff = (struct sized_buffer *)user;
  this_conv = (struct conversation *)this->data;
  memcpy(sbuff->buffer+sbuff->length,&this_conv->fake_src,sizeof(in_addr_t));
  sbuff->length += sizeof(in_addr_t);
  memcpy(sbuff->buffer+sbuff->length,&this_conv->fake_dst,sizeof(in_addr_t));
  sbuff->length += sizeof(in_addr_t);
}

void
hash_get_conversation_list(struct hash * keycache_table,
				int max_table_size, char ** buffer,
				int * length) {
  struct sized_buffer sbuff;

  if ((*buffer = malloc(keycache_table->table->elements*2*sizeof(in_addr_t)
			+ MAX_BLOCK_LENGTH)) == NULL)
    errx(1,"unable to allocate memory in hash_get-conversation_list");
  sbuff.buffer = *buffer;
  sbuff.length = 0;
  hash_walk(keycache_table->table,copy_conv_to_buffer,&sbuff);
  *length = sbuff.length;
  hash_destroy(keycache_table);
}

void
print_substitution(struct hash_table_element * this, void * user) {
  struct substitution * this_subs;

  this_subs = (struct substitution *)this->data;
  printf("real: %s; ", inet_ntoa(*(struct in_addr *)&this_subs->real_addr));
  printf("fake: %s\n", inet_ntoa(*(struct in_addr *)&this_subs->fake_addr));
}

void print_conversation(struct hash_table_element * this, void * user) {
  struct conversation * this_conv;

  this_conv = (struct conversation *)this->data;
  printf("fake_src: %s; ",inet_ntoa(*(struct in_addr *)&this_conv->fake_src));
  printf("fake_dst: %s; ",inet_ntoa(*(struct in_addr *)&this_conv->fake_dst));
  printf("session: not printed\n");
}

void hash_print_substitution_table(struct hash * subs_table) {

  hash_walk(subs_table->table,print_substitution,NULL);
}

void hash_print_key_schedule_cache(struct hash * keycache_table) {

  hash_walk(keycache_table->table,print_conversation,NULL);
}

int hash_get_substitution_table_size(struct hash * subs_table) {
    return subs_table->table->elements;
}
