/*
 * COPYRIGHT    2002
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <err.h>
#include "../crypto/crypto.h"

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

#include "btree.h"
#include "list.h"
#include "hash.h"

/* The list of real IP/substituted IP addresses: */
static struct list * ipList;
static DB *ipDB;
static struct hash * subs_table;
static int subs_type;

/* The key cache / list of conversations: */
static struct list * ipPairList;
static DB *ipPairDB;
static struct hash * keycache_table;
static int keycache_type;

void init_substitution_table(int type) {

  subs_type = type;
  switch (subs_type) {
  case 0:
    ipList = newList(0);
    break;
  case 1:
    ipDB = newDB("btree.s");
    break;
  case 2:
    subs_table = hash_init_substitution_table();
    break;
  default:
    errx(1,"unknown type in init_substitution_table");
  }
}

void init_key_schedule_cache(int type) {

  keycache_type = type;
  switch (keycache_type) {
  case 0:
    ipPairList = newList(0);
    break;
  case 1:
    ipPairDB = newDB("btree.t");
    break;
  case 2:
    keycache_table = hash_init_key_schedule_cache();
    break;
  default:
    errx(1,"unknown type in init_key_schedule_cache");
  }
}

in_addr_t obscure_address(in_addr_t real_address) {
  switch (subs_type) {
  case 0:
    return new_substitution(ipList,real_address);
  case 1:
    return newDB_substitution(ipDB,real_address);
  case 2:
    return hash_obscure_address(subs_table,real_address);
  default:
    errx(1,"reveal_address not implemented for type %d",subs_type);
  }
}

int reveal_address(in_addr_t * fake_address) {
  switch (subs_type) {
  case 0:
    return find_real_ip(ipList,fake_address);
  case 1:
    return findDB_real_ip(ipDB,fake_address);
  default:
    errx(1,"reveal_address not implemented for type %d",subs_type);
  }
}

void insert_substitution(in_addr_t real_address, in_addr_t fake_address) {
  switch (subs_type) {
  case 0:
    list_insert_substitution(ipList,real_address,fake_address);
    break;
  case 1:
    insertDB_substitution(ipDB,real_address,fake_address);
    break;
  default:
    errx(1,"insert_substitution not implemented for type %d",subs_type);
  }
}

struct crypto_session * 
get_key_schedule(in_addr_t fake_src,
		 in_addr_t fake_dst,
		 struct ip * ip,		 
		 struct crypto_session * vol_key_session) {
  switch (keycache_type) {
  case 0:
    return new_conversation(ipPairList,fake_src,fake_dst,ip,vol_key_session);
  case 1:
    return newDB_conversation(ipPairDB,fake_src,fake_dst,ip,vol_key_session);
  case 2:
    return hash_get_key_schedule(keycache_table,
				 fake_src,fake_dst,ip,vol_key_session);
  default:
    errx(1,"get_key_schedule not implemented for type %d",keycache_type);
  }
}

void get_substitution_table(int max_table_size, char ** buffer, 
			    int * length) {

  switch (subs_type) {
  case 0:
    list_get_substitution_table(ipList,max_table_size,buffer,length);
    break;
  case 1:
    if (getDB_substitution_table(ipDB,max_table_size,buffer,length) < 0)
      errx(1,"Error retrieving substitution table from database");
    break;
  case 2:
    hash_get_substitution_table(subs_table,max_table_size,buffer,length);
    break;
  default:
    errx(1,"get_substitution_table not implemented for type %d",subs_type);
  }
}

void get_conversation_list(int max_table_size, char ** buffer,
			   int * length) {

  switch (keycache_type) {
  case 0:
    list_get_conversation_list(ipPairList,max_table_size,buffer,length);
    break;
  case 1:
    if (getDB_conversation_list(ipPairDB,max_table_size,buffer,length) < 0)
      errx(1,"Error retrieving conversation list from database");
    break;
  case 2:
    hash_get_conversation_list(keycache_table,max_table_size,buffer,length);
    break;
  default:
    errx(1,"get_key_schedule_cache not implemented for type %d",keycache_type);
  }
}

void print_substitution_table() {

  printf("\ns table (list of real/substituted addresses):\n");
  switch (subs_type) {
  case 0:
    walkList(ipList, printList, 0);
    break;
  case 2:
    print_substitution_table(subs_table);
    break;
  default:
    printf("unable to print substitution table of type %d\n",subs_type);
  }
}

void print_key_schedule_cache() {

  printf("\nt table (list of conversations)\n");
  switch (keycache_type) {
  case 0:
    walkList(ipPairList, printList, 0);
    break;
  case 2:
    print_key_schedule_cache(keycache_table);
    break;
  default:
    printf("unable to print key cache/conversation list of type %d\n",
	   keycache_type);
  }
}

int get_substitution_table_size() {
    switch (subs_type) {
    case 2:
	return hash_get_substitution_table_size(subs_table);
    default:
	errx(1,"get_substitution_table_size not implemented for type %d",
	     subs_type);
    }
}


