/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/time.h>
#ifdef LOCAL_BPF_HEADERS
/* So we can build this on a system w/o kernel changes */
#include "../sys/net/bpf.h"
#else
#include "/usr/src/sys/net/bpf.h"
#endif
#include <pcap.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <err.h>
#include <assert.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include "../dump/cache.h"
#include "../bpf/parse_bpf.h"

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

/*XXX The btree, list, hash stuff expects this global variable even if
  it's not needed.  Fix me. */
int cryptalg=0;

/* The fixed-length part of the IP header is all we care about: */
#define IP_HDR_SIZE 20

void
usage(char * prog) {
    printf("usage: %s -o output_file [-f file0] [-d] file1 file2 ...\n",prog);
    printf(
"translates the addresses in each input file, creating a single pcap output\n"
"file with IP addresses obscured by a one-to-one mapping.\n"
"Assumes the given input files are in pcap format.\n"
"If given -f filename, reads the names of input files from the file file0,\n"
"which should have one filename per line.\n"
"If given -d, deletes each original file after it finishes reading it.");
    exit(1);
}

void
process_packet(u_char * user, struct pcap_pkthdr * hdr, u_char * pkt) {
    FILE * outfile = (FILE *)user;
    struct ip * ip;

    memset(pkt,0xaa,12); /* obscure ethernet addresses */
    ip = (struct ip *)(pkt + ETHER_HDR_LEN);
    ip->ip_src.s_addr = obscure_address(ip->ip_src.s_addr);
    ip->ip_dst.s_addr = obscure_address(ip->ip_dst.s_addr);

    if (fwrite(hdr,sizeof(*hdr),1,outfile) != 1)
	err(1,"Error writing packet header to output file");
    if (fwrite(pkt,hdr->caplen,1,outfile) != 1)
	err(1,"Error writing packet body to output file");
}

void process_file(char * infile_name, FILE * outfile) {
    pcap_t * pcap_infile;
    char error_msg[PCAP_ERRBUF_SIZE];
    int retval;
    
    if ((pcap_infile=pcap_open_offline(infile_name,error_msg)) == NULL)
	errx(1,error_msg);
    retval = pcap_dispatch(pcap_infile,0,(pcap_handler)process_packet,
			   (void *)outfile);
    if (retval == -1)
	errx(1,pcap_geterr(pcap_infile));
    if (retval != 0)
	errx(1,
	     "I was expecting pcap_dispatch to return 0, but it returned %d\n",
	     retval);
    pcap_close(pcap_infile);
}

#define MAX_FILENAME_LENGTH 1024

char * get_line(FILE * filenamefile) {
    static char line[MAX_FILENAME_LENGTH+1];

    if (fgets(line,MAX_FILENAME_LENGTH,filenamefile) ==NULL) {
	if (ferror(filenamefile))
	    err(1,"error reading filename file");
	else
	    return(NULL); /* eof */
    }
    if (line[strlen(line)-1] != '\n')
	errx(1,"filename too long");
    /* chop off final carriage return */
    line[strlen(line)-1] = '\0';
    return(line);
}

int
main(int argc, char * argv[]) {
    int i;
    struct timeval now;
    int optch;
    extern char * optarg;
    extern int optind;
    FILE * outfile = NULL;
    FILE * filenamefile = NULL;
    char * filename;
    int dflag = 0;

    gettimeofday(&now,NULL);
#ifdef __linux__
    /*XXX This is bad, replace it. */
    
    srandom(now.tv_usec);
#else
    srandomdev();
#endif

    while ((optch = getopt(argc,argv,"o:f:d")) != -1) {
	switch (optch) {
	case 'o':
	    if ((outfile = fopen(optarg,"w")) == NULL)
		err(1,"unable to open %s",optarg);
	    break;
	case 'f':
	    if ((filenamefile = fopen(optarg,"r")) == NULL)
		err(1,"unable to open %s",optarg);
	    break;
	case 'd':
	    dflag = 1;
	    break;
	default:
	    usage(argv[0]);
	}
    }

    write_pcap_header(outfile,0);
    init_substitution_table(2);

    while ((filename = get_line(filenamefile)) != NULL) {
	printf("processing %s; translation table size %d\n",filename,
	       get_substitution_table_size());
	process_file(filename,outfile);
	if (dflag)
	    if (unlink(filename) == -1)
		err(1,"unable to delete %s",filename);
    }
    fclose(filenamefile);
    
    for (i=optind; i<argc; i++) {
	printf("processing %s; translation table size %d\n",argv[i],
	       get_substitution_table_size());
	process_file(argv[i],outfile);
	if (dflag)
	    if (unlink(filename) == -1)
		err(1,"unable to delete %s",filename);

    }
    fclose(outfile);
    exit(0);
}
