/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/time.h>
#ifdef LOCAL_BPF_HEADERS
/* So we can build this on a system w/o kernel changes */
#include "../sys/net/bpf.h"
#else
#include "/usr/src/sys/net/bpf.h"
#endif
#include <pcap.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <err.h>
#include <assert.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

#include "../bpf/parse_bpf.h"
#include "../crypto/crypto.h"
#include "../dump/btree.h"
#include "../dump/list.h"
#include "../crypto/crypto.h"
#include "../crypto/crypto_file.h"
#include "histogram.h"

/* The fixed-length part of the IP header is all we care about: */
#define IP_HDR_SIZE 20

/* The fixed-length part of the IP header is all we care about: */
#define IP_HDR_SIZE 20

void usage(char * prog) {
  printf(
"usage: %s -t time_interval -V volume_key -T tt_key\n"
"       file1 file2 ...\n"
"Reports on the highest density of packets found in the given\n"
"segment files over any interval of size time_interval\n"
"(measured in milliseconds).  Keys are needed to recover the\n"
"timestamps. Assumes the segments are given on the command line in \n"
"chronological order.\n"
"if the files are in open header format, then the keys are not required.\n",
prog);
  exit(1);
}

struct pkt_stats {
  struct timeval time;
  int size;
  struct pkt_stats * next;
};

struct window_stats {
  struct timeval start;
  struct timeval end;
  int num_pkts;
  int num_bytes;
  double pkt_density;
  double byte_density;
  char * filename;
};

struct pkt_stats * stats_head, * stats_tail;
struct window_stats high_byte_window, high_pkt_window;
int num_bytes, num_pkts;
int window_size = 1000000; /* 1 second default */
char * this_filename;

void init_pkt_stats() {
  stats_head = NULL;
  stats_tail = NULL;
  num_pkts = 0;
  num_bytes = 0;
}

void process_this_window() {
  double this_byte_density, this_pkt_density;
  int this_span;

  this_span = (stats_head->time.tv_sec - stats_tail->time.tv_sec)*1000000
    + (stats_head->time.tv_usec - stats_tail->time.tv_usec);
  if (this_span == 0)
    this_span = 1;
  this_pkt_density = (double)num_pkts/window_size;
  this_byte_density = (double)num_bytes/window_size;
  if (this_pkt_density > high_pkt_window.pkt_density) {
    memcpy(&high_pkt_window.start,&stats_tail->time, sizeof(struct timeval));
    memcpy(&high_pkt_window.end,  &stats_head->time, sizeof(struct timeval));
    free(high_pkt_window.filename);
    if ((high_pkt_window.filename = strdup(this_filename)) == NULL)
      errx(1,"unable to allocate memory for filename %s",this_filename);
    high_pkt_window.pkt_density = this_pkt_density;
    high_pkt_window.byte_density = this_byte_density;
    high_pkt_window.num_pkts = num_pkts;
    high_pkt_window.num_bytes = num_bytes;
  }
  if (this_byte_density > high_byte_window.byte_density) {
    memcpy(&high_byte_window.start,&stats_tail->time, sizeof(struct timeval));
    memcpy(&high_byte_window.end,  &stats_head->time, sizeof(struct timeval));
    free(high_byte_window.filename);
    if ((high_byte_window.filename = strdup(this_filename)) == NULL)
      errx(1,"unable to allocate memory for filename %s",this_filename);
    high_byte_window.pkt_density = this_pkt_density;
    high_byte_window.byte_density = this_byte_density;
    high_byte_window.num_pkts = num_pkts;
    high_byte_window.num_bytes = num_bytes;
  }
}

void process_new_pkt(struct timeval * pkt_time, int pkt_size) {
  struct pkt_stats * new_pkt_stats, * old_pkt_stats;
  int dt;

  if ((new_pkt_stats = malloc(sizeof(struct pkt_stats))) == NULL)
    errx(1,"Unable to allocate memory for packet statistics");
  memcpy(&new_pkt_stats->time,pkt_time,sizeof(struct timeval));
  new_pkt_stats->size = pkt_size;
  new_pkt_stats->next = NULL;
  if (num_pkts == 0) {
    stats_head = new_pkt_stats;
    stats_tail = new_pkt_stats;
    num_pkts++;
    num_bytes += stats_head->size;
  } else {
    dt = (new_pkt_stats->time.tv_sec - stats_tail->time.tv_sec) * 1000000
      + (new_pkt_stats->time.tv_usec - stats_tail->time.tv_usec);
    if (dt <0)
      errx(1,"we've gone backwards in time; unordered segments?");
    if (dt > window_size) {
      process_this_window();
      while (dt > window_size) {
	old_pkt_stats = stats_tail;
	stats_tail = stats_tail->next;
	num_pkts--;
	num_bytes -= old_pkt_stats->size;
	free(old_pkt_stats);
	if (stats_tail == NULL) {
	  stats_head = NULL;
	  break;
	}
	dt = (new_pkt_stats->time.tv_sec - stats_tail->time.tv_sec) * 1000000
	  + (new_pkt_stats->time.tv_usec - stats_tail->time.tv_usec);
      }
    }
    if (stats_head == NULL) {
      stats_head = new_pkt_stats;
      stats_tail = new_pkt_stats;
    } else {
      stats_head->next = new_pkt_stats;
      stats_head = stats_head->next;
    }
    num_pkts++;
    num_bytes += stats_head->size;
  }
}

void flush_pkt_stats() {
  struct pkt_stats * old_pkt_stats;

  process_this_window();
  while (stats_tail) {
    old_pkt_stats = stats_tail;
    stats_tail = stats_tail->next;
    free(old_pkt_stats);
  }
}

void print_window_stats(struct window_stats * window) {

  printf("segment: %s\n",window->filename);
  printf("start: %ld.%06ld; end: %ld.%06ld\n",
	 window->start.tv_sec, window->start.tv_usec,
	 window->end.tv_sec, window->end.tv_usec);
  printf("packets: %d\n",window->num_pkts);
  printf("bytes: %d\n",window->num_bytes);
  printf("packets per second: %f\n",1000000 * window->pkt_density);
  printf("bytes per second: %f\n",1000000 * window->byte_density);
}

void print_stats() {

  printf("highest byte densities observed:\n");
  print_window_stats(&high_byte_window);
  printf("highest packet densities observed:\n");
  print_window_stats(&high_pkt_window);
}

void process_packet(struct crypto_file * infile,
		    struct crypto_packet * packet,
		    void * user) {

  process_new_pkt(get_pkt_timestamp(packet),get_pkt_size(packet));
}

int main(int argc, char * argv[])
{
  struct crypto_file * infile;
  char * volkey_filename = NULL;
  char * tkey_filename = NULL;
  int optch;
  extern char * optarg;
  extern int optind;
  int i;
  struct crypto_key_ring * keys = NULL;

  while ((optch = getopt(argc,argv,"t:T:V:")) != -1) {
    switch(optch) {
    case 't':
      window_size = atoi(optarg);
      printf("using window size of %d microseconds\n",window_size);
      break;
    case 'T':
      tkey_filename = optarg;
      break;
    case 'V':
      volkey_filename = optarg;
      break;
    case '?':
    default:
      usage(argv[0]);
    }
  }

  if (optind >= argc)
    usage(argv[0]);

  init_pkt_stats();

  for (i=optind; i<argc; i++) {
    infile = open_input_crypto_file(argv[i]);
    if ((i == optind) && (infile->format != 0)) {
      keys = init_crypto_keys(infile->cryptalg);
      get_vol_key(keys,volkey_filename);
      get_tr_key(keys,tkey_filename);
    }
    this_filename = argv[i];
    read_crypto_file(infile,keys,&process_packet,NULL);
    close_input_crypto_file(infile);
  }

  flush_pkt_stats();
  print_stats();
  if (keys)
    destroy_crypto_keys(keys);

  exit(0);
}

