/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#ifdef LOCAL_BPF_HEADERS
/* So we can build this on a system w/o kernel changes */
#include "../sys/net/bpf.h"
#else
#include "/usr/src/sys/net/bpf.h"
#endif
#include <pcap.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <err.h>

#include "../bpf/parse_bpf.h"
#include "../crypto/crypto.h"
#include "../crypto/crypto_file.h"

void usage(char * prog) {
  printf("usage: %s\n",prog);
  printf(
"-o file: file to write cleartext packets to, in pcap\n"
"         format (packets will be appended if file exists)\n"
"-c file: encrypted file to read\n"
"-G file: file containing segment key\n"
"decrypts file that was written in segment format\n");
	exit(1);
}

void process_packet(struct crypto_file * infile,
		    struct crypto_packet * packet,
		    void * user) {
  FILE * outfile = (FILE *)user;
  struct pcap_pkthdr clear_pkt_hdr;

  clear_pkt_hdr.ts = *get_pkt_timestamp(packet);
  clear_pkt_hdr.caplen = clear_pkt_hdr.len = get_pkt_size(packet);
  if (fwrite(&clear_pkt_hdr,sizeof(clear_pkt_hdr),1,outfile) != 1)
    errx(1,"Error writing packet header to clear file");
  if (fwrite(get_pkt_contents(packet),1,get_pkt_size(packet),outfile)
      != get_pkt_size(packet))
    errx(1,"error writing packet contents to clear file");
}

int main(int argc, char * argv[]) {
  FILE * outfile = NULL;
  struct crypto_file * infile = NULL;
  char * segment_key_filename = NULL;
  struct crypto_key_ring * keys;
  int need_header=1;
  char * outfilename = NULL;
  int optch;
  extern char * optarg;
  extern int optind;
  int bytes;

  umask(077);	/* Make files only visible to the owner */

  while ((optch = getopt(argc,argv,"o:c:G:")) != -1) {
    switch(optch) {
    case 'o':
      outfilename = optarg;
      /* If the output file doesn't already exist, then we'll need to
         add a header: */
      need_header=access(outfilename,F_OK);
      if ((outfile=fopen(outfilename,"a")) == NULL) {
	errx(1,"Error opening output file %s\n",outfilename);
      }
      break;
    case 'c':
      infile = open_input_crypto_file(optarg);
      break;
    case 'G':
      segment_key_filename = optarg;
      break;
    case '?':
    default:
      usage(argv[0]);
    }
  }
  if (infile == NULL)
    usage(argv[0]);
  keys = init_crypto_keys(infile->cryptalg);
  get_seg_key(keys,segment_key_filename);

  if (infile->format != 3)
    errx(1,"This file was encrypted in format %d;"
	 " use decrypt or decrypt_packets.\n",infile->format);
  
  if (need_header) {
    bytes = (write_pcap_header(outfile,0));
    if (!bytes)
      errx(1,"error writing file %s",outfilename);
  }

  read_crypto_file(infile,keys,&process_packet,outfile);
  exit(0);
}
