/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#ifdef LOCAL_BPF_HEADERS
/* So we can build this on a system w/o kernel changes */
#include "../sys/net/bpf.h"
#else
#include "/usr/src/sys/net/bpf.h"
#endif
#include <pcap.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <err.h>

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

#include "../bpf/parse_bpf.h"
#include "../crypto/crypto.h"
#include "../crypto/crypto_file.h"

void usage(char * prog) {
  printf("usage: %s [options]\n",prog);
  printf(
"-o file: file to write cleartext packets to, in pcap format\n"
"         (packets will be appended if file exists)\n"
"-c file: encrypted file to read\n"
"-s addr: source IP address\n"
"-d addr: destination IP address\n"
"-f addr: fake source IP address (open header format only)\n"
"-g addr: fake destination IP address (open header format only)\n"
"-C file: file containing conversation key\n"
"-S file: file containing source endpoint key (endpoint format only)\n"
"-D file: file containing destination endpoint key (endpoint format only)\n");
  exit(1);
}

void process_packet(struct crypto_file * infile,
		    struct crypto_packet * packet,
		    void * user) {
  FILE * outfile = (FILE *)user;
  struct pcap_pkthdr clear_pkt_hdr;

  if (!packet->encrypted) {
    clear_pkt_hdr.ts = *get_pkt_timestamp(packet);
    clear_pkt_hdr.caplen = clear_pkt_hdr.len = get_pkt_size(packet);
    if (fwrite(&clear_pkt_hdr,sizeof(clear_pkt_hdr),1,outfile) != 1)
      errx(1,"Error writing packet header to clear file");
    if (fwrite(get_pkt_contents(packet),1,get_pkt_size(packet),outfile)
	!= get_pkt_size(packet))
      errx(1,"error writing packet contents to clear file");
  }
}

void parse_address(char * in, in_addr_t * out) {
  if (inet_pton(AF_INET,in,out) <=0)
    errx(1,"Error parsing source address %s",in);
}

int main(int argc, char * argv[])
{
  FILE * outfile = NULL;
  struct crypto_file * infile = NULL;
  struct crypto_key_ring * keys = NULL;
  char * key_filename = NULL;
  char key_type = 0;
  char * ofilename = NULL;
  in_addr_t src_addr, dst_addr, fake_src_addr, fake_dst_addr;
  extern char * optarg;
  int optch;
  extern int optind;
  int need_header = 1;

  umask(077);	/* Make files only visible to the owner */

  while ((optch = getopt(argc,argv,"o:c:d:f:g:s:C:S:D:")) != -1) {
    switch(optch) {
    case 'o':
      ofilename = optarg;
      /* If the output file doesn't already exist, then we'll need to
         add a header: */
      need_header=access(ofilename,F_OK);
      if ((outfile=fopen(ofilename,"a")) == NULL)
	errx(1,"Error opening output file %s\n",ofilename);
      break;
    case 'c':
      infile = open_input_crypto_file(optarg);
      break;
    case 's':
      parse_address(optarg,&src_addr);
      break;
    case 'd':
      parse_address(optarg,&dst_addr);
      break;
    case 'f':
      parse_address(optarg,&fake_src_addr);
      break;
    case 'g':
      parse_address(optarg,&fake_dst_addr);
      break;
    case 'C':
    case 'S':
    case 'D':
      if (key_type)
	errx(1,"There should only be one C, S, or D argument");
      key_type = optch;
      key_filename = optarg;
      break;
    case '?':
    default:
      usage(argv[0]);
    }
  }

  if (!infile)
    errx(1,"must specify an input file");
  if (((key_type == 'S') || (key_type == 'D')) && (infile->format != 2))
    errx(1,"File is in format %d, which does not use endpoint keys",
	 infile->format);
  keys = init_crypto_keys(infile->cryptalg);
  if (need_header) {
    if (!write_pcap_header(outfile,0))
      errx(1,"Failure writing to file %s",ofilename);
  }
  switch (key_type) {
  case 'C':
    if (infile->format == 0)
      get_conv_key_open_headers(keys,key_filename,src_addr,dst_addr,
				fake_src_addr,fake_dst_addr);
    else
      get_conv_key(keys,key_filename,src_addr,dst_addr);
    break;
  case 'S':
    get_src_key(keys,key_filename,src_addr);
    break;
  case 'D':
    get_dst_key(keys,key_filename,dst_addr);
    break;
  }

  read_crypto_file(infile,keys,&process_packet,outfile);
  exit(0);
}
