/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */
#include <sys/types.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <err.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include "../crypto/crypto.h"

#ifdef __FreeBSD__
typedef u_int32_t in_addr_t;
#endif

void usage(char * progname) {

  printf("Usage: %s [-e algorithm] -s src_addr -d dst_addr\n",progname);
  printf("       -V vol_key_file -o conv_key_file\n");
  printf("generates a key which will reveal all keys with source address\n");
  printf("src_addr and destination address dst_addr.  It uses the key in\n");
  printf("vol_key_file and writes the new key to conv_key_file.\n");
  printf("The algorithm must be chosen from the following list:\n");
  print_algorithms();
  /* Note that we assume both the volume key and the generated key are
     of the same type.  In the future we might want to use two different
     ciphers for the two purposes.*/
  exit(1);
}

int main(int argc, char * argv[]) {
  extern char *optarg;
  extern int optind;
  int ch;
  int cryptalg = DEFAULT_ALGORITHM;
  int keylen;
  in_addr_t src_addr,dst_addr;
  char volume_key[MAX_KEY_LENGTH], * conv_key;
  struct crypto_session vol_key_session;
  struct ip dummy_ip_packet;
  FILE * outfile;

  while ((ch = getopt(argc, argv, "e:s:d:V:o:")) != -1) {
    switch (ch) {
    case 'e':
      cryptalg = atoi(optarg);
      break;
    case 's':
       if (inet_pton(AF_INET,optarg,&src_addr) <=0)
        errx(1,"Error parsing source address %s",optarg);
      break;
    case 'd':
      if (inet_pton(AF_INET,optarg,&dst_addr) <=0)
        errx(1,"Error parsing destination address %s",optarg);
      break;
    case 'V':
      read_key_file(optarg,volume_key,cryptalg);
      get_crypto_session(&vol_key_session,cryptalg,CBC_MODE,ENCRYPT,
			 volume_key,0);
      break;
    case 'o':
      if ((outfile = fopen(optarg,"w")) == NULL)
	errx(1,"Error opening %s",optarg);
      break;
    case '?':
    default:
      usage(argv[0]);
    }
  }
  if (optind<5)
    usage(argv[0]);

  dummy_ip_packet.ip_src.s_addr = src_addr;
  dummy_ip_packet.ip_dst.s_addr = dst_addr;
  conv_key = make_conv_key(&vol_key_session,&dummy_ip_packet,cryptalg);
  keylen = algorithms[cryptalg].key_length;
  if (fwrite(conv_key,1,keylen,outfile) != keylen)
    errx(1,"error writing key");
  fclose(outfile);
  exit(0);
}

