/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

/* We assume throughout that the key length is a multiple of the block
   length. We also frequently assume that the same algorithm is used
   for everything (generating keys, encrypting packets, encrypting
   tables, etc.)*/

struct crypto_file {
  char * filename;
  FILE * file;
  int format;
  int cryptalg;
  char iv[MAX_BLOCK_LENGTH];
};

struct crypto_packet {
  char data[MAX_PACKET_SIZE];
  /* Don't use these to get the length of a packet; use get_pkt_size, below. */
  short int total_length; /* including bpf headers, keys, etc., with padding */
  short int data_length; /* same, but without padding */
  int encrypted;
  int cryptalg;
  int format;
};

struct conv_key_descriptor {
  struct crypto_session session;
  in_addr_t real_src;
  in_addr_t real_dst;
  in_addr_t fake_src;
  in_addr_t fake_dst;
};

struct endpoint_key_descriptor {
  struct crypto_session session;
  in_addr_t addr;
};

struct crypto_key_ring {
  int cryptalg;
  struct crypto_session * vol_key;
  struct crypto_session * tr_key;
  struct conv_key_descriptor * conv_key;
  struct endpoint_key_descriptor * src_key;
  struct endpoint_key_descriptor * dst_key;
  struct crypto_session * seg_key;
};

/* Used by read_crypto_file, below. */

typedef void crypto_infile_callback(struct crypto_file * infile,
				    struct crypto_packet * packet,
				    void * user);

/* These functions generally report errors and die on their own, as opposed
   to returning error information to the caller. */

struct crypto_file * open_input_crypto_file(char * filename);

void close_input_crypto_file(struct crypto_file * infile);

/* read_crypto_file reads a file, opened with open_input_crypto_file,
   attempts to decrypt each packet using the given keys, and calls
   the user-provided handle_packet for each packet.  (The user can check
   packet->encrypted in their handle_packet function to see whether
   that packet was succesfully decrypted or not.) The "user" pointer
   is passed unchanged to handle_packet. */

void read_crypto_file(struct crypto_file * infile,
		      struct crypto_key_ring * keys,
		      crypto_infile_callback handle_packet,
		      void * user);

/* Reads, decrypts, and initializes the substitution table in the
   given file. */
void init_subs_table(struct crypto_key_ring * keys,
		     char * filename);

/* The following functions return information about a packet stored in
   a struct crypto_packet, and die if that information is not
   available (for example, timestamps can't be deduced from
   undecrypted packets in conversation and endpoint formats). */
struct timeval * get_pkt_timestamp(struct crypto_packet * packet);

/* Returns length of ethernet header + payload */
int get_pkt_size(struct crypto_packet * packet);

/* pointer to the beginning of the ethernet header */
char * get_pkt_contents(struct crypto_packet * packet);


struct crypto_key_ring * init_crypto_keys(int cryptalg);

void destroy_crypto_keys(struct crypto_key_ring * keys);

/* The following functions read in key data from the given file,
   generate a key schedule, and store it in the given keyring. Some
   additional information is stored with conversation and endpoint
   keys.*/
void get_vol_key(struct crypto_key_ring * keys, char * filename);

void get_tr_key(struct crypto_key_ring * keys, char * filename);

void get_conv_key(struct crypto_key_ring * keys, char * filename,
		  in_addr_t src, in_addr_t dst);

void get_conv_key_open_headers(struct crypto_key_ring * keys, char * filename,
			       in_addr_t real_src, in_addr_t real_dst,
			       in_addr_t fake_src, in_addr_t fake_dst);

void get_src_key(struct crypto_key_ring * keys, char * filename,
		 in_addr_t addr);

void get_dst_key(struct crypto_key_ring * keys, char * filename,
		 in_addr_t addr);

void get_seg_key(struct crypto_key_ring * keys, char * filename);


