/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <pcap.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#ifndef __linux__
#ifdef LOCAL_BPF_HEADERS
#include "../sys/net/bpf.h"
#else
#include <net/bpf.h>
#endif
#endif
#include "parse_bpf.h"
#include <err.h>

#define TIME_ZONE -4
#define TCPDUMP_MAGIC 0xa1b2c3d4

int get_padded_pkt_length(struct bpf_hdr * hdr, int padding) {
  int tot_length, padding_length;
  if (padding) {
    tot_length = hdr->bh_hdrlen + hdr->bh_caplen;
    padding_length = BPF_WORDALIGN(tot_length) - tot_length;
  }
  else {
    padding_length = 0;
  }
  return(hdr->bh_caplen + padding_length);
}

int read_bpf(FILE * infile, struct bpf_hdr * hdr,
	     char * bpf_content, int padding) {
  int padded_pkt_length;

  if (fread(hdr,BPF_HDR_SIZE,1,infile) != 1) {
    return(-1);
  }
  if (hdr->bh_hdrlen != BPF_HDR_SIZE) {
    printf("Woah! Found a bpf header of length %d\n",
	   hdr->bh_hdrlen);
    exit(1);
  }
  if (hdr->bh_caplen > MAX_PACKET_SIZE-BPF_HDR_SIZE) {
    perror("captured packet is too long!");
    exit(1);
  }
  padded_pkt_length = get_padded_pkt_length(hdr, padding);
  if (fread(bpf_content,1,padded_pkt_length,infile) != padded_pkt_length) {
    return(-1);
  }
  return(0);
}

int write_bpf(FILE * outfile, struct bpf_hdr * hdr, char * packet,
	      int padding) {
  int padded_pkt_length;

  if (fwrite(hdr,1,BPF_HDR_SIZE,outfile) != BPF_HDR_SIZE) {
    return(-1);
  }
  padded_pkt_length = get_padded_pkt_length(hdr, padding);
  if (fwrite(packet,1,padded_pkt_length,outfile) != padded_pkt_length) {
    return(-1);
  }
  return(0);
};

void pprint_bpf(FILE * outfile, struct bpf_hdr * hdr, char * packet) {

  fprintf(outfile,"timeval: ?\n");
  fprintf(outfile,"caplen: %#x\n",hdr->bh_caplen);
  fprintf(outfile,"datalen: %#x\n",hdr->bh_datalen);
  fprintf(outfile,"hdrlen: %#x\n",hdr->bh_hdrlen);
}

int write_pcap_header(FILE * outfile, int snaplen) {
  struct pcap_file_header hdr;
  
  hdr.magic = TCPDUMP_MAGIC;
  hdr.version_major = PCAP_VERSION_MAJOR;
  hdr.version_minor = PCAP_VERSION_MINOR;
  /* This should be set to the time zone. */
  hdr.thiszone = TIME_ZONE;
  hdr.snaplen = (snaplen ? snaplen : ETHERMTU+ETHER_HDR_LEN);
  hdr.sigfigs = 0;
  /*this should be set to the real link type*/
  hdr.linktype = DLT_EN10MB; 
  if (fwrite(&hdr,sizeof(hdr),1,outfile)!=1)
    return(0);
  return(sizeof(hdr));
};
