/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <err.h>

#define DEFAULT_SIGNAL SIGHUP
#define BUFFER_SIZE 1024

int outfile;
char *outfile_name, *pidfile_name;

/* signame_to_signum and printsignals are both taken from
   /usr/src/bin/kill/kill.c: */

int
signame_to_signum(sig)
	char *sig;
{
	int n;

	if (!strncasecmp(sig, "sig", 3))
		sig += 3;
	for (n = 1; n < NSIG; n++) {
		if (!strcasecmp(sys_signame[n], sig))
			return (n);
	}
	return (-1);
}

void
printsignals(fp)
	FILE *fp;
{
	int n;

	for (n = 1; n < NSIG; n++) {
		(void)fprintf(fp, "%s", sys_signame[n]);
		if (n == (NSIG / 2) || n == (NSIG - 1))
			(void)fprintf(fp, "\n");
		else
			(void)fprintf(fp, " ");
	}
}

void
open_outfile() {
  if ((outfile = open(outfile_name, O_WRONLY|O_CREAT|O_APPEND, 0666)) < 0)
    err(1, "open_outfile can't open %s", outfile_name);
  printf("Opened %s\n", outfile_name);
}

void
write_pidfile() {
  FILE *ofd;

  if ((ofd = fopen(pidfile_name, "w")) == NULL)
    err(1, "write_pidfile can't fopen %s", pidfile_name);
  fprintf(ofd, "%d\n", getpid());
  if (fclose(ofd) != 0)
    warn("write_pidfile can't fclose %s", pidfile_name);
}

void
reopen_outfile(int signo) {
  close(outfile);
  open_outfile();
}

void
usage(char *s) {
    fprintf(stderr, "usage: %s -o logfile -p pidfile [-s signal number]\n", s);
    exit(1);
  }

int
main(int argc, char *argv[]) {
  char buffer[BUFFER_SIZE];
  int bytes;
  int reopen_signal = DEFAULT_SIGNAL;
  extern char *optarg;
  extern int optind;
  char optch;

  while ((optch = getopt(argc, argv, "o:p:s:")) != -1) {
    switch (optch) {
      case 'o':			/* output file name */
        outfile_name = optarg;
        break;
      case 'p':			/* file to record pid */
        pidfile_name = optarg;
        break;
      case 's':			/* signal to send */
        if (isdigit(*optarg))
          reopen_signal = atoi(optarg);
        else
          if ((reopen_signal = signame_to_signum(optarg)) == -1) {
	    fprintf(stderr,"unknown signal %s; valid signals:\n",argv[2]);
	    printsignals(stderr);
	    exit(1);
          }
        break;
      case '?':
      default:
        usage(argv[0]);
    }
  }

  if (!outfile_name || !pidfile_name)
    usage(argv[0]);

  open_outfile();
  write_pidfile();

  if (signal(reopen_signal, reopen_outfile) == SIG_ERR)
    err(1, "can't set reopen_signal handler");
  /*
   * Ignore SIGINT to prevent logmonitor exit while there is still output
   * in the pipe.  When the pipe writer dies, the read() below will return
   * zero, and we'll exit.
   */
  if (signal(SIGINT, SIG_IGN) == SIG_ERR)
    err(1, "can't ignore SIGINT");

  while(1) {
    bytes = read(STDIN_FILENO,buffer,BUFFER_SIZE);
    if (bytes == 0) {
      printf("End of file\n");
      exit(0);
    } else  if (bytes < 0)
      err(1, "Read error");
    if (write(outfile,buffer,bytes) != bytes)
      err(1, "Write error");
  }
}
