#!/usr/local/bin/tclsh

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

#
# background tasks for listening to the net.
#
# Start by initializing the two keys: volume key, and the translation key.
#
# We want to create `volumes' of data to be tar'ed onto a dlt drive...
#
# Each volume shouldn't exceed 1G, so for sanity's sake,
# We may yet decide to up this based on disk drive sizes...
#
# here's how it works....
# we sniff the net using the listen command.
# it makes files in a directory called. 'raw'.
#
# The pkt_dump command then is used to encrypt the data in the 
# sniffed data.  It generates three files from each data file.
# The first file is called the data file, the second the `s' file
# and the third the `t' file.  The s file contains th substituted 
# ip addresses, and the `t' file contains the substituted ip/pair
# addresses and their associated payload keys.
#



#
# configuration --------------------------------------------------------
#

set rawMount	/mfs
set rawDir	/mfs/raw/
set ufsMount	/scratch0
set ufsDir	$ufsMount/volumes
set volDir	volumes
set volInfoDir	volInfo
set listenDir	../listen
set dumpDir	../dump
set utilDir	../util
set pilotDir	.

#
# The following can all be specified via command-line option as
# segsize=??, segtime=??, mfssize=??, volsize=??,
# mfsthreshold=??, ufsthreshold=??, interface=??
# segsize, volsize, mfssize, mfsthreshold, ufsthreshold are all specified in MB
#    (i.e. --segsize=20 means 20MB segments, --mfssize=256 means 256MB MFS)
# segtime is in seconds
#

# segSize is when we start worrying about a new segment, size-wise...
# segTime is when we start worrying about a new segment, time-wise...
set segSize	[expr 16*1024*1024]
set segTime	60

# mfsLimit is when we start worrying about more memory...
# This is the number of 512-byte sectors.
# For 256MB of MFS, we need 256*2*1024 512-byte sectors.

set mfsLimit	[expr 256*2*1024]

# limitSize is when we start worrying about a new volume...

set limitSize	[expr 1000*1024*1024]

# mfsThreshold is when we refuse to store more bytes into MFS...
# ufsThreshold is when we refuse to store more bytes into UFS...
set mfsThreshold	32
set ufsThreshold	1024

# interface is whence the vault fetches packets
set interface	fxp0

#
# globals --------------------------------------------------------------
#

set volume	UNDEF
set volumeSize	UNDEF

set highVol	$volInfoDir/highVolNum
set incrHighVol	$volInfoDir/incrVolNum
set resetVol	$volInfoDir/resetVolNum

set highVolSet	$volInfoDir/highVolSetNum
set incrHighVolSet	$volInfoDir/incrVolSetNum

set apvsync	$utilDir/apvsync

# Default values for things changable by command-line arguments
set debuglvl  0
set cleartext 0
set cryptlvl  2
set cryptfmt  1

#
# subroutines ----------------------------------------------------------
#

# return the volume path given the volume number.

proc vPath { n } {
    global volDir

    return $volDir/$n
}

# return the current date and time, for logging output.

proc now { } {
    return [exec date "+\[%m/%d/%y %H:%M:%S\]"]
}

# make new volume and translation table keys.

proc makeNewKeys { } {
    global pilotDir volume

    puts "[now] making new vol/trans keys"
    catch {exec $pilotDir/mkVolKey  [vPath $volume]/volKey}
    puts "[now] Done with volkey: exec $pilotDir/mkVolKey [vPath $volume]/volKey"
    catch {exec $pilotDir/mkVolKey  [vPath $volume]/transKey}
    puts "[now] Done with transkey: exec $pilotDir/mkVolKey  [vPath $volume]/transKey"
    }

# check if volume size exceeded.  switch to new volume if so.

proc volSizeCheck { f } {
    global volumeSize limitSize volume apvsync

    puts "[now] Current volume size: $volumeSize ([expr $volumeSize/(1024*1024)]M)"
    if {$volumeSize >= $limitSize} {
	puts "[now] volumeSize $volumeSize exceeds limitSize $limitSize"
	exec $apvsync produce $volume
	makeVolume
    }
}

# make a new ufs volume within the current volume set.

proc makeVolume { } {
    global volumeSize volume volInfoDir
    global incrHighVol 

    #
    # Now start the next volume...
    #
    set volumeSize 0
    set volume [exec $incrHighVol $volInfoDir]
    catch {exec mkdir [vPath $volume]}

    #
    # now make new keys...
    #
    makeNewKeys
}

#
# Display options and exit
#

proc displayHelpMessage {} {
    global argv0;

    puts "Usage:"
    puts "$argv0 may include any or none of the following options:"
    puts "\t--help\t\t\tDisplays this message"
    puts "\t--debug\[=level\]\t\tSets debugging to one, or the given level"
    puts "\t--crypt\[=<none|desx|aes>\]\n\t\t\t\t\Specify encryption method (default is aes)"
    puts "\t--cleartext\t\tSave cleartext packets (for debug/verification)"
    puts "\t--interface=<intf>\tSpecify the ethernet interface to use"
    puts "\t--segsize=<size>\tSpecify max segment size in MB"
    puts "\t--segtime=<secs>\tSpecify max segment time in seconds"
    puts "\t--volsize=<size>\tSpecify volume size in MB"
    puts "\t--mfssize=<size>\tSpecify MFS file system size in MB"
    puts "\t--mfsthreshold=<size>\tSpecify MFS threshold size in MB"
    puts "\t--ufsthreshold=<size>\tSpecify UFS threshold size in MB"
    puts "\t--format=<prototype|conversation|endpoint>\tSpecify encryption format (default is conversation)"

    exit 0;
}


# 
# Process command-line arguments.
# They should be specified using "--keyword" or "--keyword=value".
# Any other specification of arguments will be ignored.  
#
# NOTE: Don't forget to declare global variables
#       as "global" to this routine.
#

proc processArgs {} {
    global argv0 argc argv
    global cryptlvl debuglvl cleartext interface cryptfmt
    global segSize segTime limitSize mfsLimit ufsMount mfsThreshold ufsThreshold

    for { set i 0 } { $i < $argc } { incr i } {
	set cur [lindex $argv $i]
	# puts "Argument $i is $cur"

	# Try parsing as either a keyword=value or just a keyword
	set keyvaluepair [ regexp {^--(.*)\=(.*)} $cur junk key1 value ]
	set keyonly [ regexp {^--(.*)$} $cur junk key2 ]

	#
	# If a keyword=value, then process the keyword and value
	#

	if { $keyvaluepair } {
	    # puts "Key $key1 = value $value"
	    switch -exact $key1 {
		crypt {
		    switch -exact $value {
			none { set cryptlvl 0 }
			desx { set cryptlvl 1 }
			aes  { set cryptlvl 2 }
			default { 
			    puts "Unrecognized crypt value '$value', using default"
			}
		    }
		    # puts "Crypt level is $cryptlvl"
		}
		format {
		    switch -exact $value {
			prototype { set cryptfmt 0 }
			conversation { set cryptfmt 1 }
			endpoint { set cryptfmt 2 }
			default { 
			    puts "Unrecognized crypt format '$value', using default"
			}
		    }
		}
		debug {
		    set debuglvl $value
		    # puts "Debug level is $debuglvl"
		}
		interface {
		    set interface $value
		}
		segsize {
		    set segSize [ expr $value*1024*1024 ]
		}
		segtime {
		    set segTime $value
		}
		volsize {
		    set limitSize [ expr $value*1024*1024 ]
		}
		mfssize {
		    # mfs size is in 512-byte sectors;
		    # need to multiply by 2 and 1024 to get MB
		    set mfsLimit [ expr $value*2*1024 ]
		}
		ufsmount {
		    # mfs mount point
		    set ufsMount $value
		}
		mfsthreshold {
		    # mfs threshold size is in MB
		    set mfsThreshold $value
		}
		ufsthreshold {
		    # ufs threshold size is in MB
		    set ufsThreshold $value
		}
		default { puts "Ignoring unrecognized keyword '$key1'" }
	    }

	#
	# If just a keyword, then process the keyword
	#

	} elseif { $keyonly } {
	    switch -exact $key2 {
		help { displayHelpMessage }
		debug { set debuglvl 1 }
		crypt { set cryptlvl 1 }
		cleartext { set cleartext 1 }
		default { puts "Ignoring unrecognized keyword '$key2'" }
	    }
	#
	# else neither a keyword or keyword=value, ignore it
	#

	} else {
	    puts "Ignoring unrecognized argument '$cur'"
	}
    }
}


#
# main script ----------------------------------------------------------
#

processArgs

puts "[now] Listener begins."

puts "[now] max bytes in volume: $limitSize ([expr $limitSize/(1024*1024)]M)"
puts "[now] max bytes in mfs: $mfsLimit ([expr $mfsLimit/(1024*1024)/2]M)"

exec $pilotDir/umount.sh $rawMount
exec mount_mfs -s $mfsLimit swap $rawMount
exec /bin/rm -rf $rawDir
exec mkdir $rawDir

exec $pilotDir/mount.sh $ufsMount
exec /bin/rm -f $volDir
exec /bin/ln -s $ufsDir $volDir

set volume [exec $incrHighVol $volInfoDir]
puts "[now] New high Volume number: $volume"

# create a new volume

catch {exec mkdir [vPath $volume] }
puts "[now] New Volume directory created: [vPath $volume]"
set volumeSize	0

makeNewKeys

#
# start the listener...
#

exec $listenDir/listen.sh -f $rawDir -s $segSize -t $segTime -u $ufsMount -M $mfsThreshold -U $ufsThreshold $interface &

while {1} {
    foreach file [glob -nocomplain $rawDir/:*] {
	set i [file tail $file] 
	if { $debuglvl > 0 } { puts "[now] Doing $i <$file>" }
	set p [vPath $volume]
	if { $debuglvl > 0 } { puts "[now] into volume <$p>" }

	if { $cleartext == 1 } {
	    set cleartextOption "-c $p/c$i"
	} else {
	    set cleartextOption ""
	}

	set pkt_dump_cmd "$dumpDir/pkt_dump.sh"
	set pkt_dump_args "-b -y -e $cryptlvl \
                -V $p/volKey -T $p/transKey \
                -o $p/$i -s $p/s$i -t $p/t$i \
		-w ${rawDir}X$i -x $p/x$i -z $cryptfmt $cleartextOption $file"

	if { $debuglvl > 0 } { puts "[now] exec $pkt_dump_cmd $pkt_dump_args" }

	#--------------------------------------------------------------
	# Execute the pkt_dump program (actually a wrapper script).
	# If it fails, then decipher the return code and exit.
	#--------------------------------------------------------------
	if { [ catch { exec $pkt_dump_cmd $pkt_dump_args } ] } {
		set retcode [ lindex [ split $errorCode { } ] 2 ]
		if { $retcode > 0x7f } {
			set retcode [ expr $retcode - 256 ]
		}
		puts "[now] pkt_dump returned $retcode. Terminating!"
		exit 1
	}

	if { $debuglvl > 0 } { puts "[now] removing <$file>" }
	exec rm -f $file
	if { $debuglvl > 0 } { puts "[now] removing <${rawDir}X$i>" }
	exec rm -f ${rawDir}X$i
	#
	# see if we're above the limit.
	#
	incr volumeSize [file size $p/$i]
	incr volumeSize [file size $p/s$i]
	incr volumeSize [file size $p/t$i]
	volSizeCheck $p/$i
    }

    if {[catch {glob test:*} ]} {
	after 250
    }
}
