#!/usr/bin/perl

#
# COPYRIGHT    2000
# THE REGENTS OF THE UNIVERSITY OF MICHIGAN
# ALL RIGHTS RESERVED
# 
# Permission is granted to use, copy, create derivative works
# and redistribute this software and such derivative works
# for any purpose, so long as the name of The University of
# Michigan is not used in any advertising or publicity
# pertaining to the use of distribution of this software
# without specific, written prior authorization.  If the
# above copyright notice or any other identification of the
# University of Michigan is included in any copy of any
# portion of this software, then the disclaimer below must
# also be included.
# 
# THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
# FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
# PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O 
# MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
# WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
# REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
# FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
# CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
# OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
# IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGES.
#

package Archiver;

use strict;
use English;
use Changer;
use Cwd;
use Getopt::Long;
use File::Basename;
use Errno qw(:POSIX);
use IPC::SysV qw(IPC_PRIVATE IPC_RMID S_IRWXU
		 IPC_CREAT IPC_EXCL IPC_NOWAIT SEM_UNDO SETVAL);


#------------------------------------------------------------------------
# Global variable declarations
#------------------------------------------------------------------------

#-----------------------------------------------
# These variables may be customized as required
#-----------------------------------------------

my $util_dir = "../util";		# Location of the utility program
					# directory (where the apvsync 
					# program may be found)

my $DEFAULT_VOLUME_PATH = "/scratch0/volumes";
					# Default path to locate volumes
					# to be written to tape

my $DEFAULT_VOLUMES_PER_TAPE = 100;	# Default number of (1GB) volumes
					# to be written to a tape

my $DEFAULT_MYSQL_TABLE = "firstgen";	# This is the default name of the
					# mySQL DB table (if --db_mysql)

my $DEFAULT_FILEDB_NAME = "/scratch0/fileDB/firstgen";
					# This is the default name of the
					# flat file DB (if --db_file)

my $DEFAULT_CHANGER_DEVICE = "/dev/ch0";
					# Default name of the changer device

my $DEFAULT_TAPE_DRIVE_DEVICE = "/dev/nrst";
					# Default name of the tape drive device
					# (w/o the device number)

my $DEFAULT_TAPE_DRIVE_NUMBER = 0;
					# Default drive number within the
					# changer

my $tar_pgm = "/usr/local/bin/tar";	# Location of a 'working' tar program
					# (The default tar shipped with
					# OpenBSD does not properly return
					# an indication of failures.)

my $mt_pgm = "/bin/mt";			# Location of the mt (magnetic
					# tape) program

my $asymm_pgm = "/usr/local/bin/gpg";	# Location of the GNU Privacy Guard
					# asymmetric encryption program

#------------------------------------
# These shouldn't need to be changed
#------------------------------------

my $sleep_seconds = 5;
my $time_to_quit = 0;
my $vol_count;

my $apvsync = cwd() . "/" . $util_dir . "/apvsync";
my $debug = 0;
my $volumes_per_tape;
my $db_file = "";
my $table_name = "";
my $db_mysql = 0;
my $changer_device = "";
my $tape_device_name = "";
my $tape_device_number = 0;
my $full_tape_device = "";
my $volume_path_base = "";
my $tar_to_dev_null = 0;
my $skip_db_write = 0;
my %opts;
my $niceness;
my $explicit_tape_marks = 0;
my $changer_ref;

#----------------------------------------------------------------
# The following are used only if we are using the mysql database
#----------------------------------------------------------------
my $dbh;

# SQL statements and their handles when "prepared"
my ($sql_update_status_to_archiving, $usta_sth);
my ($sql_update_status_to_complete, $ustc_sth);
my ($sql_select_by_status_full, $sbsf_sth);

#-----------
# Constants
#-----------
my $LOCK_SHARED = 1;
my $LOCK_EXCLUSIVE = 2;
my $LOCK_NOBLOCK = 4;
my $LOCK_UNLOCK = 8;

#########################################################################
#############  M A I N   E X E C U T I O N   P O I N T  #################
#########################################################################

#
# Make sure we are running as root
#
verify_we_are_root();

#
# Parse input options.
#
process_arguments();


# This must be called after processing arguments so that we know
# the changer device name and tape drive number
verify_we_are_alone();	

setpriority(0, 0, (getpriority(0, 0) + $niceness));

#
# Check the tar command to see if it properly reports errors
#
die "Fix tar before trying to run this program\n"
	if ( tar_command_is_broken() ) ;

#
# Tell the Changer code to debug if we're debugging
#
if ($debug) {
	$Changer::debug = $debug;
}

#
# Instantiate a Changer and get a reference to it
#
$changer_ref = Changer->new($changer_device, $tape_device_name);

#
# Initialize the use of a database
#
database_init($changer_ref);


#
# Set up handler for SIGINT to notify us when to quit...
#
print scalar(localtime), ": Installing SIGINT interrupt handler.\n";
$SIG{INT} = \&SIGINT_Handler;


#
# Process volumes until told to quit (basically forever)
#
do {
	my ($status, $tapeid, $volid);

	print scalar(localtime), ": Locating a scratch tape...\n";
	# Mount a scratch tape on the drive
	$tapeid = $changer_ref->mount_scratch($tape_device_number, $dbh);
	print "got $tapeid from mount_scratch\n" if ($debug);
	if ($tapeid == -1) { $time_to_quit = 1; }

	for (   $vol_count = 1;
		!$time_to_quit && $vol_count <= $volumes_per_tape;
		$vol_count++) {

		# Find a volume ready to be written
		print scalar(localtime),
				": Looking for a volume to process ...\n";

		$volid = semaphore_consume_get();
		if ($volid == -1) {
			# Break out of the loop if we've been told to quit
			# Otherwise, it is an error.
			last if ($time_to_quit);
			print "Error getting volume to process!\n";
			last;
		}
		
		print scalar(localtime),
			": Processing volume $volid (", $vol_count, " of ",
			"$volumes_per_tape for tape $tapeid) ...\n";

		# Write the volume
		$status = process_full_volume($volid, $tapeid, $vol_count);
		if ($status > 2) {
			print scalar(localtime), ": Possible tape error ",
			"-- getting new tape.\n";
			last;		# Break out of the for loop
		}

		# Write a tape mark (if requested)
		if ($explicit_tape_marks) {
			write_tape_mark(1);
		}

		# Break out of the loop if we've been told to quit
		last if ($time_to_quit);
	}

	# Write a final tape mark for EOT (if requested)
	if ($explicit_tape_marks) {
		write_tape_mark(1);
	}

	if ($time_to_quit) {
		print_tty(scalar(localtime),
			": Unloading $full_tape_device ...\n");
	}
	else {
		print scalar(localtime), ": Unloading $full_tape_device ...\n";
	}
	$changer_ref->unload_drive($tape_device_number);

} until $time_to_quit;

print_tty(scalar(localtime), ": Archiver terminated normally.\n");
exit 0;


#########################################################################
#############           S U B R O U T I N E S           #################
#########################################################################

#------------------------------------------------------------------------
# Process command-line arguments
#------------------------------------------------------------------------
sub process_arguments {
	GetOptions( \%opts,	"help",
				"debug:i",
				"volspertape=i",
				"db_file:s",
#				"db_mysql:s",
				"changer=s",
				"drive=s",
				"drivenum=i", 
				"pathtovols=s",
				"tardevnull",
				"tapemarks",
				"skipdbwrite",
				"nice:i",
		  );

	#
	# If --help was specified, just spit back the options available ...
	#

	if ( exists $opts{help} ) {
	print "\nUsage: archiver takes the following options:
	--help			- produce this help text
	--debug[=<level>]	- produce debugging output
				  (debugging level optional)
	--volspertape=<v>	- number of volumes to be written to each tape
	--db_file[=<dbfile>]	- use file DB, optionally specify file to use
	--changer=<chngr_dev>	- the tape changer device name
	--drive=<drive_dev>	- the tape drive device name
				  (w/o the drive number - i.e. '/dev/nrst')
	--drivenum=<dev_num>	- the tape drive number ( i.e. '0', '1', etc)
	--pathtovols=<path>	- the pathname where the volume directories
				  created by the pkt_dumper can be found
	--tardevnull		- for debugging; write tar files to /dev/null
	--tapemarks		- explicitly write tape marks
				  (the driver *should* do these by default)
	--skipdbwrite		- don't actually write the entry in the
				  database.  Use this while testing so that
				  all tapes look like scratch tapes!
	--nice[=<nice level>]	- for debugging; set the 'nice' level
	\n";

	# mysql not currently supported
	#	--db_mysql[=<table>]	- use mySQL DB, optionally specify the
	#				  table name to use
		exit 1;
	}

	#
	# Redirect STDOUT to a log file
	#

	# open (LOGF, '>>/var/log/archiver.log') ||
	# 	die "Opening /var/log/archiver.log";
	# *STDOUT = *LOGF;

	#
	# If --debug was specified, but $debug is zero, then set $debug to 1
	#
	if ( exists $opts{debug} ) {
		if ( $opts{debug} == 0 ) { $debug = 1; }
		else { $debug = $opts{debug} }
	}
	else { $debug = 0 }

	#
	# If --tardevnull was specified, then set $tar_to_dev_null to 1
	#
	if ( exists $opts{tardevnull} ) {
		if ( $opts{tardevnull} == 0 ) { $tar_to_dev_null = 1; }
		else { $tar_to_dev_null = $opts{tardevnull} }
	}
	else { $tar_to_dev_null = 0 }

	#
	# If --tapemarks was specified, then set $explicit_tape_marks to 1
	#
	if ( exists $opts{tapemarks} ) {
		if ( $opts{tapemarks} == 0 ) { $explicit_tape_marks = 1; }
		else { $explicit_tape_marks = $opts{tapemarks} }
	}
	else { $explicit_tape_marks = 0 }

	#
	# If --skipdbwrite was specified, then set $skip_db_write to 1
	#
	if ( exists $opts{skipdbwrite} ) {
		if ( $opts{skipdbwrite} == 0 ) { $skip_db_write = 1; }
		else { $skip_db_write = $opts{skipdbwrite} }
	}
	else { $skip_db_write = 0 }

	#
	# If --nice was specified without a value, then set $niceness to 4
	# otherwise use the value specified
	#
	if ( exists $opts{nice} ) {
		if ( $opts{nice} == 0 ) { $niceness = 4; }
		else { $niceness = $opts{nice} }
	}
	else { $niceness = 0 }

	#
	# If both db_file and db_mysql are specified, report an error.
	# If neither is specified, default to db_file
	#

	if ( exists $opts{db_file} && exists $opts{db_mysql} ) {
		die "Specify only one of db_file or db_mysql\n";
	}
	elsif ( !exists $opts{db_file} && !exists $opts{db_mysql} ) {
		$opts{db_file} = "";
	}

	if ( exists $opts{db_file} ) {
		if ( $opts{db_file} eq '' ) { $db_file = $DEFAULT_FILEDB_NAME }
		else { $db_file = $opts{db_file} }
	}
	else { $db_file = 0 }

	if ( exists $opts{db_mysql} ) {
		$db_mysql = 1;
		if ( $opts{db_mysql} eq '' )
			{ $table_name = $DEFAULT_MYSQL_TABLE }
		else
			{ $table_name = $opts{db_mysql} }
	}
	else { $db_mysql = 0 }

	#
	# Allow specification of number of volumes to be put on a tape
	#

	if ( exists $opts{volspertape} )
		{ $volumes_per_tape = $opts{volspertape} }
	else
		{ $volumes_per_tape = $DEFAULT_VOLUMES_PER_TAPE }

	#
	# Allow specification of the path where we should look for volumes
	#

	if ( exists $opts{pathtovols} )
		{ $volume_path_base = $opts{pathtovols} }
	else
		{ $volume_path_base = $DEFAULT_VOLUME_PATH }

	#
	# Allow command-line options to specify the changer device and
	# the tape drive to be used for this archiver.
	# If not specified, use the default values.
	#

	if ( exists $opts{changer} ) { $changer_device = $opts{changer} }
	else { $changer_device = $DEFAULT_CHANGER_DEVICE }

	if ( exists $opts{drive} ) { $tape_device_name = $opts{drive} }
	else { $tape_device_name = $DEFAULT_TAPE_DRIVE_DEVICE }

	if ( exists $opts{drivenum} ) { $tape_device_number = $opts{drivenum} }
	else { $tape_device_number = $DEFAULT_TAPE_DRIVE_NUMBER }

	$full_tape_device = $tape_device_name . $tape_device_number;

	#
	# Print out the results of command-line option processing
	#

	print scalar(localtime),
			": Starting archiver with the following options:\n
		debug		'$debug'
		db_file		'$db_file'
		changer		'$changer_device'
		drive		'$tape_device_name'
		drivenum	'$tape_device_number'
		volspertape	'$volumes_per_tape'
		pathtovols	'$volume_path_base'
		tardevnull	'$tar_to_dev_null'
		tapemarks	'$explicit_tape_marks'
		skipdbwrite	'$skip_db_write'
		nice		'$niceness'
	\n";
}

# my_sql DB currently not supported
#		db_mysql	'$table_name'

#------------------------------------------------------------------------
# Verify we are running as root 
#------------------------------------------------------------------------
sub verify_we_are_root {
	if ($UID != 0  || $EUID != 0) {
		print "Sorry, this program must be run as root.\n";
		print "(UID is $UID EUID is $EUID)\n";
		exit 1;
	}
}

#------------------------------------------------------------------------
# Verify we are the only instance of this script running!
#------------------------------------------------------------------------
sub verify_we_are_alone {
	my $key = 0;
	my $val;
	my ($semid, $return);
	my ($op1, $op2, $opstring);
	my @ascii;

	my $SEM_GET = 1;
	my $SEM_WAIT = 0;
	my $SEM_PUT = -1;


	# An admitted hack.  Use the combination of the
	# changer device name and tape drive number
	# to create a unique key for the semaphore.

	@ascii = unpack("C*", $changer_device);
	foreach $val (@ascii) {
		$key += $val;
	}

	# This should allow for 99 drives for each changer device
	$key *= 100;
	$key += $tape_device_number;

	# Get a handle to the semaphore.
	# If it does not yet exist, create it.

	$semid = semget($key, 1, 0);
	if (!defined($semid)) {
		if ($! == ENOENT) {
			# If it doesn't exist, create it
			print "Creating new mutex semaphore\n" if ($debug);
			$semid = semget($key, 1, IPC_CREAT | IPC_EXCL | 0666);
			if (!defined($semid)) {
 			       die "Unexpected error from semget (create): $!";
			}

			# Initialize the semaphore
			semctl($semid, 0, SETVAL, 0);
		}
		else {
			die "Unexpected error from semget: $!";
		}
	}

	# Now get/lock the semaphore.  The SEM_UNDO makes sure that
	# the mutex is released when our process goes away

	$op1 = pack("s*", 0, $SEM_WAIT, IPC_NOWAIT);
	$op2 = pack("s*", 0, $SEM_GET, IPC_NOWAIT | SEM_UNDO);
	$opstring = $op1 . $op2;

	if (! semop($semid, $opstring) ) {
		if ($! == EAGAIN) {
			die "There appears to already be an instance of $0 ",
			    "running for:\n\tdevice $changer_device",
			    "\n\tdrive $tape_device_number\n"
		}
		else {
			die "Unexpected error obtaining semaphore: $!";
		}
	}
}

#------------------------------------------------------------------------
# Print a message to both the terminal and STDERR
#------------------------------------------------------------------------
sub print_tty {
	my (@msg) = @_;
	my $tty = 1;

	local *TTY;
	open (TTY, "> /dev/tty") or $tty = 0;
	if ($tty) {
		print TTY @msg;
		close(TTY);
	}
	print STDERR @msg;
}

#------------------------------------------------------------------------
# Process an interrupt signal (sigint, a.k.a Ctrl-C)
#------------------------------------------------------------------------
sub SIGINT_Handler {
	print_tty scalar(localtime), ": Caught Ctrl-C (SIGINT)\n";
	print_tty scalar(localtime), ": Shutting down... \n";
	$time_to_quit = 1;
}

#------------------------------------------------------------------------
# Check the results of a system command
#------------------------------------------------------------------------
sub check_system_rc {
	my ($description, $rc, $dbg, @args) = @_;

	if ( $rc == 0) {
		print scalar(localtime), ": ", $description,
			" completed successfully!\n" if ($dbg);
		return 0;
	}
	elsif ( $rc == 0xff00 )  {
		print scalar(localtime), ": ", $description,
			" failed: !$\n";
		return 1;
	}
	elsif ( $rc > 0x80 ) {
		$rc >>= 8;
		print scalar(localtime), ": ", $description,
			" returned with status $rc\n";
		return 1;
	}
	else {
		my $coredump = 0;
		if ( $rc & 0x80 ) {
			$rc &= ~0x80;
			$coredump = 1;
		}
		printf scalar(localtime), ": ", $description,
			" ended from signal %d (%s a core dump)\n",
			$rc, $coredump ? "with" : "without";
		return 1;
	}
}

#------------------------------------------------------------------------
# Connect to the database, die if it fails
#------------------------------------------------------------------------
sub database_connect {
	my $db_spec = "DBI:mysql:vault";
	my $db_user = 'root';
	my $db_password = 'mysql';


	$dbh = DBI->connect($db_spec, $db_user, $db_password,
			{ RaiseError => 1, AutoCommit => 1 })
		or die "database_connect: Could not open database $db_spec: $DBI::errstr; stopped";
}

#------------------------------------------------------------------------
# Prepare all the SQL statements and get handles to them
#------------------------------------------------------------------------
sub sql_prepare {

	#
	# Update a record to ARCHIVING status, fill in the tapeid
	#
	$sql_update_status_to_archiving = "UPDATE $table_name SET tapeid=?,status='ARCHIVING' WHERE volid=?";

	#
	# Update a record to COMPLETE status
	#
	$sql_update_status_to_complete = "UPDATE $table_name SET status='COMPLETE' WHERE volid=?";

	#
	# This will select all fields of all records waiting to be filled.
	#
	$sql_select_by_status_full= "SELECT volid FROM $table_name WHERE status='FULL'";


	#
	# Prepare all the statements
	#

	$usta_sth = $dbh->prepare_cached($sql_update_status_to_archiving);
	$ustc_sth = $dbh->prepare_cached($sql_update_status_to_complete);
	$sbsf_sth = $dbh->prepare_cached($sql_select_by_status_full);
}

#------------------------------------------------------------------------
# Initialize use of the database (whichever database is being used)
#------------------------------------------------------------------------
sub database_init {
	if ($db_mysql) {
		require DBI;
		my $rv;

		# Tell the changer we're using mySQL
		$Changer::db_mysql = $db_mysql;

		# Initialize use of the database
		database_connect();
		sql_prepare();

		# Do changer-specific intialization
		$changer_ref->db_init($dbh, $table_name);
	}
	elsif ($db_file) {
		my $rv;

		# Tell the changer we're using a flat file DB
		$Changer::db_file = $db_file;
	}
	else {
		die "Some form of database must be selected!\n";
	}
}

#------------------------------------------------------------------------
# Update status of sql db record to 'ARCHIVING'
#------------------------------------------------------------------------
sub sql_update_to_archiving {
	my ($tapeid, $volid, @args) = @_;

	# Update record's status to ARCHIVING
	$usta_sth->execute($tapeid, $volid);
}

#------------------------------------------------------------------------
# Update status of sql db record to 'COMPLETE'
#------------------------------------------------------------------------
sub sql_update_to_complete {
	my ($volid, @args) = @_;

	# Update record's status to COMPLETE
	$ustc_sth->execute($volid);
}


#------------------------------------------------------------------------
# Return the directory path for the specified volid
#------------------------------------------------------------------------
sub volume_path {
	my ($volid, @args) = @_;

	return $volume_path_base . "/" . $volid;
}


#------------------------------------------------------------------------
# Parse all the "x" files containing statistics for each segment
# Accumulate values for the volume and return the final values
# in the supplied, referenced hash
#------------------------------------------------------------------------
sub process_stats_files {
	my ($volid, $hash_ref) = @_;
	my ($statfile, $xfile, @xfilelist);
	my ($key, @stats_keys);
	my $dirname;
	my $segments = 0;
	my ($first_start, $last_end, $delta_time);

	# These are the entries expected in the stats file
	@stats_keys = qw(
		starttime endtime
		lis_pkts lis_pkts_drop lis_bytes lis_bytes_snap
		lis_bytes_drop lis_bytes_per_sec
		dmp_pkts dmp_pkts_written dmp_bytes dmp_bytes_written
		dmp_bytes_per_sec);

	# Get a list of all the x:* files in the volume directory
	$dirname = volume_path($volid);
 	if ( opendir(VOLDIR, $dirname) ) {
		@xfilelist = grep { /^x:/ } readdir(VOLDIR);
		closedir(VOLDIR);
	}
	else {
		# We're not going to get very far if we can't open
		# the directory, but we need to set some values
		# and return...
		print "process_stats_files: could not opendir '$dirname': $!\n";
		foreach $key (@stats_keys) {
			$hash_ref->{$key} = -1;
		}
		return 1;
	}

	$first_start = 999999999999999999999;
	$last_end = 0;

	FILELOOP: foreach $xfile (@xfilelist) {
		$segments++;
		$statfile = $dirname . "/" . $xfile;
		print "process_stats_files: Processing file '$statfile'\n"
			if ($debug > 2);

		if ( ! open STATSFILE, $statfile ) {
			print "process_stats_files: Could not open $statfile: $!\n";
			next FILELOOP;
		}
		while (<STATSFILE>) {
			chop;			# Get rid of the newline
			# Split at the colon, return the key and value
			my ($junk, $key, $value) = split /^([^:]+):\s*/;
			print "process_stat_files: key $key, value $value\n"
				if ($debug > 2);

			# Keep track of the first starttime and
			# the last endtime, otherwise, accumulate
			# the values for all other keys

			if ($key eq "starttime" && $value < $first_start ) {
				$first_start = $value;
				$hash_ref->{$key} = $value;
			}
			elsif ($key eq "endtime" && $value > $last_end ) {
				$last_end = $value;
				$hash_ref->{$key} = $value;
			}
			else {
				print "process_stats_files: $key was $hash_ref->{$key}, adding $value\n" if ($debug > 5);
				$hash_ref->{$key} += $value;
				print "segment $segments: $key is now $hash_ref->{$key}\n" if ($debug > 5);
			}
		}
		close STATSFILE;

		# Verify we got all the right data
		while ($key = shift @stats_keys) {
			if (!exists $hash_ref->{$key} ) { 
				print "process_stats_files: No '$key' in $statfile!\n";
				next FILELOOP;
			}
		}
	}

	# now set the overall starttime, endtime, and rate values

	$delta_time = $last_end - $first_start;
	print "final: start $first_start, end $last_end, delta $delta_time\n"
		if ($debug > 2);
	$hash_ref->{starttime} = $first_start;
	$hash_ref->{endtime} = $last_end;
	$hash_ref->{segments} = $segments;
	if ( $delta_time != 0 ) {
		$hash_ref->{lis_bytes_per_sec} =
				$hash_ref->{lis_bytes} / $delta_time;
		printf "lis_bytes_per_sec = $hash_ref->{lis_bytes_per_sec}\n"
			if ($debug > 2);
	}
	if ( $segments != 0 ) {
		# XXX Average the dumper rate XXX
		$hash_ref->{dmp_bytes_per_sec} =
				$hash_ref->{dmp_bytes_per_sec} / $segments;
		printf "dmp_bytes_per_sec = $hash_ref->{dmp_bytes_per_sec}\n"
			if ($debug > 2);
	}

	return 0;
}


#------------------------------------------------------------------------
# Process a full volume
# 
# Return values:
#	0: Everything completed successfully
#	1: The volume was dumped, but there was an error
#	   cleaning up or creating the DB entry
#       2: The volume was not dumped.  It could not be found.
#       3: The volume was not dumped.  There may be a tape
#	   error.  Close the current tape and get a new one.
#------------------------------------------------------------------------
sub process_full_volume {
	my ($volid, $tapeid, $fileseq, @args) = @_;

	# DB Pre-processing
	if ($db_mysql) {
		# update the volume's status to archiving
		sql_update_to_archiving($volid);
	}

	# Make sure there is really a directory for the volume
	if ( ! -d volume_path($volid) ) {
		print scalar(localtime), ": Unable to locate volume directory ",
			volume_path($volid), " -- skipping!!\n";
		return 2;
	}

	#
	# Process the volume!
	#

	# Write out a "volnum" file
	write_volnum_file($volid);

	# Encrypt the key files and delete the cleartext versions
	print scalar(localtime), ": Encrypt symmetric keys for $volid\n";
	if (encrypt_symmetric_key($volid, "volKey") != 0 ||
	    encrypt_symmetric_key($volid, "transKey") != 0) {
		return 2;
	}

	print scalar(localtime), ": Begin archiving volume $volid on tape $tapeid\n";

	while ( !$time_to_quit && dump_data_to_tape($volid) ) {
		if ( semaphore_consume_retry($volid) ) {
			print scalar(localtime),
				": Too many errors trying to dump volid $volid\n";
			# Make sure there is a record in the database
			# for this tapeid so we don't continue to
			# try and use it.

			filedb_add_entry(-1, $tapeid, -1);
			return 3;
		}
	}
	if ($time_to_quit) {
		return 1;
	}

	print scalar(localtime), ": Done archiving volume $volid on tape $tapeid\n";

	# DB Post-processing
	if ($db_mysql) {
		# now update the volume's status to complete
		sql_update_to_complete($volid);
	}
	elsif ($db_file) {
		if ( filedb_add_entry($volid, $tapeid, $fileseq) ) {
			# Don't release the data or slot because
			# we were unable to write a DB entry
			print scalar(localtime), ": *** NOT RELEASING DATA OR SLOT for volumd $volid ***\n";
			return 1;
		}
	}

	# File System clean-up -- delete the data from disk
	if ( purge_data_from_disk($volid) ) {
		# Don't release the slot because the
		# data may still be there taking up space!
		print "*** NOT RELEASING SLOT for volume $volid ***\n";
		return 1;
	}

	# After data has been successfully deleted,
	# we can release the slot to a packet dumper
	semaphore_consume_complete($volid);

	return 0;
}

#------------------------------------------------------------------------
# Try a tar command that should fail.
# See if we properly get an error indication from it.
# Throw all the output away so we don't see the expected error messages.
#------------------------------------------------------------------------
sub tar_command_is_broken {
	my $rc;

	print "Testing the tar cmd for error indication ...\n";
	$rc = 0xffff & system("$tar_pgm -xf /dev/nst9 . >/dev/null 2>/dev/null");
	print "
	*******************************************************
	***            The tar command is broken!           ***
	***                                                 ***
	***    It failed to return an error indication.     ***
	*******************************************************
	\n" if ($rc == 0);
	print "tar_command_is_broken: return code was $rc\n" if ($debug) ;
	print "... tar correctly returned an error status\n\n" if ($rc != 0);
	return ! $rc;
}

#------------------------------------------------------------------------
# Write a tape mark to the tape to separate tar files and
# facilitate skipping forward or backwards on the tape
# via tape marks.
#------------------------------------------------------------------------
sub write_tape_mark {
	my ($count, @args) = @_;
	my $rc;

	print "write_tape_mark: '$mt_pgm -f $full_tape_device eof $count'\n"
		if ($debug);
	$rc = 0xffff & system($mt_pgm, "-f", $full_tape_device, "eof", $count);
	return ! $rc;
}

#------------------------------------------------------------------------
# Encrypt a volume's symmetric key
#------------------------------------------------------------------------
sub encrypt_symmetric_key {
	my ($volid, $keyfile, @args) = @_;

	my $rc;
	my $volpath = volume_path($volid);
	my $devname;

	print "encrypt_symmetric_key: encrypting $keyfile for volume $volid\n"
		if ($debug);

	if (! chdir $volpath ) {
		print "encrypt_symmetric_key: Unable to cd to $volpath\n";
		return 1;
	}

	# if the keyfile was already encrypted by a previous run, we're done
	if ( -f "$keyfile.gpg" ) {
		print "encrypt_symmetric_key: $keyfile already encrypted\n";

		# if the previous run failed to remove the cleartext,
		# remove it now
		if ( -f $keyfile ) {
			$rc = 0xffff & system("/bin/rm $keyfile");
			if ($rc != 0) {
				print scalar(localtime),
					"encrypt_symmetric_key: could not remove $keyfile\n";
				return 1;
			}
			return 0;
		}
		return 0;
	}

	print "The encrypt command is '$asymm_pgm --batch -e -r apv10 $keyfile'\n" if ($debug);

	$rc = 0xffff & system("$asymm_pgm --batch -e -r apv10 $keyfile");
	if (check_system_rc ($asymm_pgm, $rc, 0) ) {
		return 1;
	}

	print scalar(localtime),
		": $keyfile encryption completed successfully!\n";

	# make sure an encrypted keyfile was created
	if ( ! -f "$keyfile.gpg" ) {
		print "encrypt_symmetric_key: $asymm_pgm did not create encrypted keyfile $keyfile.pgp\n";
		return 1;
	}

	# remove the cleartext keyfile
	$rc = 0xffff & system("/bin/rm $keyfile");
	if ($rc != 0) {
		print scalar(localtime),
			"encrypt_symmetric_key: could not remove $keyfile\n";
		return 1;
	}

	return 0;
}


#------------------------------------------------------------------------
# Write a "volnum" file so the volume number
# can be restored upon retrieval
#------------------------------------------------------------------------
sub write_volnum_file {
	my ($volid, @args) = @_;

	my $volpath = volume_path($volid);

	if (! chdir $volpath ) {
		print "write_volnum_file: Unable to cd to $volpath\n";
		return 1;
	}

        # Write out a "volnum" file, so the volume id can
        # be restored upon retrieval
        if ( open(VOLNUM, "> volnum") ) {
                print VOLNUM $volid;
                close(VOLNUM);
        }
        else {   
                print scalar(localtime), ": Could not open volnum file: $!\n";
        }

	return 0;
}

#------------------------------------------------------------------------
# Dump a volume's data to tape
#------------------------------------------------------------------------
sub dump_data_to_tape {
	my ($volid, @args) = @_;

	my $rc;
	my $volpath = volume_path($volid);
	my $devname;
	my $pid;

	print "dump_data_to_tape: Dumping data for volume $volid\n" if ($debug);

	if (! chdir $volpath ) {
		print "dump_data_to_tape: Unable to cd to $volpath\n";
		return 1;
	}

	if ($tar_to_dev_null) {
		print "*****  Doing tar to /dev/null !!  *****\n";
		$devname = "/dev/null";
	}
	else {
		$devname = $full_tape_device;
	}

	# Fork off the tar command and wait for it to complete.
	# We ignore SIGINT within the tar/child process so that
	# it completes...

	print "The tar command is '$tar_pgm -cf $devname .'\n" if ($debug);
	if ($pid = fork) {
		# (PARENT) We'll catch SIGINT here and let'em
		# know that we are in the middle of a tar.
		local $SIG{INT} = sub {
			print_tty scalar(localtime),
				": Signal caught.  Tar command in progress.  ",
				"Delaying shutdown until complete ",
				"(could take a few minutes) ...\n";
			$time_to_quit = 1;
		};
		waitpid($pid, 0);
		$rc = $?;
	}
	else {
		die "Unable to fork: $!" unless defined $pid;
		# (CHILD) Ignore SIGINT and run the tar command
		$SIG{INT} = "IGNORE";
		exec($tar_pgm, "-cf", $devname, ".");
	}

	if ( $rc == 0) {
		print scalar(localtime),
			": tar command for $volid completed successfully!\n";
		return 0;
	}
	elsif ( $rc == 0xff00 )  {
		print scalar(localtime),
			": tar command for $volid failed: !$\n";
		return 1;
	}
	elsif ( $rc > 0x80 ) {
		$rc >>= 8;
		print scalar(localtime),
			": tar command for $volid returned with status $rc\n";
		return 1;
	}
	else {
		my $coredump = 0;
		if ( $rc & 0x80 ) {
			$rc &= ~0x80;
			$coredump = 1; 
		}
		printf "%s: tar command for %d ended from signal %d (%s a core dump)\n",
			scalar(localtime), $volid, $rc, $coredump ? "with" : "without";
		return 1;
	}
}

#------------------------------------------------------------------------
# Purge a volume's data from the disk
#------------------------------------------------------------------------
sub purge_data_from_disk {
	my ($volid, @args) = @_;

	my $volpath = volume_path($volid);

	print "purge_data_from_disk: Purging data in $volpath\n" if ($debug);

	if ( ! chdir $volpath ) {
		print "purge_data_from_disk: Unable to cd to $volpath";
		return 1;
	}
	my @failed = grep {not unlink} <*>;
	if (@failed) {
		print "purge_data_from_disk: Unable to remove @failed from $volpath: $!\n";
		return 1;
	}

	if ( ! rmdir($volpath) ) {
		print "purge_data_from_disk: Unable to rmdir $volpath: $!\n";
		return 1;
	}

	return 0;
}

#------------------------------------------------------------------------
# Add a record to the end of the file/DB for the
# volume we just processed
#------------------------------------------------------------------------
sub filedb_add_entry {
	my ($volid, $tapeid, $fileseq, @args) = @_;

	if ($skip_db_write) {
		print "NOT writing DB entry for volume $volid!\n";
		return 0;
	}

	my %stats_info;

	# ignore the return code from process_stats_files
	# it has been changed to do its best, and we need
	# to write a DB entry!

	process_stats_files($volid, \%stats_info);

	if ($debug) {
		my ($k, $v);
		print "After processing done file:\n";
		while ( ($k, $v) = each %stats_info ) {
			print "   $k => $v\n";
		}
	}

	if ( ! open(DBFILE, ">> $db_file") ) {
		print "filedb_add_entry: Unable to open $db_file: $!\n";
		return 1;
	}
	flock DBFILE, $LOCK_EXCLUSIVE;
	seek DBFILE, 0, 2;		# Seek to the end of the file

	printf DBFILE "%08d\t%s\t%04d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%05.3f\t%d\t%d\t%d\t%d\t%05.3f\n",
		$volid, $tapeid, $fileseq,
		$stats_info{starttime}, $stats_info{endtime},
		$stats_info{segments},
		$stats_info{lis_pkts}, $stats_info{lis_pkts_drop},
		$stats_info{lis_bytes}, $stats_info{lis_bytes_snap},
		$stats_info{lis_bytes_drop}, $stats_info{lis_bytes_per_sec},
		$stats_info{dmp_pkts}, $stats_info{dmp_pkts_written},
		$stats_info{dmp_bytes}, $stats_info{dmp_bytes_written},
		$stats_info{dmp_bytes_per_sec};

	flock DBFILE, $LOCK_UNLOCK;
	close DBFILE;

	return 0;
}


#------------------------------------------------------------------------
# Use the apvsync program to wait for and obtain a volume to be processed
#------------------------------------------------------------------------
sub semaphore_consume_get {
	my $consume_get_cmd = "$apvsync consume_get";
	my $get_output;
	my $volid;

	$get_output = `$consume_get_cmd`;
	if ( ($volid) = $get_output =~ /.*returning volid (\d+)/s) {
		print "semaphore_consume_get: got volid $volid\n" if ($debug);
		return $volid;
	}
	else {
		return -1;
	}
}

#------------------------------------------------------------------------
# Use the apvsync program to inform others that we're done with a volume
#------------------------------------------------------------------------
sub semaphore_consume_complete {
	my ($volid, @args) = @_;
	my @consume_complete_args = ($apvsync, "consume_complete", $volid);
	my $rc;

	$rc = 0xffff & system (@consume_complete_args);

	if ($rc) {
		print "semaphore_consume_complete: sync call failed ($rc)\n";
	}
}

#------------------------------------------------------------------------
# Use the apvsync program to inform others that we're done with a volume
#------------------------------------------------------------------------
sub semaphore_consume_retry {
	my ($volid, @args) = @_;
	my @consume_retry_args = ($apvsync, "consume_retry", $volid);
	my $rc;

	print "consume_retry doing '", @consume_retry_args, "'\n";

	$rc = 0xffff & system (@consume_retry_args);

	printf "consume_retry returned %d (0x%x)\n", $rc, $rc;

	if ($rc != 0) {
		print "semaphore_consume_retry:  call failed ($rc)\n";
	}
	return $rc;
}
