/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#define KERNEL_BPF_WRITE

#include        <stdio.h>
#include	<fcntl.h>
#include	<unistd.h>
#include	<string.h>
#include	<err.h>

#include        <sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/errno.h>
#include	<sys/ioctl.h>
#include	<sys/stat.h>

/* #include	"../net/bpf.h" */
/* #include	"/usr/src/sys/net/bpf.h" */
#include	"../sys/net/bpf.h"
#include	<net/if.h>

#include <sys/param.h>
#include <sys/mount.h>

#include	"bpf.h"

int p_fd = -1;
int w_fd = -1;

#define	PACK_SIZE	(4*64*1024)
char datum[PACK_SIZE];
int readLen = PACK_SIZE;

/*
** ==================================================
**
*/

#define	STATMAX		1000
struct bpf_stat2 bstat[STATMAX];
time_t btime[STATMAX];
int32_t bfree[STATMAX];
int bstat_i = 0;

/*
** ==================================================
**
** 	reading the interface on NetBsd.
**
*/


int intfInit( char *intf, char *file )
{
    int value;

    /*
    ** find a bpf interface which we can use....
    */
    {
	int which = 0;

	while (p_fd == -1)
	{
	    char name[128];
	    sprintf( name, "/dev/bpf%d", which++ );
	    p_fd = open( name, O_RDWR );
	}
    }

    if (file) {
	intfFileSave(file);
    } else {
	/*
	** Set the buffer size...
	*/
	value = PACK_SIZE;
	if (ioctl( p_fd, BIOCSBLEN, &value ) < 0) {
	    perror( "intfInit BIOCSBLEN error" );
	    exit(1);
	}
    }

    /*
    ** find the network interface which we can connect the bpf instance into...
    ** use the passed in name as the interface name...
    */

    {
	struct ifreq ifq[1];
	int s = socket( AF_INET, SOCK_DGRAM, 0 );

	memset( ifq, 0, sizeof(ifq) );
	strncpy( ifq->ifr_name, intf, sizeof(ifq->ifr_name) );

	if (ioctl(s, SIOCGIFFLAGS, ifq ) == -1) {
	   perror("if ioctl SIOCGIFFLAGS error");
	   exit(1);
	}
	close(s);

	if (ioctl( p_fd, BIOCSETIF, ifq ) < 0) {
	   perror("if ioctl BIOCGETIF error");
	   exit(1);
	}
    }

    /*
    if ((w_fd = open( "data", O_WRONLY | O_CREAT )) < 0) {
	perror( "intfInit: can't write data" );
    }
    */
    return 0;
}

/*
** --------------------------------------------------
**
** turn on promiscuous listening.
**
*/

int intfPromiscOn()
{
#ifdef NOPE
    int on = 1;
#endif
    if (ioctl( p_fd, BIOCPROMISC, 0 ) < 0) {
	perror( "intfPromiscOn: ioctl BIOCPROMISC failure" );
	exit(1);
    }
#ifdef NOPE
    if (ioctl( p_fd, BIOCIMMEDIATE, &on )  < 0) {
	perror( "intfPromiscOn: BIOCIMMEDIATE failure" );
	exit(1);
    }
#endif
    return 0;
}

/*
** --------------------------------------------------
**
** stuff data stream into the specified file...
** Calling the same function with zero for the file name
** turns off saving data into the file...
**
*/

int intfFileSave(char *f)
{
    static int onlyOnce = 1;

    if (w_fd != -1 && w_fd != -2) {
	close(w_fd);
    }

    if (!f)
	return 0;

    if (f) {
	if (strcmp(f, "/dev/null") == 0) {
		w_fd = -2;
		fprintf(stderr, "Telling kernel to just toss the data!\n");
	} else if ((w_fd = open( f, O_CREAT | O_TRUNC | O_RDWR, 0444 )) < 0) {
		perror( f );
		return(1);
	}
    }


#if defined(KERNEL_BPF_WRITE)
    /*
    ** This is the call that gives the bpf code the file descriptor
    ** of the place to write it's output.
    **/
    if (ioctl( p_fd, BIOCSTFD, &w_fd ) < 0) {
	perror( f );
	return(1);
    }
#endif	/* KERNEL_BPF_WRITE */

    /*
    ** If possible, set the bpf packet size to the basic block size
    ** of the file system...
    */
    if (onlyOnce && w_fd != -1 && w_fd != -2) {
	struct stat s[1];
	int value;

	onlyOnce = 0;

	if (fstat( w_fd, s ) < 0) {
	    perror( f );
	    return(1);
	}
	printf( "Basic buffer size: %d %d\n", s->st_blksize, p_fd );

	/*value = s->st_blksize;*/
	value = PACK_SIZE;
	printf( "set buffer size: %d\n", value );
	if (ioctl( p_fd, BIOCSBLEN, &value ) < 0) {
	    perror( "intfInit BIOCSBLEN error" );
	    exit(1);
	}
	if (ioctl( p_fd, BIOCGBLEN, &value ) < 0) {
	    perror( "intfInit BIOCGBLEN error" );
	    exit(1);
	}
	printf( "get buffer size: %d\n", value );
	readLen = value;
    }
    return(0);
}

/*
** --------------------------------------------------
**
** intfRead.
**
*/

int totLen = 0;
int totPackets = 0;


int intfRead( int (*callOut)() )
{
    int len;
    char * d = datum;

    if ((len = read( p_fd, d, readLen )) < 0) {
	perror( "intfRead data read failure" );
	exit(1);
    }

    totLen += len;
    return 0;
}

/*--------------------------------------------------
 * Read bpf statistics from the kernel,
 * adding them to the bstat array.
 */
int intfStat()
{
    time_t now = time(0);
    struct statfs statfsBuf;

    if (bstat_i < (sizeof bstat / sizeof bstat[0])) {
	if (ioctl( p_fd, BIOCGSTATS2, &bstat[bstat_i++] ) < 0)
	    warn("intfStat BIOCGSTATS2 failure" );
	if (fstatfs( w_fd, &statfsBuf ) < 0)
	    warn("intfStat fstatfs failure" );
	btime[bstat_i-1] = now;
	bfree[bstat_i-1] = statfsBuf.f_bavail*statfsBuf.f_bsize/(1024*1024);
	totPackets = bstat[bstat_i-1].bs_recv;
    }
    return 0;
}

/*--------------------------------------------------
 * Print out the bpf statistics collected from
 * the kernel.
 *
 * Reset the "baseline" entry to be the last entry
 * that we collected.  This prevents a problem
 * encountered where we were calling intfStatDump
 * and then making a separate call to intfStat,
 * losing numbers in between the two calls.
 */
int intfStatDump()
{
    int i, Bps, lbfree;
    /* printf("intfStatDump: bstat_i is %d\n", bstat_i); */
    for (i=1; i<bstat_i; ++i) {
	struct tm *tm_p = localtime(&btime[i]);
	if (btime[i]-btime[i-1]) {
	    Bps = (bstat[i].bs_brecv-bstat[i-1].bs_brecv)/(btime[i]-btime[i-1]);
	} else {
	    Bps = (bstat[i].bs_brecv-bstat[i-1].bs_brecv); /* round up to 1s */
	}
	printf("intfStat: %02d/%02d/%02d %02d:%02d:%02d dt:%d pr:%u(+%u) br:%qu(+%qu) bs:%qu(+%qu) pd:%u(+%u) bd:%qu(+%qu) Bps:%d Mbps:%d mf:%dM\n",
	    tm_p->tm_mon+1,
	    tm_p->tm_mday,
	    tm_p->tm_year-100,
	    tm_p->tm_hour,
	    tm_p->tm_min,
	    tm_p->tm_sec,
	    btime[i]-btime[i-1],
	    bstat[i].bs_recv, bstat[i].bs_recv-bstat[i-1].bs_recv,
	    bstat[i].bs_brecv, bstat[i].bs_brecv-bstat[i-1].bs_brecv,
	    bstat[i].bs_bsrecv, bstat[i].bs_bsrecv-bstat[i-1].bs_bsrecv,
	    bstat[i].bs_drop, bstat[i].bs_drop-bstat[i-1].bs_drop,
	    bstat[i].bs_bdrop, bstat[i].bs_bdrop-bstat[i-1].bs_bdrop,
	    Bps,
	    Bps*8/(1024*1024),
	    bfree[i]);
    }
    bstat[0] = bstat[bstat_i-1];	/* save current numbers as baseline */
    btime[0] = btime[bstat_i-1];	/* save time values too! */
    lbfree = bfree[bstat_i-1];
    bstat_i = 1;
    return lbfree;
}

/*--------------------------------------------------
 * Write out statistics for a segment to a file.
 * These are used later to fill in the database
 * entry for the segment.
 */
int intfWriteXFile(char *filename)
{
    int n;
    u_int64_t bps;
    FILE *file;

    /* printf("intfWriteXFile: file '%s' bstat_i = %d\n", filename, bstat_i); */
    if ((file = fopen( filename, "w+" )) == NULL) {
	perror( filename );
	exit(1);
    }
    /*
     * bstat_i is the next index to be written; so we want to calculate
     * the difference between the previous index and the first index (0).
     *
     * Note that the bstat numbers are continually increasing.
     * This is not going to work in the long term??? XXX
     */
    n = bstat_i - 1;
    if (btime[n]-btime[0]) {
	bps = (bstat[n].bs_brecv-bstat[0].bs_brecv)/(btime[n]-btime[0]);
    } else {
	bps = (bstat[n].bs_brecv-bstat[0].bs_brecv); /* round up to 1s */
    }
    fprintf(file, "%u %u %u %u %qu %qu %qu %qu\n",
		btime[0], btime[n],			/* starttime, endtime */
		bstat[n].bs_recv-bstat[0].bs_recv,	/* packets */
		bstat[n].bs_drop-bstat[0].bs_drop,	/* packets_dropped */
		bstat[n].bs_brecv-bstat[0].bs_brecv, 	/* bytes */
		bstat[n].bs_bsrecv-bstat[0].bs_bsrecv, 	/* bytes snapped */
		bstat[n].bs_bdrop-bstat[0].bs_bdrop, 	/* bytes dropped */
		bps                                     /* bytes per sec */
	   );
    fclose(file);
    return 0;
}


/*
** --------------------------------------------------
**
** close down the intf.
**
*/

int intfClose()
{
    static struct bpf_stat2 bs[1];
    time_t now = time(0);
    struct tm *tm_p = localtime(&now);

    putchar('\n');
    intfStatDump();

    if (ioctl( p_fd, BIOCGSTATS2, bs ) < 0)
	warn("intfClose BIOCGSTATS2 failure");

    totPackets = bs->bs_recv;
    printf("intfClose: %02d/%02d/%02d %02d:%02d:%02d pr:%u br:%qu bs:%qu pd:%u bd:%qu\n",
	tm_p->tm_mon+1, tm_p->tm_mday, tm_p->tm_year-100,
	tm_p->tm_hour, tm_p->tm_min, tm_p->tm_sec,
	bs->bs_recv, bs->bs_brecv, bs->bs_bsrecv, bs->bs_drop, bs->bs_bdrop);
    close(p_fd);

    if (w_fd > 0) {
	close(w_fd);
	w_fd = -1;
    }
    return 0;
}
