/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include "../crypto/crypto.h"
#include "list.h"

extern int cryptalg;

struct list * newList( struct list * l )
{
    struct list * new = (struct list *)malloc( sizeof(*new) );
    assert(new);
    new->l_cnt = 1;
    if (l) {
	new->l_next = l->l_next;
	l->l_next = new;
    }
    return new;
}

/*
** ------------------------------------------------------------
**
**
*/

void insert_substitution(struct list * l,
			 struct in_addr * real_addr,
			 struct in_addr * fake_addr) {
  struct list * v;

  v = newList(l);
  v->l_type = L_TYPE_SUBST;
  memcpy(&v->contents.subst.real_addr,real_addr,
	 sizeof(v->contents.subst.real_addr));
  memcpy(&v->contents.subst.fake_addr,fake_addr,
	 sizeof(v->contents.subst.fake_addr));
}

int find_real_ip(struct list * l, in_addr_t * addr) {
  while (l) {
    if ((l->l_type == L_TYPE_SUBST)
	&& (l->contents.subst.fake_addr.s_addr == *addr)) {
      l->l_cnt++;
      *addr = l->contents.subst.real_addr.s_addr;
      return(1);
    }
    l = l->l_next;
  }
  return(0);
}

struct list * find_substitution( struct list * l, in_addr_t addr )
{
    while(l) {
	if ((l->l_type == L_TYPE_SUBST)
	    && (l->contents.subst.real_addr.s_addr == addr)) {
	    l->l_cnt++;
	    break;
	}
	l = l->l_next;
    }
    return(l);
}

in_addr_t new_substitution(struct list * l, in_addr_t addr)
{
    struct list * v;

    if ((v = find_substitution(l, addr)) == 0) {
	v = newList( l );
	v->l_type = L_TYPE_SUBST;
	v->contents.subst.real_addr.s_addr = addr;
	{
	  static int nth = 0;
	  int substaddr = htonl(nth);
	  v->contents.subst.fake_addr.s_addr = substaddr;
	  nth++;
	}
    }
    return v->contents.subst.fake_addr.s_addr;
}

/*
** ------------------------------------------------------------
**
**
*/

struct list * find_conversation(struct list * l, in_addr_t addrA,
				in_addr_t addrB) {

  while(l) {
    if ((l->l_type == L_TYPE_CONV)
	&& (l->contents.conv.addr_a.s_addr == addrA)
	&& (l->contents.conv.addr_b.s_addr == addrB)) {
      l->l_cnt++;
      break;
    }
    l = l->l_next;
  }
  return(l);
}

struct crypto_session *
new_conversation(struct list * l,
		 in_addr_t fake_addrA, in_addr_t fake_addrB,
		 struct ip * ip,
		 struct crypto_session * vol_key_session)
{
    char * conv_key;
    struct list * v;

    if ((v = find_conversation(l, fake_addrA, fake_addrB )) == 0) {
      v = newList( l );
      v->l_type = L_TYPE_CONV;

      v->contents.conv.addr_a.s_addr = fake_addrA;
      v->contents.conv.addr_b.s_addr = fake_addrB;

      conv_key = make_conv_key(vol_key_session, ip, cryptalg);
      get_crypto_session(&v->contents.conv.session,
			 cryptalg,CBC_MODE,ENCRYPT,conv_key,1);
    }
    return &v->contents.conv.session;
}

/*
** ------------------------------------------------------------
**
*/

/* get_substitution_table gathers together the entries in the
   substitution table, which are kept in a linked list, into a single
   buffer to be encrypted. */
void get_substitution_table(struct list * l, int max_packets,
			    char ** buffer, int * length) {
  int allocated, record_size;

  record_size = sizeof(l->contents.subst);
  /* Sloppy overestimate of the amount of space we'll need;
     note that each packet may contain at worst two new
     IP addresses, and that the padding added by the encryption
     must be less than MAX_BLOCK_LENGTH.*/
  allocated = 2*max_packets*record_size + MAX_BLOCK_LENGTH;
  if (!(*buffer = (char *)malloc(allocated))) {
    perror("couldn't allocate memory to copy substitution table");
    exit(1);
  }
  *length = 0;
  while(l) {
    if (l->l_type == L_TYPE_SUBST) {
      memcpy(*buffer+*length,&l->contents.subst,record_size);
      *length += record_size;
    }
    l = l->l_next;
  }
  if (*length+MAX_BLOCK_LENGTH > allocated) {
    printf("Too many packets in get_substitution_table.\n");
    exit(1);
  }
};

/* get_conversation_list gathers together the entries in the
   conversation list, which are kept in a linked list, into a single
   buffer to be encrypted. */
void get_conversation_list(struct list * l, int max_packets,
			   char ** buffer, int * length) {
  int allocated, record_size;

  record_size = 2*sizeof(l->contents.conv.addr_a);
  /* Sloppy overestimate of the amount of space we'll need, including
     crypto padding: */
  allocated = max_packets*record_size + MAX_BLOCK_LENGTH;
  if (!(*buffer = (char *)malloc(allocated))) {
    perror("couldn't allocate memory to copy conversation list");
    exit(1);
  }
  *length = 0;
  while(l) {
    if (l->l_type == L_TYPE_CONV) {
      memcpy(*buffer+*length,&l->contents.conv.addr_a,record_size);
      *length += record_size;
    }
    l = l->l_next;
  }
  if (*length+MAX_BLOCK_LENGTH > allocated) {
    printf("Too many packets in get_conversation_list.\n");
    exit(1);
  }
};


/*
** ------------------------------------------------------------
**
**
*/

char * geth( struct in_addr * i )
{
    struct hostent * h = 0;
    /* h = gethostbyaddr( (char *)i, 4, 2 ); not for timming */
    if (h) {
	return(h->h_name);
    } else {
	return("<none>");
    }
}

void walkList( struct list * l, void (*p)(), void * rock )
{
    while(l) {
	p(l, rock);
	l = l->l_next;
    }
}

void printList( struct list * l, void * rock )
{
    switch (l->l_type) {
	case L_TYPE_NULL:
	    printf( "-null- %d\n", l->l_cnt );
	    break;

	case L_TYPE_SUBST:
	    {
	    printf( "%s %s ",
		inet_ntoa(l->contents.subst.fake_addr),
		geth(&l->contents.subst.fake_addr) );

	    printf( "%s %s [%d]\n",
		inet_ntoa(l->contents.subst.real_addr),
		geth(&l->contents.subst.real_addr),
		l->l_cnt );
	    }
	    break;
	
	case L_TYPE_CONV:
	    {
	    printf( "%s %s :",
		inet_ntoa(l->contents.conv.addr_a),
		geth(&l->contents.conv.addr_a) );
	    printf( "%s %s [%d]\n",
		inet_ntoa(l->contents.conv.addr_b),
		geth(&l->contents.conv.addr_b),
		l->l_cnt );
	    }
	    break;
    }
}
