/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <unistd.h>
#include <stdlib.h>
#include "../crypto/crypto.h"
#include "db.h"

extern int cryptalg;

#define	DATABASE	"access.db"
#define	DATABASE2	"access2.db"

/*
 * Create a new database.
 *
 * Function/Operation:
 *	Creates a database instance.  Although a file is created for each
 *	database instance by the database subsystem, the database is stored
 *	in memory and never written to disk.  The database page and cache
 *	sizes were determined experimentally.
 *
 * Parameters:
 *	const char *db_name	File name of the database.
 *
 * Returns:
 *	DB 			Handle of created database.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */
DB *
newDB(const char *db_name)
{

	DB *dbp;
	int rc, pagesize, cachesize;

	(void)unlink(db_name);

	if ((rc = db_create(&dbp, NULL, 0)) != 0) {
		printf("db_create: %s\n", db_strerror(rc));
		exit(1);
	}

	dbp->set_errfile(dbp, stderr);
	dbp->set_errpfx(dbp, db_name);	

	pagesize = 8192;
	if ((rc = dbp->set_pagesize(dbp, pagesize)) != 0) {
		dbp->err(dbp, rc, "set_pagesize");
		exit(1);
	}

	cachesize = 8 * 1024 * 1024;
	if ((rc = dbp->set_cachesize(dbp, 0, cachesize, 0)) != 0) {
		dbp->err(dbp, rc, "set_cachesize");
		exit(1);
	}

	if ((rc = dbp->open(dbp, db_name, NULL, DB_BTREE, DB_CREATE, 0664)) != 0) {
		dbp->err(dbp, rc, "open: %s", db_name);
		exit(1);
	}

	return dbp;
}

/*
 * Add an entry into the substitution table database.
 *
 * Function/Operation:
 *	Establishes the mapping between a real and a substituted IP address.
 *	The real address is looked up in the database.  If not found, a
 *	new substituted address is generated and the real and substituted
 *	address pair are entered into the database.  In both cases the
 *	substituted address is returned.
 *
 *	User-specified buffers are used with the database routines
 *	to improve performance.
 *
 * Parameters:
 *	DB *dbp			Database handle.
 *	in_addr_t real_addr	IP address to be added.
 *
 * Returns:
 *	in_addr_t		Substituted address.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */
in_addr_t
newDB_substitution(DB *dbp, in_addr_t real_addr)
{
	static DBT key, data;
	in_addr_t db_key, db_data;
	static in_addr_t subs_addr = 0;
	int rc;

	key.data = &db_key;
	key.size = key.ulen = sizeof db_key;
	data.data = &db_data;
	data.size = data.ulen = sizeof db_data;
	key.flags |= DB_DBT_USERMEM;
	data.flags |= DB_DBT_USERMEM;

	db_key = real_addr;

	if ((rc = dbp->get(dbp, NULL, &key, &data, 0)) == 0) {
		return db_data;
	} else if (rc ==  DB_NOTFOUND) {
		db_data = htonl(subs_addr++);
		if ((rc = dbp->put(dbp, NULL, &key, &data, 0)) == 0) {
			return db_data;
		}
		else {
			dbp->err(dbp, rc, "DB->put");
			exit(1);
		}
	} else {
		dbp->err(dbp, rc, "DB->get");
		exit(1);
	}
}

extern
void insertDB_substitution(DB *dbp,in_addr_t real_addr,in_addr_t fake_addr) {
  	static DBT key, data;
	in_addr_t db_key, db_data;
	int rc;

	key.data = &db_key;
	key.size = key.ulen = sizeof db_key;
	data.data = &db_data;
	data.size = data.ulen = sizeof db_data;
	key.flags |= DB_DBT_USERMEM;
	data.flags |= DB_DBT_USERMEM;

	db_key = fake_addr;
	db_data = real_addr;
	if ((rc = dbp->put(dbp, NULL, &key, &data, 0)) != 0) {
	        dbp->err(dbp, rc, "DB->put");
		exit(1);
	}
}

int
findDB_real_ip(DB *dbp, in_addr_t * addr) {
	static DBT key, data;
	in_addr_t db_key, db_data;
	int rc;

	key.data = &db_key;
	key.size = key.ulen = sizeof db_key;
	data.data = &db_data;
	data.size = data.ulen = sizeof db_data;
	key.flags |= DB_DBT_USERMEM;
	data.flags |= DB_DBT_USERMEM;

	db_key = *addr;

	if ((rc = dbp->get(dbp, NULL, &key, &data, 0)) == 0) {
	        *addr = db_data;
		return(1);
	} else if (rc ==  DB_NOTFOUND) {
		return(0);
	} else {
		dbp->err(dbp, rc, "DB->get");
		exit(1);
	}

}


/*
 * Retrieve all entries from the substitution table database.
 *
 * Function/Operation:
 *	Gets the number of records in database, allocates a buffer via
 *	malloc(), and reads all records in the database into the buffer.
 *	Both key and data values for each record are stored in the buffer.
 *	The buffer is allocated with some extra trailing space to permit
 *	encryption of the buffer contents later (we overestimate these
 *	crypto padding requirements a bit).
 *
 *	User-specified buffers are used with the database routines
 *	to improve performance.
 *
 *	It is the responsibility of the caller to free() the buffer
 *	when it is no longer needed.
 *
 * Parameters:
 *	DB *dbp			Handle of substitution table database.
 *	int max_list_length	This parameter is historical and is ignored.
 *	char **buffer		Pointer to location where returned memory
 *				buffer address is stored.
 *	int *length		Pointer to location where length of returned
 *				memory buffer is stored (exclusive of crypto
 *				padding).
 * Returns:
 *	int			Zero.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */

int
getDB_substitution_table(DB *dbp, int max_list_length,
			    char ** buffer, int * length)
{
	DBC *dbcp;
	DB_BTREE_STAT *statp;
	static DBT key, data;
	int rc, nrec;
	int allocated;

	if ((rc = dbp->stat(dbp, &statp, NULL, 0)) != 0) {
		dbp->err(dbp, rc, "DB->stat");
		exit(1);
	}
	nrec = (u_long)statp->bt_ndata;
	free(statp);
	allocated = 2*sizeof(in_addr_t)*nrec + MAX_BLOCK_LENGTH;
	if (!(*buffer = (char *)malloc(allocated))) {
		perror("couldn't allocate memory to copy substitution table");
		exit(1);
	}
	*length = 0;

	if ((rc = dbp->cursor(dbp, NULL, &dbcp, 0)) != 0) {
		dbp->err(dbp, rc, "DB->cursor");
		exit(1);
	}

	key.data = *buffer;
	key.size = key.ulen = sizeof(in_addr_t);
	data.data = *buffer + sizeof(in_addr_t);
	data.size = data.ulen = sizeof(in_addr_t);
	key.flags |= DB_DBT_USERMEM;
	data.flags |= DB_DBT_USERMEM;

	while ((rc = dbcp->c_get(dbcp, &key, &data, DB_NEXT)) == 0) {
		*length += 2*sizeof(in_addr_t);
		key.data += 2*sizeof(in_addr_t);
		data.data += 2*sizeof(in_addr_t);
	}

	if (*length+MAX_BLOCK_LENGTH > allocated) {
		printf("Too many packets in get_substitution_table.\n");
		exit(1);
	}

	if (rc != DB_NOTFOUND) {
		dbp->err(dbp, rc, "DB->c_get");
		exit(1);
	}

	if ((rc = dbcp->c_close(dbcp)) != 0) {
		dbp->err(dbp, rc, "DB->c_close");
		exit(1);
	}

	return 0;
}

/*
 * Creates a new conversation key, if necessary.
 *
 * Add the substituted IP addresses into the conversation list database,
 * and create a new conversation key given the real IP addresses.
 *
 * Function/Operation:
 *	Establishes a conversation key for each unique pair of IP addresses.
 *	A key consisting of a pair of substituted IP addresses (sorted in
 *	ascending order to prevent duplicates) is looked up in the database.
 *	If not found, a conversation key is generated using the corresponding
 *	pair of real IP addresses and stored in the database.  In both cases
 *	the conversation key crypto session is returned (in a static buffer,
 *	to improve performance).
 *
 *	The address sorting function is arbitrary.
 *
 *	User-specified buffers are used with the database routines
 *	to improve performance.
 *
 * Parameters:
 *	DB *dbp			Handle of substitution table database.
 *	in_addr_t subs_addrA	First substituted IP address.
 *	in_addr_t subs_addrB	Second substituted IP address.
 *	in_addr_t real_addrA	First real IP address.
 *	in_addr_t real_addrB	Second real IP address.
 *	struct crypto_session * vol_key_session
 *                              Pointer to volume key crypto session.
 *
 * Returns:
 *	struct crypto_session *	Pointer to conversation key crypto session.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */

struct crypto_session *
newDB_conversation(DB *dbp,
	in_addr_t subs_addrA, in_addr_t subs_addrB,
	struct ip * ip,
	struct crypto_session * vol_key_session)
{
	static DBT key, data;
	in_addr_t db_key[2];
	static struct crypto_session db_data;
	int rc;
	char * conv_key;

	key.data = &db_key;
	key.size = key.ulen = sizeof db_key;
	data.data = &db_data;
	data.size = data.ulen = sizeof db_data;
	key.flags |= DB_DBT_USERMEM;
	data.flags |= DB_DBT_USERMEM;

	db_key[0] = subs_addrA;
	db_key[1] = subs_addrB;

	if ((rc = dbp->get(dbp, NULL, &key, &data, 0)) == 0) {
		return &db_data;
	} else if (rc ==  DB_NOTFOUND) {
	        conv_key = make_conv_key(vol_key_session, ip, cryptalg);
		get_crypto_session(&db_data, cryptalg, CBC_MODE, ENCRYPT,
				   conv_key,1);
		if ((rc = dbp->put(dbp, NULL, &key, &data, 0)) == 0) {
			return &db_data;
		}
		else {
			dbp->err(dbp, rc, "DB->put");
			exit(1);
		}
	} else {
		dbp->err(dbp, rc, "DB->get");
		exit(1);
	}
}

/*
 * Retrieve all entries from the conversation list database.
 *
 * Function/Operation:
 *	Gets the number of records in database, allocates a buffer via
 *	malloc(), and reads all records in the database into the buffer.
 *	Only the key values for each record are stored in the buffer.
 *	The buffer is allocated with some extra trailing space to permit
 *	encryption of the buffer contents later (we overestimate these
 *	crypto padding requirements a bit).
 *
 *	User-specified buffers are used with the database routines
 *	to improve performance, as is a partial (data=NULL, size=0,
 *	dlen=0, doff=0) retrieval of the data records, since only the
 *	keys are needed here.
 *
 *	It is the responsibility of the caller to free() the buffer
 *	when it is no longer needed.
 *
 * Parameters:
 *	DB *dbp			Handle of conversation table database.
 *	int max_list_length	This parameter is historical and is ignored.
 *	char **buffer		Pointer to location where returned memory
 *				buffer address is stored.
 *	int *length		Pointer to location where length of returned
 *				memory buffer is stored (exclusive of crypto
 *				padding).
 * Returns:
 *	int			Zero.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */

int
getDB_conversation_list(DB *dbp, int max_list_length,
			    char ** buffer, int * length)
{
	DBC *dbcp;
	DB_BTREE_STAT *statp;
	static DBT key, data;
	int rc, nrec;
	int allocated;

	if ((rc = dbp->stat(dbp, &statp, NULL, 0)) != 0) {
		dbp->err(dbp, rc, "DB->stat");
		exit(1);
	}
	nrec = (u_long)statp->bt_ndata;
	free(statp);
	allocated = 2*sizeof(in_addr_t)*nrec + MAX_BLOCK_LENGTH;
	if (!(*buffer = (char *)malloc(allocated))) {
		perror("couldn't allocate memory to copy substitution table");
		exit(1);
	}
	*length = 0;

	if ((rc = dbp->cursor(dbp, NULL, &dbcp, 0)) != 0) {
		dbp->err(dbp, rc, "DB->cursor");
		exit(1);
	}

	key.data = *buffer;
	key.size = key.ulen = 2*sizeof(in_addr_t);
	key.flags |= DB_DBT_USERMEM;
	data.flags |= (DB_DBT_USERMEM|DB_DBT_PARTIAL);

	while ((rc = dbcp->c_get(dbcp, &key, &data, DB_NEXT)) == 0) {
		*length += 2*sizeof(in_addr_t);
		key.data += 2*sizeof(in_addr_t);
	}

	if (*length+MAX_BLOCK_LENGTH > allocated) {
		printf("Too many packets in get_conversation_list.\n");
		exit(1);
	}

	if (rc != DB_NOTFOUND) {
		dbp->err(dbp, rc, "DB->c_get");
		exit(1);
	}

	if ((rc = dbcp->c_close(dbcp)) != 0) {
		dbp->err(dbp, rc, "DB->c_close");
		exit(1);
	}

	return 0;
}

/*
 * Closes a database.
 *
 * Function/Operation:
 *	Closes a database instance.  The database is not stored to disk
 *	before closing.
 *
 * Parameters:
 *	DB *dbp			Handle of created database.
 *
 * Returns:
 *	None.
 *
 * Errors:
 *	All errors cause an error message and immediate exit.
 */

void
closeDB(DB *dbp)
{
	int rc;

	if ((rc = dbp->close(dbp, DB_NOSYNC)) != 0) {
		printf("DB->close: %s\n", db_strerror(rc));
		exit(1);
	}
	return;
}










