/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#ifndef _APV_CRYPTO_H_
#define _APV_CRYPTO_H_

#include "../des/des.h"
#include "rijndael/rijndael-alg-fst.h"

/* operations: */
#define ENCRYPT 0
#define DECRYPT 1

/* algorithms: */
#define NONE 0
#define DESX 1
#define AES  2
#define DEFAULT_ALGORITHM 2

/* chaining options: */
#define ECB_MODE 0
#define CBC_MODE 1

#define MAX_BLOCK_LENGTH 16
#define MAX_KEY_LENGTH 24

struct algorithm_struct {
  char * name;
  int block_length; /* in bytes */
  int key_length; /* in bytes */
  int key_schedule_length; /* in bytes */
};

/* The number of rounds used by Rijndael is
   10 if we use 128-byte blocks and 128-byte keys
   12 if we use 192-byte blocks or 192-byte keys
   14 if we use 256-byte blocks or 256-byte keys: */
#define AES_ROUNDS 10

#define NUM_ALGORITHMS 3

extern struct algorithm_struct algorithms[NUM_ALGORITHMS];

struct desx_ks_struct {
  char prewhitening[8];
  DES_KS des;
  char postwhitening[8];
};

typedef struct desx_ks_struct * desx_ks;

struct crypto_session {
  int algorithm;
  int chaining;
  int operation;
  char key[MAX_KEY_LENGTH];
  union {
    struct desx_ks_struct desx;
    DES_KS des;
    u32 aes[4*(AES_ROUNDS+1)];
  } key_schedule;
};

/* get_crypto_session: algorithm and chaining must be one of constants
   defined above, and operation must be either ENCRYPT or DECRYPT
   (defined above).  Returns session (which must already have memory
   allocated for it). Make save_key nonzero if you want a copy of the
   key kept in crypto_session. */
void get_crypto_session(struct crypto_session * session,
			int algorithm, int chaining,
			int operation, char * key, int save_key);

/* crypt_buffer: operates on buffer in place, doing either encryption
   or decryption, depending on which was requested by the call to
   get_crypto_session that returned session.  The number of bytes
   encrypted is returned; note that this will be longer than length
   (the length of the original buffer) if length is not a multiple of
   the algorithm's block size.  You can call get_encrypted_length,
   below, if you need to determine this number beforehand. The iv
   will be updated if there's chaining. */
int crypt_buffer(struct crypto_session * session,
		 char * buffer, int length, char * iv);

/* get_encrypted_length tells you how big a buffer crypt_buffer would
   need to encrypt length bytes using session.*/
int get_encrypted_length(struct crypto_session * session, int length);

/* Writes a key appropriate for the given algorithm to a file named
   keyfile.  Uses /dev/random to get randomness. Dies on error.*/
void write_random_key(char * keyfile, int algorithm);

/* The next three functions use session to generate keys in a
   deterministic way from the ip addresses in the given ip packet.
   They return a pointer to the generated key, which will have length
   appropriate for cryptalg.  The pointer is to a buffer which will be
   overwritten next time the same function is called, so the caller
   must use it (or copy it to a buffer of its own) immediately. */
char * make_conv_key(struct crypto_session * session,
		     struct ip * ip, int cryptalg);

char * make_src_key(struct crypto_session * session,
		    struct ip * ip, int cryptalg);

char * make_dst_key(struct crypto_session * session,
		    struct ip * ip, int cryptalg);

/* Print a list of algorithms, for the user's convenience. */
void print_algorithms();

/* Read a key file.  It is assumed that space is already allocated for the
   key. */
void read_key_file(char * filename, char * key, int cryptalg);

/* Save a buffer to a file using the format
     iv, unencrypted length of buffer, encrypted buffer
   The iv is chosen by function, not the caller. */
void save_cbc_encrypted_buffer(struct crypto_session * session,
			       char * filename,
			       char * buffer, int length);

/* Reads a buffer from a file in the format written by the previous
   function. */
void read_cbc_encrypted_buffer(struct crypto_session * session,
			       char * filename,
			       char ** buffer, int * length);


#endif /* _APV_CRYPTO_H_ */
