/*
 * COPYRIGHT    2001
 * THE REGENTS OF THE UNIVERSITY OF MICHIGAN
 * ALL RIGHTS RESERVED
 *
 * Permission is granted to use, copy, create derivative works
 * and redistribute this software and such derivative works
 * for any purpose, so long as the name of The University of
 * Michigan is not used in any advertising or publicity
 * pertaining to the use of distribution of this software
 * without specific, written prior authorization.  If the
 * above copyright notice or any other identification of the
 * University of Michigan is included in any copy of any
 * portion of this software, then the disclaimer below must
 * also be included.
 *
 * THIS SOFTWARE IS PROVIDED AS IS, WITHOUT REPRESENTATION
 * FROM THE UNIVERSITY OF MICHIGAN AS TO ITS FITNESS FOR ANY
 * PURPOSE, AND WITHOUT WARRANTY BY THE UNIVERSITY O
 * MICHIGAN OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING
 * WITHOUT LIMITATION THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 * REGENTS OF THE UNIVERSITY OF MICHIGAN SHALL NOT BE LIABLE
 * FOR ANY DAMAGES, INCLUDING SPECIAL, INDIRECT, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, WITH RESPECT TO ANY CLAIM ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OF THE SOFTWARE, EVEN
 * IF IT HAS BEEN OR IS HEREAFTER ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGES.
 */

#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include "../sys/net/bpf.h"
#include "parse_bpf.h"

void usage(char * progname) {

  printf("Usage: %s -p numpackets [-l length] file\n",progname);
  printf("Writes numpackets bpf packets of the given length (default is 60\n");
  printf("bytes) in file.  All packets will look to pkt_dump like ethernet\n");
  printf("frames containing IP packets, with all the IP addresses distinct.\n");
  exit(1);
}

void make_generic_bpf_header(struct bpf_hdr * hdr, int length) {

  /* Note that for benchmarking pkt_dump, the length is really the only
     value that matters. */
  hdr->bh_tstamp.tv_sec = 0;
  hdr->bh_tstamp.tv_usec = 0;
  hdr->bh_caplen = length;
  hdr->bh_datalen = length;
  hdr->bh_hdrlen = BPF_HDR_SIZE;
};

void make_generic_bpf_packet(char * packet, int length,
			    int src, int dst) {
  /* Only the length and the ip addresses are relevant to the
     performance of pkt_dump.  Since pkt_dump gets the packet length
     from the bpf header, we're not even bothering with length for
     now.  But eventually we may need more correct values. */
  struct ether_header * ether_start;
  struct ip * ip_start;

  ether_start = (struct ether_header *)packet;
  ether_start->ether_type = htons(ETHERTYPE_IP);
  ip_start = (struct ip *)(packet + ETHER_HDR_LEN);
  ip_start->ip_src.s_addr = (in_addr_t)src;
  ip_start->ip_dst.s_addr = (in_addr_t)dst;
};

int main(int argc, char * argv[]) {
  FILE * outfile;
  char * outfile_name;
  struct bpf_hdr this_hdr;
  char packet[MAX_PACKET_SIZE];
  int i;
  extern int optind;
  extern char *optarg;
  int ch;
  int numpackets = 0;
  int packet_length = 60; /*(minimum size on an ethernet)*/

  if (argc ==1) {
    usage(argv[0]);
  }
  while ((ch = getopt(argc, argv, "l:p:")) != -1) {
    switch(ch) {
    case 'p':
      numpackets = atoi(optarg);
      break;
    case 'l':
      packet_length = atoi(optarg);
      break;
    case '?':
    default:
      usage(argv[0]);
    }
  }
  if (optind+1 != argc) {
    usage(argv[0]);
  }
  outfile_name = argv[optind];
  if (!(outfile = fopen(outfile_name,"w"))) {
    fprintf(stderr,"Failed to open %s\n",outfile_name);
    perror(NULL);
    exit(1);
  }

  make_generic_bpf_header(&this_hdr,packet_length);

  for (i=0; i<numpackets; i++) {
    make_generic_bpf_packet(packet,packet_length,i,i+numpackets);
    if (write_bpf(outfile,&this_hdr,packet,1)) {
      perror("Error writing bpf packet");
      exit(1);
    }
  }
  
  fclose(outfile);

  exit(0);
}
